//=============================================================================
// RPG Maker MZ - Alternative Save Screen
//=============================================================================

/*:
 * @target MZ
 * @plugindesc セーブ／ロード画面のレイアウトを変更し、背景画像を設定し、ヘルプウィンドウを非表示にします。
 * @author PL
 *
 * @help PL_AltSaveScreen.js
 *
 * このプラグインは、セーブ／ロード画面のレイアウトを変更します。
 * セーブファイル一覧を20スロット表示、座標(145,170)から縦に配置、スロットサイズは640x41。
 * スロットには現在地(x=140,y=0)とプレイ時間(x=460,y=0)を表示。
 * ステータスウィンドウは座標(960,0)から960x1080で表示。
 * セーブ画面には pictures\pl_system\pl_save_bg.png を、
 * ロード画面には pictures\pl_system\pl_load_bg.png を背景として表示。
 * ヘルプウィンドウは非表示。
 * ウィンドウの縁・背景・パディングを削除、スロット背景も削除（選択演出は保持）。
 *
 * プラグインコマンドはありません。
 */

(() => {
    // フォントを事前ロード
    FontManager.load("Keifont", "keifont.ttf");

    // 最大セーブファイル数を21に設定（オートセーブ1＋通常セーブ20）
    DataManager.maxSavefiles = function () {
        return 21;
    };

    // DataManager.makeSavefileInfoをオーバーライド
    const _DataManager_makeSavefileInfo = DataManager.makeSavefileInfo;
    DataManager.makeSavefileInfo = function () {
        const info = _DataManager_makeSavefileInfo.call(this);
        //info.mapName = $gameMap.displayName();
    // 現在のマップIDからマップ名を取得
    const mapId = $gameMap.mapId();
    const mapInfo = $dataMapInfos[mapId];
    info.mapName = mapInfo ? mapInfo.name : ""; // マップ名を安全に取得
        info.timestampFormat = new Date().toLocaleString();
        // アクター1のレベル
        const actor1 = $gameActors.actor(1);
        info.actor1Level = actor1 ? actor1.level : 0;
        // 変数2,23, 43, 63, 83
        info.var2 = $gameVariables.value(2) || 0;
        info.var23 = $gameVariables.value(23) || 0;
        info.var43 = $gameVariables.value(43) || 0;
        info.var63 = $gameVariables.value(63) || 0;
        info.var83 = $gameVariables.value(83) || 0;
        // アクター71～74の名前
        info.name71 = $gameActors.actor(71)?.name() || "";
        info.name72 = $gameActors.actor(72)?.name() || "";
        info.name73 = $gameActors.actor(73)?.name() || "";
        info.name74 = $gameActors.actor(74)?.name() || "";
        return info;
    };

    // Scene_FileのcreateBackgroundメソッドをオーバーライド
    const _Scene_File_createBackground = Scene_File.prototype.createBackground;
    Scene_File.prototype.createBackground = function () {
        // 親クラスのcreateBackgroundを呼び出して初期化
        _Scene_File_createBackground.call(this);
        // 既存の背景スプライトをクリア
        if (this._backgroundSprite) {
            this.removeChild(this._backgroundSprite);
        }
        // 新しい背景スプライトを作成
        this._backgroundSprite = new Sprite();
        // シーンに応じて背景画像を設定
        const bgPath = this instanceof Scene_Save ? "img/pictures/pl_system/pl_save_bg.png" : "img/pictures/pl_system/pl_load_bg.png";
        this._backgroundSprite.bitmap = Bitmap.load(bgPath);
        // スプライトをシーンに追加
        this.addChild(this._backgroundSprite);
    };

    // Scene_Fileのcreateメソッドをオーバーライド
    const _Scene_File_create = Scene_File.prototype.create;
    Scene_File.prototype.create = function () {
        // 元のcreateメソッドを呼び出す（ヘルプウィンドウとリストウィンドウを作成）
        _Scene_File_create.call(this);
        // ヘルプウィンドウを非表示にする
        if (this._helpWindow) {
            this._helpWindow.hide();
        }
        // セーブファイル一覧ウィンドウの位置とサイズを設定
        this._listWindow.x = 110;
        this._listWindow.y = 159;
        this._listWindow.width = 700;
        this._listWindow.height = 41 * 21; // 21スロット分の高さ（41×21=861）
        // ウィンドウのフレームを透明にする
        this._listWindow.opacity = 0;
        // ステータスウィンドウの位置とサイズを(960,0)、960x1080に設定
        const x = 960;
        const y = 0;
        const width = 960;
        const height = 1080;
        const rect = new Rectangle(x, y, width, height);
        // 新しいステータスウィンドウを作成し、リストウィンドウに関連付ける
        const statusWindow = new Window_SavefileStatus(rect);
        this._listWindow.mzkp_statusWindow = statusWindow;
        // ステータスウィンドウのフレームを透明にする
        statusWindow.opacity = 0;
        // シーンにステータスウィンドウを追加
        this.addWindow(statusWindow);
    };

    // Scene_Fileのstartメソッドをオーバーライド
    const _Scene_File_start = Scene_File.prototype.start;
    Scene_File.prototype.start = function () {
        // 元のstartメソッドを呼び出す
        _Scene_File_start.call(this);
        // ヘルプウィンドウを非表示にする（念のため）
        if (this._helpWindow) {
            this._helpWindow.hide();
        }
        
        // リストウィンドウの選択を最初のスロット（index=0）に設定
        //this._listWindow.select(0);
        // リフレッシュ
        this._listWindow.refreshCursor();
        // カーソルが表示範囲内に収まるように調整
        this._listWindow.ensureCursorVisible();
        // ステータスウィンドウを直接更新
        this._listWindow.callUpdateHelp();
    };

    // Window_SavefileListのwindowWidthメソッドをオーバーライド
    Window_SavefileList.prototype.windowWidth = function () {
        return 700; // スロットの幅を700に設定
    };

    // スロットの列数を1に設定（縦に並べる）
    Window_SavefileList.prototype.maxCols = function () {
        return 1;
    };

    // スロットの項目の高さを41ピクセルに設定
    Window_SavefileList.prototype.itemHeight = function () {
        return 41;
    };

    // ウィンドウのパディングを0に設定
    Window_SavefileList.prototype.standardPadding = function () {
        return 0;
    };

    // スロットの背景を削除（選択時の演出は保持）
    Window_SavefileList.prototype.drawItemBackground = function (index) {
        // 背景を描画しない（選択時のカーソル演出は保持される）
    };

    // Window_SavefileListのdrawItemメソッドをオーバーライド
    Window_SavefileList.prototype.drawItem = function (index) {
        const savefileId = this.indexToSavefileId(index);
        const info = DataManager.savefileInfo(savefileId);
        const rect = this.itemRect(index);
        this.resetTextColor();
        // フォントをKeifontに設定
        this.contents.fontFace = "Keifont";
        // 文字色を黒に設定
        this.changeTextColor("#000000"); // 黒（16進数カラーコード）
        // 文字の縁を白に設定
        this.contents.outlineColor = "rgba(255, 255, 255, 1)"; // 白（RGBA形式）
        // 文字の縁の太さを設定
        this.contents.outlineWidth = 4; // 縁の太さを変更
        if (info) {
            // 現在地情報を座標(150,0)に描画
            this.drawText(info.mapName || "", rect.x + 150, rect.y, 300);
            // プレイ時間を座標(460,0)に描画
            this.drawText(info.playtime || "", rect.x + 460, rect.y, 160, "right");
        }
        // 描画後に設定をリセット（他のウィンドウに影響を与えないため）
        this.resetFontSettings();
        this.resetTextColor();
        this.contents.outlineColor = "rgba(0, 0, 0, 0.5)"; // デフォルトの縁色に戻す
        this.contents.outlineWidth = 4; // デフォルトの縁の太さに戻す
    };

    // Window_SavefileListのcallUpdateHelpメソッドをオーバーライド
    const _Window_SavefileList_callUpdateHelp =
        Window_SavefileList.prototype.callUpdateHelp;
    Window_SavefileList.prototype.callUpdateHelp = function () {
        // ヘルプウィンドウの更新をスキップ
        if (this.active && this.mzkp_statusWindow) {
            this.mzkp_statusWindow.setSavefileId(this.savefileId());
        }
    };

    // セーブファイルの詳細を表示する新しいウィンドウクラスを定義
    function Window_SavefileStatus() {
        this.initialize.apply(this, arguments);
    }

    // Window_Baseを継承してWindow_SavefileStatusを作成
    Window_SavefileStatus.prototype = Object.create(Window_Base.prototype);
    Window_SavefileStatus.prototype.constructor = Window_SavefileStatus;

    // Window_SavefileStatusの初期化メソッド
    Window_SavefileStatus.prototype.initialize = function (rect) {
        Window_Base.prototype.initialize.call(this, rect);
        this._savefileId = 1;
    };

    // Window_SavefileStatusのパディングを0に設定
    Window_SavefileStatus.prototype.standardPadding = function () {
        return 0;
    };

    // セーブファイルIDを設定し、ウィンドウを再描画
    Window_SavefileStatus.prototype.setSavefileId = function (id) {
        this._savefileId = id;
        this.refresh();
    };

    // ウィンドウの内容を再描画
    Window_SavefileStatus.prototype.refresh = function () {
        const info = DataManager.savefileInfo(this._savefileId);
        const rect = this.contents.rect;
        this.contents.clear();
        this.resetTextColor();
        if (info) {
            this.drawContents(info, rect);
        }
    };
// セーブファイルの詳細内容を描画
Window_SavefileStatus.prototype.drawContents = function (info, rect) {
    if (!info) return;

    // フォントをKeifontに設定
    this.contents.fontFace = "Keifont";
    // 文字色を黒に設定
    this.changeTextColor("#000000"); // 黒（16進数カラーコード）
    // 文字の縁を白に設定
    this.contents.outlineColor = "rgba(255, 255, 255, 1)"; // 白（RGBA形式）
    // 文字の縁の太さを設定
    this.contents.outlineWidth = 6; // 縁の太さを6ピクセルに

    // 座標と幅、オフセットの定数を定義
    const baseX = 380; // レベル、所持金、プレイ時間、セーブ時間、現在地のX座標
    const baseY = 80; // レベル、所持金、プレイ時間、セーブ時間、現在地の基準Y座標
    const offsetY = 74; // 各項目のY座標間隔
    const varX1 = 260; // 変数23、63、ニックネーム71、72のX座標
    const varX2 = 680; // 変数43、83、ニックネーム73、74のX座標
    const varY1 = 640; // 変数23、43のY座標（基準）
    const varY2 = 855; // 変数63、83のY座標（基準）
    const nicknameOffset = 42; // ニックネームのY座標オフセット
    const width1 = 320; // レベル、所持金、プレイ時間、セーブ時間、現在地の幅
    const width2 = 180; // 変数とニックネームの幅

    // フォントサイズ48で描画
    this.contents.fontSize = 42;
    // レベル（アクター1）
    this.drawText(`${info.actor1Level || 1}`, baseX, baseY, width1, "right");
    // 所持金（データベースの通貨単位を使用）
    //this.drawText(`${info.gold || 0} ${TextManager.currencyUnit}`, baseX, baseY + offsetY * 1, width1, "right");
    // 敗北数
    this.drawText(`${info.var2 || 0}`, baseX, baseY + offsetY * 1, width1, "right");
    // プレイ時間
    this.drawText(info.playtime || "0:00:00", baseX, baseY + offsetY * 2, width1, "right");
    // セーブ時間
    this.drawText(info.timestampFormat || "", baseX, baseY + offsetY * 3, width1, "right");
    // 現在地
    this.drawText(info.mapName || "", baseX, baseY + offsetY * 4, width1, "right");

    // 文字の縁の太さを設定
    this.contents.outlineWidth = 7; // 縁の太さを7ピクセルに

    // フォントサイズ32で描画
    this.contents.fontSize = 32;
    // アクター71～74の二つ名（変数の元の位置に表示）
    this.drawText(info.name71 || "", varX1, varY1, width2, "right");
    this.drawText(info.name72 || "", varX2, varY1, width2, "right");
    this.drawText(info.name73 || "", varX1, varY2, width2, "right");
    this.drawText(info.name74 || "", varX2, varY2, width2, "right");
    // 変数23～83（二つ名の元の位置に表示）
    this.drawText(`${info.var23 || 0}`, varX1, varY1 + nicknameOffset, width2, "right");
    this.drawText(`${info.var43 || 0}`, varX2, varY1 + nicknameOffset, width2, "right");
    this.drawText(`${info.var63 || 0}`, varX1, varY2 + nicknameOffset, width2, "right");
    this.drawText(`${info.var83 || 0}`, varX2, varY2 + nicknameOffset, width2, "right");

    // フォント設定をリセット（他のウィンドウに影響を与えないため）
    this.resetFontSettings();
    this.resetTextColor();
    this.contents.outlineColor = "rgba(0, 0, 0, 0.5)"; // デフォルトの縁色に戻す
    this.contents.outlineWidth = 4; // デフォルトの縁の太さに戻す
};
})();