//=============================================================================
// PL_MPP_ChoiceEX_Plugin.js
//=============================================================================

/*:ja
 * @target MZ
 * @plugindesc 【選択肢】MPP_ChoiceEXより後に読み込む必要あり。
 * 選択肢の移動でスイッチをONにできたり、現在のカーソル位置を取得できるようにします。
 * @author PL
 * @url 
 *
 * @help 
 * 選択肢の移動でスイッチをONにできたり、現在のカーソル位置を取得できるようにします。
 * スイッチ変更で並列処理のコモンイベントを実行するなどが基本用途です。
 * 
 * @command choiceOnSwitch
 *     @text 次の1回、選択肢を移動する度に該当のスイッチIDをONにする。
 *     @desc 次の1回、選択肢を移動する度に該当のスイッチIDをONにする。
 *     @arg switchId
 *         @text スイッチ
 *         @desc 
 *         @type switch
 *         @default 0
 * 
 * @param CurrentCursorIndexToVariableID
 *     @text 現在の選択肢のカーソル位置を保存する変数番号。
 *     @desc 現在の選択肢のカーソル位置を保存する変数番号。
 *     0の場合は、どこにも代入しない。
 *     また、ウィンドウが閉じた時は0を代入する。（選択肢表示時のカーソル位置は、1から）
 *     @type number
 *     @default 1012
 */

(() => {
    'use strict';

    const pluginName = 'PL_MPP_ChoiceEX_Plugin';

    // プラグインパラメータの取得
    const parameters = PluginManager.parameters(pluginName);
    const CURRENT_CURSOR_VARIABLE_ID = Number(parameters['CurrentCursorIndexToVariableID']) || 0;

    // プラグイン内部で閉じた変数として _choiceSwitchOnID を定義
    let _choiceSwitchOnID = 0;

    //=========================================================================
    // 内部変数のアクセサメソッド
    //=========================================================================

    /**
     * スイッチON設定用のIDを設定する
     * @param {number} value - スイッチID
     */
    function setChoiceSwitchOnID(value) {
        //dm('★setChoiceSwitchOnID', value);
        _choiceSwitchOnID = value;
    }

    /**
     * スイッチON設定用のIDを取得する
     * @returns {number} 現在のスイッチID
     */
    function getChoiceSwitchOnID() {
        return _choiceSwitchOnID;
    }

    //=========================================================================
    // プラグインコマンドの登録
    //=========================================================================

    PluginManager.registerCommand(pluginName, 'choiceOnSwitch', (args) => {
        const switchId = Number(args.switchId);
        setupChoiceOnSwitch(switchId);
    });

    /**
     * 選択肢移動時に指定スイッチをONにする設定を行う
     * @param {number} switchId - ONにするスイッチのID
     */
    function setupChoiceOnSwitch(switchId) {
        dm('■setupChoiceOnSwitch', switchId);
        setChoiceSwitchOnID(switchId);
    }

    //=========================================================================
    // Window_ChoiceListのカスタマイズ
    //=========================================================================

    const _Window_ChoiceList_select = Window_ChoiceList.prototype.select;
    Window_ChoiceList.prototype.select = function(index) {
        //dm('PL_MPP_ChoiceEX_Plugin.js : select | 実行タイミング：選択肢のカーソルが移動する度');

        // カーソル位置を指定変数に保存（1-based index）
        this.setCurrentCursorIndexToVariableID(index + 1);

        // スイッチON設定が有効な場合、スイッチをONにする
        const switchId = getChoiceSwitchOnID();
        if (switchId > 0) {
            dm('@select', switchId);
            $gameSwitches.setValue(switchId, true);
        }

        _Window_ChoiceList_select.call(this, index);
    };

    /**
     * 現在のカーソル位置を指定された変数に保存する
     * @param {number} value - 保存するカーソル位置（0:閉じた時、それ以外:選択中）
     */
    Window_ChoiceList.prototype.setCurrentCursorIndexToVariableID = function(value) {
        //dm('@setCurrentCursorIndexToVariableID | value(カレント選択肢ID)', value);
        if (CURRENT_CURSOR_VARIABLE_ID > 0) {
            $gameVariables.setValue(CURRENT_CURSOR_VARIABLE_ID, value);
        }
    };

    const _Window_ChoiceList_close = Window_ChoiceList.prototype.close;
    Window_ChoiceList.prototype.close = function() {
        dm('PL_MPP_ChoiceEX_Plugin.js : close | 実行タイミング：選択肢ウィンドウが閉じた時');

        // カーソル位置を0にリセット
        this.setCurrentCursorIndexToVariableID(0);

        // スイッチと設定をリセット
        const switchId = getChoiceSwitchOnID();
        if ($gameSwitches && switchId > 0) {
            $gameSwitches.setValue(switchId, false);
        }
        setChoiceSwitchOnID(0);

        _Window_ChoiceList_close.call(this);
    };
})();