// FadePlus.js Ver.1.0.1
// MIT License (C) 2023 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MZ
* @plugindesc Reinforces fade in/out.
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/498644942.html
* @help Ver.1.0.1
* The parameter affects the number of frames in the combo box.
*
* @param instant
* @text Instant
* @desc Number of frames for fade in/out.
* @type number
* @default 1
*
* @param faster
* @text Faster
* @desc Number of frames for fade in/out.
* @type number
* @default 12
*
* @param normal
* @text Normal
* @desc Number of frames for fade in/out.
* @type number
* @default 24
*
* @param slower
* @text Slower
* @desc Number of frames for fade in/out.
* @type number
* @default 48
*
* @command fadeOut
* @text Fadeout Screen
* @desc Fade out.
*
* @arg duration
* @text Duration
* @desc Specify the number of frames.
* Enter a number or select from the combo box.
* @default Normal
* @type combo
* @option Instant
* @option Faster
* @option Normal
* @option Slower
*
* @arg wait
* @text Wait for Completion
* @desc Wait until the fade is complete.
* @default true
* @type boolean
*
* @command fadeIn
* @text Fadein Screen
* @desc Fade in.
*
* @arg duration
* @text Duration
* @desc Specify the number of frames.
* Enter a number or select from the combo box.
* @default Normal
* @type combo
* @option Instant
* @option Faster
* @option Normal
* @option Slower
*
* @arg wait
* @text Wait for Completion
* @desc Wait until the fade is complete.
* @default true
* @type boolean
*
*/

/*:ja
* @target MZ
* @plugindesc フェードイン・アウトを補強します。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/498644942.html
* @help パラメータはコンボボックスでのフレーム数に影響します。
* [更新履歴]
* 2023/03/19：Ver.1.0.0　公開。
* 2023/04/30：Ver.1.0.1　「瞬時」の不具合を修正。
*
* @param instant
* @text 瞬時
* @desc フェードイン・アウトのフレーム数です。
* @type number
* @default 1
*
* @param faster
* @text 速い
* @desc フェードイン・アウトのフレーム数です。
* @type number
* @default 12
*
* @param normal
* @text 普通
* @desc フェードイン・アウトのフレーム数です。
* @type number
* @default 24
*
* @param slower
* @text 遅い
* @desc フェードイン・アウトのフレーム数です。
* @type number
* @default 48
*
* @command fadeOut
* @text 画面のフェードアウト
* @desc フェードアウトします。
*
* @arg duration
* @text 時間
* @desc フレーム数を指定します。
* 数値を入力するかコンボボックスから選択します。
* @default 普通
* @type combo
* @option 瞬時
* @option 速い
* @option 普通
* @option 遅い
*
* @arg wait
* @text 完了までウェイト
* @desc フェードが完了するまで待ちます。
* @default true
* @type boolean
*
* @command fadeIn
* @text 画面のフェードイン
* @desc フェードインします。
*
* @arg duration
* @text 時間
* @desc フレーム数を指定します。
* 数値を入力するかコンボボックスから選択します。
* @default 普通
* @type combo
* @option 瞬時
* @option 速い
* @option 普通
* @option 遅い
*
* @arg wait
* @text 完了までウェイト
* @desc フェードが完了するまで待ちます。
* @default true
* @type boolean
*
*/

'use strict';
{
	const useMZ = Utils.RPGMAKER_NAME === "MZ";
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const hasPluginCommonBase = typeof PluginManagerEx === "function";
	const parameters = PluginManager.parameters(pluginName);

	const instant = Number(parameters["instant"]);
	const faster = Number(parameters["faster"]);
	const normal = Number(parameters["normal"]);
	const slower = Number(parameters["slower"]);

	//-----------------------------------------------------------------------------
	// PluginManager

	function getDuration(duration) {
		switch (String(duration).toLowerCase()) {
		case "instant":
		case "瞬時":
			return instant;
		case "faster":
		case "速い":
			return faster;
		case "normal":
		case "普通":
			return normal;
		case "slower":
		case "遅い":
			return slower;
		}
		return Number(duration || 0);
	}

	PluginManager.registerCommand(pluginName, "fadeIn", function (args) {
		const duration = getDuration(args.duration);
		$gameScreen.startFadeIn(duration);
		if (args.wait === "true") {
			this.wait(duration);
		}
	});

	PluginManager.registerCommand(pluginName, "fadeOut", function (args) {
		const duration = getDuration(args.duration);
		$gameScreen.startFadeOut(duration);
		if (args.wait === "true") {
			this.wait(duration);
		}
	});

	if (hasPluginCommonBase) {
		PluginManagerEx.registerCommand(document.currentScript, "fadeIn", function (args) {
			const duration = getDuration(args.duration);
			$gameScreen.startFadeIn(duration);
			if (args.wait) {
				this.wait(duration);
			}
		});

		PluginManagerEx.registerCommand(document.currentScript, "fadeOut", function (args) {
			const duration = getDuration(args.duration);
			$gameScreen.startFadeOut(duration);
			if (args.wait) {
				this.wait(duration);
			}
		});
	}
}