// AWY_MotionPresetter.js Ver.2.0.0
// MIT License (C) 2023 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MZ
* @base AWY_MotionMaker
* @orderAfter PluginCommonBase
* @orderAfter OptimizedMovementRoute
* @plugindesc Expand your motion capabilities.
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/501228970.html
* @help Ver.2.0.0
* 【How to Use】
* Create a motion preset by placing the plug-in command
* and movement route in the common event specified by the parameter.
* The method of creating motion is exactly the same as AWY_MotionMaker.
* Created motions can be called using plugin commands or scripts.
*
* 【Scripts (Movement Route)】
* this.callMotion("name");
* this.callMotion("name", wait);
* this.callMotion("name", wait, loop);
* this.callMotion("name", wait, loop, speed);
* //Call the motion.
* Enter the motion name in "name",
* the truth value (true/false) to indicate whether to wait for completion in "wait",
* the playback speed in "speed",
* and the number of loops in "loop". .
* If "wait" is not entered, the preset will be followed.
* If "loop" is not input, it will follow the preset, and if it is -1, it will loop infinitely.
* If "speed" is not entered, the preset will be followed.
* this.waitForMotion();  //Wait until the motion is finished.
*
* @param commonEventId
* @text Common Event
* @desc Specify common events for creating presets.
* @type common_event
* @default 1
*
* @command callMotion
* @text Call Motion
* @desc Call up a preset motion.
*
* @arg characterId
* @text Character ID
* @desc If it is left blank, it will be the last target.
*
* @arg name
* @text Motion Name
* @desc The name of the motion to call.
* @type combo
* @option It is useful.
* @option to edit these
* @option and add motion names
* @option Motion1
* @option Motion2
*
* @arg loop
* @text Number of Loops
* @desc If left blank, the preset will be followed.
* Infinite loop with either 無限, infinity, or -1
* @type combo
* @option infinity
*
* @arg speed
* @text Speed
* @desc Specify as a percentage. 100 is the same size.
*
* @arg wait
* @text Wait for Completion
* @desc Wait until the motion ends.
* If left blank, the preset will be followed.
* @type boolean
*
* @arg script
* @text Script
* @desc This is a script that is always executed when the motion ends or is interrupted.
* @type multiline_string
*
* @command presetMotion
* @text Preset Motion
* @desc Add the movement route settings to the preset.
* It will not remain in the save data.
*
* @arg name
* @text Motion Name
* @desc This is the name when calling.
*
* @arg loop
* @text Number of Loops
* @desc If left blank, the movement route option will be followed.
* Infinite loop with either 無限, infinity, or -1
* @type combo
* @option infinity
*
* @arg speed
* @text Speed
* @desc Specify as a percentage. 100 is the same size.
* @default 100
*
* @arg script
* @text Script
* @desc This is a script that is always executed when the motion ends or is interrupted.
* @type multiline_string
*
*/

/*:ja
* @target MZ
* @base AWY_MotionMaker
* @orderAfter PluginCommonBase
* @orderAfter OptimizedMovementRoute
* @plugindesc モーション機能を拡張します。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/501228970.html
* @help
* 【使い方】
* パラメータで指定したコモンイベントにプラグインコマンドと移動ルートを配置し、
* モーションのプリセットを作成します。
* モーションの作成方法はAWY_MotionMakerと全く同じです。
* 作成したモーションはプラグインコマンドやスクリプトで呼び出せます。
*
* 【スクリプト（移動ルート）】
* this.callMotion("name");
* this.callMotion("name", wait);
* this.callMotion("name", wait, loop);
* this.callMotion("name", wait, loop, speed);
* //モーションを呼び出します。
* nameにはモーション名を、
* waitには完了までウェイトするかを真偽値（true/false）で、
* speedには再生速度、
* loopにはループ回数を入力してください。
* waitを未入力だとプリセットに従います。
* loopを未入力だとプリセットに従い、-1だと、無限ループします。
* speedを未入力だとプリセットに従います。
* this.waitForMotion();  //モーションが終わるまで待ちます。
*
* [更新履歴]
* 2023/10/25：Ver.1.0.0　公開。
* 2023/11/10：Ver.2.0.0　AWY_MotionMaker Ver.4.0.0に対応。
*
* @param commonEventId
* @text コモンイベント
* @desc プリセットを作成するためのコモンイベントを指定します。
* @type common_event
* @default 1
*
* @command callMotion
* @text モーションの呼び出し
* @desc プリセットのモーションを呼び出します。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと最後の対象になります。
*
* @arg name
* @text モーション名
* @desc 呼び出すモーション名です。
* @type combo
* @option ここを編集して
* @option モーション名を追加しておくと
* @option 便利です。
* @option モーション1
* @option モーション2
*
* @arg loop
* @text ループ回数
* @desc 未記入だとプリセットに従います。
* 無限、infinity、-1のいずれかで無限ループ
* @type combo
* @option 無限
*
* @arg speed
* @text 再生速度
* @desc 百分率で指定します。100が等倍。
*
* @arg wait
* @text 完了までウェイト
* @desc モーションが終了するまで待ちます。
* 未記入だとプリセットに従います。
* @type boolean
*
* @arg script
* @text スクリプト
* @desc モーションが終了・中断された時に必ず実行するスクリプトです。
* @type multiline_string
*
* @command presetMotion
* @text モーションのプリセット
* @desc 直下の移動ルートの設定をプリセットに追加します。
* セーブデータには残りません。
*
* @arg name
* @text モーション名
* @desc 呼び出すときの名前です。
*
* @arg loop
* @text ループ回数
* @desc 未記入だと移動ルートのオプションに従います。
* 無限、infinity、-1のいずれかで無限ループ
* @type combo
* @option 無限
*
* @arg speed
* @text 再生速度
* @desc 百分率で指定します。100が等倍。
* @default 100
*
* @arg script
* @text スクリプト
* @desc モーションが終了・中断された時に必ず実行するスクリプトです。
* @type multiline_string
*
*/

'use strict';
{
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const parameters = PluginManager.parameters(pluginName);
	const hasPluginCommonBase = typeof PluginManagerEx === "function";
	const commonEventId = Number(parameters.commonEventId || 0);

	//-----------------------------------------------------------------------------
	// DataManager

	const _DataManager_createGameObjects = DataManager.createGameObjects;
	DataManager.createGameObjects = function() {
		_DataManager_createGameObjects.call(this);
		const interpreter = new Game_Interpreter();
		const commonEvent = $dataCommonEvents[commonEventId];
		if (commonEvent) {
			interpreter.setup(commonEvent.list);
			while(interpreter.isRunning()) {
				//interpreter.update();
				interpreter.executeCommand(); // modified
			}
		}
	};

	//-----------------------------------------------------------------------------
	// PluginManager

	const motionPresets = {};
	PluginManager.registerCommand(pluginName, "callMotion", function(args) {
		const motionPreset = motionPresets[args.name];
		if (!motionPreset) {
			return;
		}
		//準備
		motion = JsonEx.makeDeepCopy(motionPreset);
		let numOfLoop = args.loop;
		if (!numOfLoop) {
			numOfLoop = motion.loop;
		} else if (["無限", "infinity"].includes(numOfLoop)) {
			numOfLoop = -1;
		}
		motion.loop = +numOfLoop;
		if (args.speed > 0) {
			motion.speed = +args.speed;
		}
		if (args.wait) {
			motion.wait = args.wait === "true";
		}
		motion.script = args.script || motion.script;
		//実行
		const characterId = Number(args.characterId || this._characterId);
		this.callCharacterMotion(characterId, motion);
	});

	PluginManager.registerCommand(pluginName, "presetMotion", function(args) {
		const index = findReplaceCommandIndex(this._index, this._list);
		if (index === -1) {
			return;
		}
		const command = this._list[index];
		const motion = command.parameters[1];
		command.presetMotion = true;
		motionPresets[args.name] = motion;
		let numOfLoop = args.loop;
		if (!numOfLoop) {
			numOfLoop = params[1].repeat ? -1 : 1;
		} else if (["無限", "infinity"].includes(numOfLoop)) {
			numOfLoop = -1;
		}
		motion.loop = +numOfLoop;
		motion.speed = Number(args.speed || 100);
		motion.script = args.script;
	});

	//置換するコマンドを検索。見つかったら登録、無かったらこのコマンドのインデックスを登録。
	function findReplaceCommandIndex(index, list) {
		const scanEnd = list.length - 1;
		for (let i = index + 1; i <= scanEnd; i++) {
			const command = list[i];
			if (command.code === 205) {
				return i;
			}
		}
		return -1;
	};

	if (hasPluginCommonBase) {
		PluginManagerEx.registerCommand(document.currentScript, "callMotion", function(args) {
			const motionPreset = motionPresets[args.name];
			if (!motionPreset) {
				return;
			}
			//準備
			const motion = JsonEx.makeDeepCopy(motionPreset);
			let numOfLoop = args.loop;
			if (numOfLoop === "") {
				numOfLoop = motion.loop;
			} else if (["無限", "infinity"].includes(numOfLoop)) {
				numOfLoop = -1;
			}
			motion.loop = numOfLoop;
			if (args.speed > 0) {
				motion.speed = args.speed;
			}
			if (args.wait !== "") {
				motion.wait = args.wait;
			}
			motion.script = args.script || motion.script;
			//実行
			const characterId = args.characterId === "" ? this._characterId : args.characterId;
			this.callCharacterMotion(characterId, motion);
		});

		PluginManagerEx.registerCommand(document.currentScript, "presetMotion", function(args) {
			const index = findReplaceCommandIndex(this._index, this._list);
			if (index === -1) {
				return;
			}
			const command = this._list[index];
			const motion = command.parameters[1];
			command.presetMotion = true;
			motionPresets[args.name] = motion;
			let numOfLoop = args.loop;
			if (numOfLoop === "") {
				numOfLoop = motion.repeat ? -1 : 1;
			} else if (["無限", "infinity"].includes(numOfLoop)) {
				numOfLoop = -1;
			}
			motion.loop = numOfLoop;
			motion.script = args.script;
			motion.speed = args.speed || 100;
		});
	}

	//-----------------------------------------------------------------------------
	// Game_Interpreter
	// 移動ルートの設定
	const _Game_Interpreter_command205 = Game_Interpreter.prototype.command205;
	Game_Interpreter.prototype.command205 = function(params) {
		return !!this.currentCommand().presetMotion || _Game_Interpreter_command205.apply(this, arguments);
	};

	Game_Interpreter.prototype.callCharacterMotion = function(characterId, motion) {
		$gameMap.refreshIfNeeded();
		this._characterId = characterId;
		const character = this.character(this._characterId);
		if (character) {
			character.setMotion(motion);
			if (motion.wait) {
				this.waitForMotion();
			}
		}
	};

	//-----------------------------------------------------------------------------
	// Game_Character

	const _Game_Character_initMembers = Game_Character.prototype.initMembers;
	Game_Character.prototype.initMembers = function() {
		_Game_Character_initMembers.call(this);
		this._waitForMotion = false;
	};

	Game_Character.prototype.callMotion = function(name, wait, loop, speed) {
		const motionPreset = motionPresets[name];
		if (motionPreset) {
			const motion = JsonEx.makeDeepCopy(motionPreset);
			motion.loop = loop ?? motion.loop;
			motion.speed = speed ?? motion.speed;
			this.setMotion(motion);
			this._waitForMotion = wait ?? motion.wait;
		}
	};

	Game_Character.prototype.waitForMotion = function() {
		this._waitForMotion = true;
	};

	const _Game_Character_updateRoutineMove = Game_Character.prototype.updateRoutineMove;
	Game_Character.prototype.updateRoutineMove = function() {
		let waiting = true;
		if (!this._waitForMotion || !this._motionPlaying) {
			this._waitForMotion = waiting = false;
		}
		if (!waiting || this._waitCount > 0) {
			_Game_Character_updateRoutineMove.call(this);
		}
	};

	//OptimizedMovementRoute
	const _Game_Character_shouldBreakHere = Game_Character.prototype.shouldBreakHere;
	Game_Character.prototype.shouldBreakHere = function(index) {
		if (this._waitForMotion && this._motionPlaying) {
			return true;
		}
		return _Game_Character_shouldBreakHere.apply(this, arguments);
	};

}