/*:
 * @target MZ
 * @plugindesc FPS低下時に描画だけ自動スロットルし、ロジックは等速維持。制限中はスイッチ147番に反映。制限モードは最低2秒持続。
 * 
 * @help
 * FPSが55未満になると描画だけを2フレームに1回（30FPS相当）に間引き、
 * FPSが58以上に戻ると60FPSに復帰します。
 * 
 * - ロジックは毎フレーム進行（等速）
 * - 描画制限中はスイッチ147番がON（true）になります
 * - 描画制限モードは最低2000ms（2秒）間は維持されます
 * 
 * 導入するだけで自動的に動作します。
 */

(() => {
    // プラグイン設定値
    const MONITOR_SWITCH_ID = 147; // 現在30fps間引きモードかを監視するスイッチ
    const FPS_LOW_THRESHOLD = 55;  // 低下判定FPS
    const FPS_RECOVER_THRESHOLD = 58; // 回復判定FPS
    const CHECK_INTERVAL = 30; // フレーム間隔（何フレームごとにFPSチェックするか）
    const FORCE_RENDER_THRESHOLD = 2000; // 2秒間レンダリングできなかったら強制描画
    const MIN_RENDER_LIMIT_DURATION = 2000; // 30fpsモード最低継続時間（ミリ秒）
    const FORCE_FULL_FPS_DURATION = 1000; // メニュー復帰時の強制60fps時間（ミリ秒）

    // 内部管理用
    let _fpsRenderMode = 1; // 1: 通常描画、2: 間引き描画（2フレームに1回）
    let _lastFpsCheckTime = performance.now();
    let _lastFpsFrameCount = Graphics.frameCount;
    let _renderLimitStartTime = 0;
    let _lastRenderTime = performance.now();
    let _forceFullFpsUntil = 0;
    let _lastSwitchState = null;

    // フルFPS強制API（外部からも呼べる）
    window.forceFullFpsFor = function(durationMs) {
        _forceFullFpsUntil = performance.now() + durationMs;
    };

    // Graphics._onTick の軽いオーバーライド
    const _Graphics_onTick = Graphics._onTick;
    Graphics._onTick = function(deltaTime) {
        this._fpsCounter.startTick();

        if (this._tickHandler) {
            this._tickHandler(deltaTime);
        }

        const now = performance.now();
        const elapsedSinceRender = now - _lastRenderTime;
        const forceFullFps = now < _forceFullFpsUntil;

        // FPSチェック
        if (Graphics.frameCount % CHECK_INTERVAL === 0) {
            const frameDiff = Graphics.frameCount - _lastFpsFrameCount;
            const timeDiffSec = (now - _lastFpsCheckTime) / 1000;
            const estimatedFps = frameDiff / timeDiffSec;

            if (_fpsRenderMode === 1 && estimatedFps < FPS_LOW_THRESHOLD) {
                _fpsRenderMode = 2;
                _renderLimitStartTime = now;
            } else if (_fpsRenderMode === 2 && estimatedFps > FPS_RECOVER_THRESHOLD) {
                if (now - _renderLimitStartTime >= MIN_RENDER_LIMIT_DURATION) {
                    _fpsRenderMode = 1;
                }
            }

            if (window.$gameSwitches && typeof $gameSwitches.setValue === "function") {
                const current = _fpsRenderMode > 1;
                if (_lastSwitchState !== current) {
                    $gameSwitches.setValue(MONITOR_SWITCH_ID, current);
                    _lastSwitchState = current;
                }
            }

            _lastFpsCheckTime = now;
            _lastFpsFrameCount = Graphics.frameCount;
        }

        // 描画するかどうか
        if (this._canRender() && (forceFullFps || _fpsRenderMode === 1 || Graphics.frameCount % _fpsRenderMode === 0 || elapsedSinceRender > FORCE_RENDER_THRESHOLD)) {
            this._app.render();
            _lastRenderTime = performance.now();
        }

        this._fpsCounter.endTick();
    };

    // Scene_Map復帰時に自動で1秒間フルFPSモードに入る
    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.call(this);
        if (typeof forceFullFpsFor === "function") {
            forceFullFpsFor(FORCE_FULL_FPS_DURATION);
        }
    };
})();


