/*:
 * @target MZ
 * @plugindesc スイッチフロー可視化：マップIDの後ろにエディタ上のマップ名を表示する改良版
 */

(() => {

    function pageKey(mapId, eventId, page) {
      return `M${mapId}_E${eventId}_P${page}`;
    }
  
    function describePage(info) {
      return `[Map${info.mapId.toString().padStart(3, '0')}:${info.mapName}] ${info.name} (${info.x},${info.y}) #${info.eventId}-P${info.page}`;
    }
  
    async function runSwitchFlowChainCheck() {
      const pageMap = {};
      const switchProducers = {};
      const switchConsumers = {};
  
      for (let mapId = 1; mapId < $dataMapInfos.length; mapId++) {
        const mapInfo = $dataMapInfos[mapId];
        if (!mapInfo || !mapInfo.name) continue;
  
        const mapName = mapInfo.name;
        const filename = `Map${mapId.toString().padStart(3, '0')}.json`;
        const url = `data/${filename}`;
  
        try {
          const response = await fetch(url);
          const mapData = await response.json();
          const events = mapData.events || [];
  
          for (const ev of events) {
            if (!ev || !ev.pages) continue;
  
            for (let pageIndex = 0; pageIndex < ev.pages.length; pageIndex++) {
              const page = ev.pages[pageIndex];
              const conditions = page.conditions || {};
              const condSwitches = [];
              if (conditions.switch1Valid) condSwitches.push(conditions.switch1Id);
              if (conditions.switch2Valid) condSwitches.push(conditions.switch2Id);
  
              const setSwitches = [];
              for (const cmd of page.list) {
                if (cmd.code === 121 && Array.isArray(cmd.parameters)) {
                  const from = Number(cmd.parameters[0]);
                  const to = Number(cmd.parameters[1]);
                  const op = cmd.parameters[2];
                  if (op === 0 && Number.isInteger(from) && Number.isInteger(to)) {
                    for (let i = from; i <= to; i++) setSwitches.push(i);
                  }
                }
              }
  
              if (condSwitches.length === 0 && setSwitches.length === 0) continue;
  
              const key = pageKey(mapId, ev.id, pageIndex + 1);
              const info = {
                mapId,
                mapName,
                eventId: ev.id,
                page: pageIndex + 1,
                name: ev.name || `EV${ev.id}`,
                x: ev.x,
                y: ev.y,
                condSwitches,
                setSwitches
              };
              pageMap[key] = info;
  
              for (const sw of condSwitches) {
                if (!switchConsumers[sw]) switchConsumers[sw] = [];
                switchConsumers[sw].push(key);
              }
              for (const sw of setSwitches) {
                if (!switchProducers[sw]) switchProducers[sw] = [];
                switchProducers[sw].push(key);
              }
            }
          }
        } catch (e) {
          console.warn(`マップ${mapId}の読み込み失敗: ${e.message}`);
        }
      }
  
      console.log("=== スイッチ連鎖フロー（マップ名付き） ===");
      const visited = new Set();
  
      for (const sw in switchProducers) {
        const producers = switchProducers[sw];
        for (const startKey of producers) {
          if (visited.has(startKey)) continue;
  
          const chain = [];
          let currentKey = startKey;
  
          while (currentKey && !visited.has(currentKey)) {
            visited.add(currentKey);
            const page = pageMap[currentKey];
            chain.push({ key: currentKey, page });
  
            const nextSwitches = page.setSwitches;
            let nextKey = null;
            for (const swId of nextSwitches) {
              const consumers = switchConsumers[swId] || [];
              for (const cKey of consumers) {
                if (!visited.has(cKey)) {
                  nextKey = cKey;
                  break;
                }
              }
              if (nextKey) break;
            }
  
            currentKey = nextKey;
          }
  
          if (chain.length >= 2) {
            console.log("---");
            for (let i = 0; i < chain.length; i++) {
              const p = chain[i].page;
              const cond = p.condSwitches.map(s => `S${s}`).join(',');
              const set = p.setSwitches.map(s => `S${s}`).join(',');
              console.log(`${i === 0 ? "START:" : "   -->"} ${describePage(p)} 条件:[${cond}] → セット:[${set}]`);
            }
          }
        }
      }
    }
  
    Scene_Map.prototype.checkStorySwitchFlowChain = runSwitchFlowChainCheck;
  
  })();
  