/*:
 * @plugindesc 拡張決定ボタントリガープラグイン
 * @target MZ
 * @param detectionDistanceVarId
 * @text 検知距離を格納するゲーム変数ID
 * @type variable
 * @default 1
 *
 * @param detectedEventVarId
 * @text 検知し起動予約されたイベントキャラのイベントIDを格納するゲーム変数ID
 * @type variable
 * @default 2
 *
 * @param detectionSwitchId
 * @text 検知エリアスイッチID
 * @type switch
 * @default 1
 *
 * @param detectionAnimationId
 * @text 検知アニメーションID
 * @type animation
 * @default 1
 *
 * @param fieldOfView
 * @text プレイヤーの視野角（度）
 * @type number
 * @min 1
 * @max 180
 * @default 60
 *
 * @help
 * このプラグインは、プレイヤーキャラクターの方向と距離に基づいてイベントを検知し、
 * 決定ボタンによるイベント起動を任意の広さの領域に拡張します。
 * 
 * 通常の「決定ボタン」によるイベント起動は、
 * プレイヤーキャラが対象のイベントに隣接かつその方向を向いているときにのみ発生しますが、
 * このプラグインを使用すると、任意の広さの領域内にいるイベントを対象にできます。
 */

(() => {
    const parameters = PluginManager.parameters('EventDetectionExtend');
    const detectionDistanceVarId = Number(parameters['detectionDistanceVarId'] || 1);
    const detectedEventVarId = Number(parameters['detectedEventVarId'] || 2);
    const detectionSwitchId = Number(parameters['detectionSwitchId'] || 1);
    const detectionAnimationId = Number(parameters['detectionAnimationId'] || 1);
    const fieldOfView = Number(parameters['fieldOfView'] || 60);

    let lastDetectedEventId = 0; // 前回のフレームで検知されたイベントIDを保持

    const _Game_Player_update = Game_Player.prototype.update;
    Game_Player.prototype.update = function(sceneActive) {
        _Game_Player_update.call(this, sceneActive);
        this.updateEventDetection();
    };

    Game_Player.prototype.updateEventDetection = function() {
        const detectionDistance = $gameVariables.value(detectionDistanceVarId);
        const direction = this.direction();
        const px = this._realX;
        const py = this._realY;
        let closestEvent = null;
        let minDistance = Infinity;

        $gameMap.events().forEach(event => {
            if (!event.isTriggerIn([0]) || event.list().length <= 1 || event.regionId() != $gamePlayer.regionId()) return; // 決定ボタントリガーでないイベント、またはコマンドがないイベントは除外

            const ex = event._realX;
            const ey = event._realY;
            const dx = ex - px;
            const dy = ey - py;
            const distance = Math.sqrt(dx * dx + dy * dy);
            const corrAngle = Number(event.event().meta.corrAngle) || 0;

            if (distance <= detectionDistance && this.isEventInFieldOfView(dx, dy, direction, corrAngle)) {
                if (distance < minDistance) {
                    minDistance = distance;
                    closestEvent = event;
                }
            }
        });

        let preventSwitchId = 24; // 機能抑制スイッチID
        if (closestEvent && !$gameMap.isEventRunning() && !$gameSwitches.value(preventSwitchId)) {
            const closestEventId = closestEvent.eventId();
            $gameVariables.setValue(detectedEventVarId, closestEventId);
            $gameSwitches.setValue(detectionSwitchId, true);

            if (lastDetectedEventId !== closestEventId) {
                $gameTemp.requestAnimation([closestEvent], detectionAnimationId);
                lastDetectedEventId = closestEventId; // 検知されたイベントIDを更新
            }
        } else {
            $gameVariables.setValue(detectedEventVarId, 0);
            $gameSwitches.setValue(detectionSwitchId, false);
            lastDetectedEventId = 0; // 検知されたイベントがない場合はリセット
        }
    };

    Game_Player.prototype.isEventInFieldOfView = function(dx, dy, direction, corrAngle) {
        const angle = Math.atan2(dy, dx);
        const playerAngle = this.directionToAngle(direction);
        const relativeAngle = this.normalizeAngle(angle - playerAngle);
        const fov = ((fieldOfView + corrAngle) / 180) * Math.PI / 2; // パラメータから視野角を設定
        return Math.abs(relativeAngle) < fov;
    };

    Game_Player.prototype.directionToAngle = function(direction) {
        switch (direction) {
            case 2: return Math.PI / 2;   // 下
            case 4: return Math.PI;       // 左
            case 6: return 0;             // 右
            case 8: return -Math.PI / 2;  // 上
            default: return 0;
        }
    };

    Game_Player.prototype.normalizeAngle = function(angle) {
        while (angle > Math.PI) angle -= 2 * Math.PI;
        while (angle < -Math.PI) angle += 2 * Math.PI;
        return angle;
    };

    var EVENTNAME_VARIABLEID = 268;
    Game_Player.prototype.triggerButtonAction = function() {
        if (Input.isTriggered('ok') && !$gameMap.isEventRunning()) {
            const eventId = $gameVariables.value(detectedEventVarId);
            if (eventId > 0) {
                const event = $gameMap.event(eventId);
                if (event && event.isTriggerIn([0]) && event.list().length > 1) { // 決定ボタントリガーでコマンドがあるイベント
                    if (event.isNormalPriority() && !this.isInVehicle()) {
                        $gameVariables.setValue(EVENTNAME_VARIABLEID, $dataMap.events[eventId].name);
                        event.start();
                        $gameSwitches.setValue(detectionSwitchId, false); // イベント起動時にスイッチをOFFにする
                        return true;
                    }
                }
            }
        }
        return false;
    };

    const _Game_Player_triggerAction = Game_Player.prototype.triggerAction;
    Game_Player.prototype.triggerAction = function() {
        if (this.triggerButtonAction()) {
            return true;
        }
        return _Game_Player_triggerAction.call(this);
    };
})();
