/*:
 * @target MZ
 * @plugindesc キャラクターに影を表示する（Shadow1画像使用、安全な初期化対応）
 * @help
 * ・プレイヤーには影が常に表示されます。
 * ・イベントには、メモ欄に <ShowShadow> を記述すると影が表示されます。
 * ・スクリプトで showShadow() / hideShadow() を呼ぶことで制御可能です。
 */

(() => {
    const SHADOW_OFFSET_X = 0;
    const SHADOW_OFFSET_Y = 6;
    const SHADOW_OPACITY = 192;
    const SHADOW_SCALE = 1.0;

    // 影スプライトの定義
    class Sprite_CharacterShadow extends Sprite {
        constructor(character) {
            super(ImageManager.loadSystem("Shadow1"));
            this._character = character;
            this.anchor.set(0.5, 0.5);
            this.opacity = SHADOW_OPACITY;
            this.scale.set(SHADOW_SCALE, SHADOW_SCALE);
            this.added = false;
        }

        update() {
            super.update();
            this.visible = this._character.isShadowVisible();
            this.x = this._character.screenX() + SHADOW_OFFSET_X;
            this.y = this._character.screenY() + SHADOW_OFFSET_Y;
        }
    }

    // スプライト初期化と影の遅延追加
    const _Sprite_Character_initialize = Sprite_Character.prototype.initialize;
    Sprite_Character.prototype.initialize = function(character) {
        _Sprite_Character_initialize.call(this, character);
        this._shadowSprite = new Sprite_CharacterShadow(character);
    };

    const _Sprite_Character_update = Sprite_Character.prototype.update;
    Sprite_Character.prototype.update = function() {
        _Sprite_Character_update.call(this);
        if (this._shadowSprite) {
            if (!this._shadowSprite.added && this.parent) {
                this.parent.addChild(this._shadowSprite);
                this._shadowSprite.added = true;
            }
            this._shadowSprite.update();
        }
    };

    // キャラクター共通：影フラグと制御メソッド
    const _Game_CharacterBase_initialize = Game_CharacterBase.prototype.initialize;
    Game_CharacterBase.prototype.initialize = function() {
        _Game_CharacterBase_initialize.call(this);
        this._showShadow = false; // 初期状態は非表示
    };

    Game_CharacterBase.prototype.showShadow = function() {
        this._showShadow = true;
    };

    Game_CharacterBase.prototype.hideShadow = function() {
        this._showShadow = false;
    };

    Game_CharacterBase.prototype.isShadowVisible = function() {
        return this._showShadow && !this.isTransparent() && this.hasImage();
    };

    Game_CharacterBase.prototype.hasImage = function() {
        return this._tileId > 0 || (this._characterName && this._characterName !== '');
    };

    // プレイヤーは常に影を表示
    const _Game_Player_initialize = Game_Player.prototype.initialize;
    Game_Player.prototype.initialize = function() {
        _Game_Player_initialize.call(this);
        this.showShadow();
    };

    // イベント：メモ欄タグによる影制御
    const _Game_Event_setupPage = Game_Event.prototype.setupPage;
    Game_Event.prototype.setupPage = function() {
        _Game_Event_setupPage.call(this);
        if (this.event().note.includes("<ShowShadow>")) {
            this.showShadow();
        } else {
            this.hideShadow();
        }
    };
})();
