
/*:
 * @target MZ
 * @plugindesc デバッグオーバーレイ v12.2（非表示中の負荷停止・表示安定） 
 * 
 * @help
 * Tabキーで表示ON/OFF。Ctrl+OKで次ページ。
 * 変化した値は90Fオレンジで表示。
 * 背景プレートとアウトラインで視認性を強化。
 * 表示OFF時は更新を完全停止して負荷軽減。
 */

(() => {
    const MAX_PER_PAGE = 40;
    const HIGHLIGHT_DURATION = 10;
    const HIGHLIGHT_COLOR = 14;

    const CUSTOM_EXPRESSIONS = [
        { label: "マップ名", expr: "$gameMap.mapId() + ':' + ($dataMapInfos[$gameMap.mapId()]?.name || '???')" },
        { label: "座標", expr: "$gamePlayer.x + ',' + $gamePlayer.y" },
        { label: "実行中EV", expr: "$gameMap._interpreter?._eventId || 'なし'" },
        { label: "実行中CE", expr: "$gameMap._interpreter?._commonEventId || 'なし'" }
    ];

    Input.keyMapper[9] = "tab"; // Tab
    let overlayVisible = false;
    let debugOverlayWindow = null;
    let switchPage = 0;
    let variablePage = 0;

    class Window_DebugOverlay extends Window_Base {
        initialize() {
            const width = 1000;
            const height = 600;
            const x = (Graphics.boxWidth - width) / 2;
            const y = (Graphics.boxHeight - height) / 2;
            super.initialize(new Rectangle(x, y, width, height));
            this.opacity = 0;
            this.contents.fontSize = 10;
            this.contents.fontFace = "mplus-1m-regular";
            this.contents.outlineWidth = 2;
            this.contents.outlineColor = "rgba(0, 0, 0, 0.5)";
            this._highlightTimers = {};
            this._lastValues = {};
            this.contents.paintOpacity = 128;
        }

        markHighlight(key, value) {
            if (this._lastValues[key] !== value) {
                this._highlightTimers[key] = HIGHLIGHT_DURATION;
                this._lastValues[key] = value;
            }
        }

        isHighlighted(key) {
            return this._highlightTimers[key] > 0;
        }

        refresh() {
            this.contents.clear();
            this.contentsBack.clear();
            this.contentsBack.fillRect(0, 0, this.width, this.height, 'rgba(0, 0, 0, 0.25)');

            const colWidth = Math.floor(this.innerWidth / 3);
            const switchStart = switchPage * MAX_PER_PAGE;
            const variableStart = variablePage * MAX_PER_PAGE;

            const switchMax = $dataSystem.switches.length - 1;
            const variableMax = $dataSystem.variables.length - 1;

            const switchIds = Array.from({ length: MAX_PER_PAGE }, (_, i) => i + 1 + switchStart)
                .filter(id => id <= switchMax);
            const variableIds = Array.from({ length: MAX_PER_PAGE }, (_, i) => i + 1 + variableStart)
                .filter(id => id <= variableMax);
            const maxLines = Math.max(switchIds.length, variableIds.length, CUSTOM_EXPRESSIONS.length);

            for (let i = 0; i < maxLines; i++) {
                const y = i * 12;

                if (i < switchIds.length) {
                    const id = switchIds[i];
                    const name = $dataSystem.switches[id] || "";
                    const value = $gameSwitches.value(id) ? "ON" : "OFF";
                    const key = `switch_${id}`;
                    this.markHighlight(key, value);
                    if (this.isHighlighted(key)) this.changeTextColor(ColorManager.textColor(HIGHLIGHT_COLOR));
                    this.drawText(`${id.padZero(4)} ${name}: ${value}`, 0, y, colWidth);
                    this.resetTextColor();
                }

                if (i < variableIds.length) {
                    const id = variableIds[i];
                    const name = $dataSystem.variables[id] || "";
                    const value = $gameVariables.value(id);
                    const key = `var_${id}`;
                    this.markHighlight(key, value);
                    if (this.isHighlighted(key)) this.changeTextColor(ColorManager.textColor(HIGHLIGHT_COLOR));
                    this.drawText(`${id.padZero(4)} ${name}: ${value}`, colWidth, y, colWidth);
                    this.resetTextColor();
                }

                if (i < CUSTOM_EXPRESSIONS.length) {
                    const entry = CUSTOM_EXPRESSIONS[i];
                    let result = null;
                    try {
                        result = eval(entry.expr);
                    } catch (e) {
                        result = "(error)";
                    }
                    const key = `expr_${i}`;
                    this.markHighlight(key, result);
                    if (this.isHighlighted(key)) this.changeTextColor(ColorManager.textColor(HIGHLIGHT_COLOR));
                    this.drawText(`${entry.label}: ${result}`, colWidth * 2, y, colWidth);
                    this.resetTextColor();
                }
            }
        }

        update() {
            super.update();
            if (!this.visible) return;

            this._frameCount = (this._frameCount || 0) + 1;
            if (this._frameCount % 5 !== 0) return;

            let needsRefresh = false;

            for (const key in this._highlightTimers) {
                if (this._highlightTimers[key] > 0) {
                    this._highlightTimers[key]--;
                    needsRefresh = true;
                }
            }

            const currentValues = {};
            const switchMax = $dataSystem.switches.length - 1;
            const variableMax = $dataSystem.variables.length - 1;

            const switchStart = switchPage * MAX_PER_PAGE;
            const variableStart = variablePage * MAX_PER_PAGE;

            const switchIds = Array.from({ length: MAX_PER_PAGE }, (_, i) => i + 1 + switchStart)
                .filter(id => id <= switchMax);
            const variableIds = Array.from({ length: MAX_PER_PAGE }, (_, i) => i + 1 + variableStart)
                .filter(id => id <= variableMax);

            for (const id of switchIds) {
                const value = $gameSwitches.value(id) ? "ON" : "OFF";
                const key = `switch_${id}`;
                currentValues[key] = value;
                if (this._lastValues[key] !== value) {
                    this._highlightTimers[key] = HIGHLIGHT_DURATION;
                    needsRefresh = true;
                }
            }

            for (const id of variableIds) {
                const value = $gameVariables.value(id);
                const key = `var_${id}`;
                currentValues[key] = value;
                if (this._lastValues[key] !== value) {
                    this._highlightTimers[key] = HIGHLIGHT_DURATION;
                    needsRefresh = true;
                }
            }

            CUSTOM_EXPRESSIONS.forEach((entry, i) => {
                let result;
                try {
                    result = eval(entry.expr);
                } catch (e) {
                    result = "(error)";
                }
                const key = `expr_${i}`;
                currentValues[key] = result;
                if (this._lastValues[key] !== result) {
                    this._highlightTimers[key] = HIGHLIGHT_DURATION;
                    needsRefresh = true;
                }
            });

            if (needsRefresh) {
                this._lastValues = currentValues;
                this.refresh();
            }
        }


        captureInitialValues() {
            const switchMax = $dataSystem.switches.length - 1;
            const variableMax = $dataSystem.variables.length - 1;

            for (let id = 1; id <= switchMax; id++) {
                this._lastValues[`switch_${id}`] = $gameSwitches.value(id) ? "ON" : "OFF";
            }
            for (let id = 1; id <= variableMax; id++) {
                this._lastValues[`var_${id}`] = $gameVariables.value(id);
            }
            CUSTOM_EXPRESSIONS.forEach((entry, i) => {
                let result = null;
                try {
                    result = eval(entry.expr);
                } catch (e) {
                    result = null;
                }
                this._lastValues[`expr_${i}`] = result;
            });
        }
    }

    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function () {
        _Scene_Map_createAllWindows.call(this);
        debugOverlayWindow = new Window_DebugOverlay();
        debugOverlayWindow.captureInitialValues();
        this.addChild(debugOverlayWindow);
        debugOverlayWindow.visible = overlayVisible;
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);
        if (debugOverlayWindow) {
            debugOverlayWindow.visible = overlayVisible;
        }
    };

    const _SceneManager_updateInputData = SceneManager.updateInputData;
    SceneManager.updateInputData = function () {
        _SceneManager_updateInputData.call(this);

        if (Input.isTriggered("tab")) {
            overlayVisible = !overlayVisible;
            if (debugOverlayWindow) {
                debugOverlayWindow.visible = overlayVisible;
            }
        }

        if ((Input.isPressed("control") || Input.isPressed("shift")) &&
            (Input.isTriggered("ok") || TouchInput.isTriggered())) {
            const switchMax = $dataSystem.switches.length - 1;
            const variableMax = $dataSystem.variables.length - 1;
            const maxSwitchPages = Math.floor(switchMax / MAX_PER_PAGE);
            const maxVariablePages = Math.floor(variableMax / MAX_PER_PAGE);
            switchPage++;
            variablePage++;
            if (switchPage > maxSwitchPages && variablePage > maxVariablePages) {
                switchPage = 0;
                variablePage = 0;
            }
        }
    };

    Number.prototype.padZero = function (length) {
        return this.toString().padStart(length, "0");
    };
})();
