/*:
 * @target MZ
 * @plugindesc 到達範囲オーバーレイ（差分更新・画面外除去対応）
 */

(() => {
  const STEP_LIMIT = 6;
  const FRAME_SKIP = 20;
  const TILE_SIZE = 48;
  const COLOR = "#4488ff";
  let TAB_COUNT = 0;
  let _enabled = false;
  let _needsClear = false;

  class Sprite_ReachableOverlay extends Sprite {
    constructor() {
      super();
      this._frame = 0;
      this._spriteMap = new Map(); // key: "x,y", value: Sprite
    }

    update() {
      super.update();
      if (_needsClear) {
        this.clearAll();
        _needsClear = false;
      }
      if (!_enabled) return;
      this._frame++;
      this.relocate();
      if (this._frame % FRAME_SKIP !== 0) return;
      this.refresh();
    }

    relocate() {
      for (const [key, sprite] of this._spriteMap) {
        sprite.x = sprite.tileX * TILE_SIZE - $gameMap.displayX() * TILE_SIZE;
        sprite.y = sprite.tileY * TILE_SIZE - $gameMap.displayY() * TILE_SIZE;
      }
    }

    refresh() {
      const nextTiles = collectReachableTiles(STEP_LIMIT);
      const nextKeys = new Set();
      for (const {x, y} of nextTiles) {
        const key = `${x},${y}`;
        nextKeys.add(key);
        if (!this._spriteMap.has(key)) {
          if (!isOnScreen(x, y)) continue;
          const sprite = this.createTileSprite(x, y);
          this._spriteMap.set(key, sprite);
          this.addChild(sprite);
        }
      }

      for (const key of this._spriteMap.keys()) {
        if (!nextKeys.has(key) || !isOnScreenKey(key)) {
          const sprite = this._spriteMap.get(key);
          this.removeChild(sprite);
          sprite.destroy();
          this._spriteMap.delete(key);
        }
      }
    }

    clearAll() {
      for (const sprite of this._spriteMap.values()) {
        this.removeChild(sprite);
        sprite.destroy();
      }
      this._spriteMap.clear();
    }

    createTileSprite(x, y) {
      const bmp = new Bitmap(TILE_SIZE, TILE_SIZE);
      bmp.fillAll(COLOR);
      const sprite = new Sprite(bmp);
      sprite.opacity = 64;
      sprite.tileX = x;
      sprite.tileY = y;
      sprite.x = x * TILE_SIZE - $gameMap.displayX() * TILE_SIZE;
      sprite.y = y * TILE_SIZE - $gameMap.displayY() * TILE_SIZE;
      return sprite;
    }
  }

  function isOnScreenKey(key) {
    const [x, y] = key.split(',').map(Number);
    return isOnScreen(x, y);
  }

  function collectReachableTiles(limit) {
    const result = [];
    const visited = new Set();
    const startX = Math.ceil($gamePlayer._realX);
    const startY = Math.ceil($gamePlayer._realY);
    const queue = [{ x: startX, y: startY, step: 0 }];
    visited.add(`${startX},${startY}`);
    while (queue.length > 0) {
      const { x, y, step } = queue.shift();
      result.push({ x, y });
      if (step >= limit) continue;
      for (const [dx, dy] of [[1,0], [-1,0], [0,1], [0,-1]]) {
        const nx = x + dx;
        const ny = y + dy;
        const key = `${nx},${ny}`;
        if (visited.has(key)) continue;
        if (!$gameMap.isValid(nx, ny)) continue;
        if (!canPassTile(x, y, dx, dy)) continue;
        if (hasSolidEventAt(nx, ny)) continue;
        visited.add(key);
        queue.push({ x: nx, y: ny, step: step + 1 });
      }
    }
    return result;
  }

  function canPassTile(x, y, dx, dy) {
    const d1 = dirFromDelta(dx, dy);
    const d2 = dirFromDelta(-dx, -dy);
    return $gameMap.isPassable(x, y, d1) && $gameMap.isPassable(x + dx, y + dy, d2);
  }

  function hasSolidEventAt(x, y) {
    return $gameMap.eventsXy(x, y).some(ev =>
      ev.isNormalPriority() &&
      ev.characterName() !== "" &&
      !ev.isThrough()
    );
  }

  function dirFromDelta(dx, dy) {
    if (dx === 1) return 6;
    if (dx === -1) return 4;
    if (dy === 1) return 2;
    if (dy === -1) return 8;
    return 0;
  }

  function isOnScreen(x, y) {
    const sx = x * TILE_SIZE - $gameMap.displayX() * TILE_SIZE;
    const sy = y * TILE_SIZE - $gameMap.displayY() * TILE_SIZE;
    return sx >= -TILE_SIZE && sy >= -TILE_SIZE && sx <= Graphics.width && sy <= Graphics.height;
  }

  const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
  Scene_Map.prototype.createAllWindows = function() {
    _Scene_Map_createAllWindows.call(this);
    this._reachableOverlay = new Sprite_ReachableOverlay();
    this.addChild(this._reachableOverlay);
  };

  const _Scene_Map_updateMain = Scene_Map.prototype.updateMain;
  Scene_Map.prototype.updateMain = function() {
    _Scene_Map_updateMain.call(this);
    if (this._reachableOverlay) this._reachableOverlay.update();
  };

  const _Input_update = Input.update;
  Input.update = function() {
    _Input_update.call(this);
    if (Input.isTriggered("tab")) toggleReachableOverlay();
  };

  function toggleReachableOverlay() {
    TAB_COUNT++;
    if (TAB_COUNT === 2) {
      _enabled = !_enabled;
      if (!_enabled) _needsClear = true;
      TAB_COUNT = 0;
    }
  }

  const _Scene_Map_terminate = Scene_Map.prototype.terminate;
  Scene_Map.prototype.terminate = function() {
    if (this._reachableOverlay) {
      this._reachableOverlay.clearAll();
      this._reachableOverlay.destroy();
      this._reachableOverlay = null;
    }
    _Scene_Map_terminate.call(this);
  };
})();
