"use strict";
/*:
 * @target MZ
 * @plugindesc [DevTool] 地形ID可視化オーバーレイ（イベント情報と連動） v1.3（キャッシュ＋座標更新対応）
 *
 * @help
 * イベント情報オーバーレイと連動してON/OFFされる、地形ID 1/2/3 の可視化。
 * - 表示対象：画面内のタイルのみ
 * - 条件：地形IDが1, 2, 3
 * - 内容：該当タイル上に地形IDを一度描画し、座標はスクロールに応じて更新
 * - 更新間隔：3フレームに1回（追加・削除・位置更新）
 */

(() => {
  const FRAME_SKIP = 3;
  const TILE_SIZE = 48;
  const targetTerrainIds = [1, 2, 3];
  let overlayVisible = false;
  let tabPressCount = 0;

  Input.keyMapper[9] = "tab";

  class Sprite_TerrainOverlay extends Sprite {
    constructor() {
      super();
      this._frameCount = 0;
      this._tileSprites = new Map(); // key: "x,y", value: {sprite, mapX, mapY}
    }

    update() {
      if (!overlayVisible) return;
      if (++this._frameCount % FRAME_SKIP !== 0) return;

      const startX = Math.floor($gameMap.displayX());
      const startY = Math.floor($gameMap.displayY());
      const visibleTilesX = Math.ceil(Graphics.boxWidth / TILE_SIZE) - 1;
      const visibleTilesY = Math.ceil(Graphics.boxHeight / TILE_SIZE) - 1;
      const endX = startX + visibleTilesX;
      const endY = startY + visibleTilesY;

      const visibleKeys = new Set();

      for (let x = startX; x <= endX; x++) {
        for (let y = startY; y <= endY; y++) {
          const terrainId = $gameMap.terrainTag(x, y);
          if (!targetTerrainIds.includes(terrainId)) continue;

          const key = `${x},${y}`;
          visibleKeys.add(key);

          const sx = $gameMap.adjustX(x) * TILE_SIZE + TILE_SIZE / 2;
          const sy = $gameMap.adjustY(y) * TILE_SIZE + TILE_SIZE / 2;

          if (!this._tileSprites.has(key)) {
            const bitmap = new Bitmap(24, 24);
            bitmap.fontFace = "mplus-1m-regular";
            bitmap.fontSize = 14;
            bitmap.outlineWidth = 2;
            bitmap.outlineColor = "rgba(0,0,0,0.6)";
            bitmap.textColor = ColorManager.textColor(0);
            bitmap.drawText(String(terrainId), 0, 0, 24, 24, "center");

            const sprite = new Sprite(bitmap);
            sprite.anchor.set(0.5, 0.5);
            sprite.x = sx;
            sprite.y = sy;
            this.addChild(sprite);
            this._tileSprites.set(key, { sprite, mapX: x, mapY: y });
          } else {
            // スクロールに応じて位置更新
            const obj = this._tileSprites.get(key);
            obj.sprite.x = sx;
            obj.sprite.y = sy;
          }
        }
      }

      // 削除対象を検出し、removeChildして破棄
      for (const [key, obj] of this._tileSprites.entries()) {
        if (!visibleKeys.has(key)) {
          this.removeChild(obj.sprite);
          this._tileSprites.delete(key);
        }
      }
    }

    clear() {
      for (const { sprite } of this._tileSprites.values()) {
        this.removeChild(sprite);
      }
      this._tileSprites.clear();
    }
  }

  const _Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
  Spriteset_Map.prototype.createLowerLayer = function () {
    _Spriteset_Map_createLowerLayer.call(this);
    this._terrainOverlay = new Sprite_TerrainOverlay();
    this.addChild(this._terrainOverlay);
  };

  const _Spriteset_Map_update = Spriteset_Map.prototype.update;
  Spriteset_Map.prototype.update = function () {
    _Spriteset_Map_update.call(this);
    if (this._terrainOverlay && overlayVisible) {
      this._terrainOverlay.update();
    }
  };

  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function () {
    _Scene_Map_update.call(this);

    if (Input.isTriggered("tab")) {
      tabPressCount++;
    } else if (tabPressCount >= 2) {
      overlayVisible = !overlayVisible;
      tabPressCount = 0;

      const spriteset = SceneManager._scene._spriteset;
      if (!overlayVisible) {
        if (spriteset._terrainOverlay) {
          spriteset._terrainOverlay.clear();
        }
        if (spriteset._eventOverlayMap && spriteset._eventOverlayContainer) {
          for (const id in spriteset._eventOverlayMap) {
            spriteset._eventOverlayContainer.removeChild(spriteset._eventOverlayMap[id]);
          }
          spriteset._eventOverlayMap = {};
        }
      }
    }
  };
})();