//=============================================================================
// Yanfly Engine Plugins - Map Select Skill
// YEP_MapSelectSkill.js
//=============================================================================

var Imported = Imported || {};
Imported.YEP_MapSelectSkill = true;

var Yanfly = Yanfly || {};
Yanfly.MSS = Yanfly.MSS || {};
Yanfly.MSS.version = 1.03;

//=============================================================================
 /*:ja
 * @plugindesc v1.03 スキルのリストを表示するウィンドウを表示し、スキルIDを変数に代入します。
 * @author Yanfly Engine Plugins
 *
 * @param Default Columns
 * @type number
 * @min 1
 * @desc ウィンドウのデフォルトの列数。
 * @default 2
 *
 * @param Default Rows
 * @type number
 * @min 1
 * @desc ウィンドウのデフォルトの行数。
 * @default 4
 *
 * @param Default X Position
 * @type combo
 * @option left
 * @option center
 * @option right
 * @desc デフォルトのウィンドウX位置。
 * left     center     right
 * @default right
 *
 * @param Default Y Position
 * @type combo
 * @option top
 * @option middle
 * @option bottom
 * @desc デフォルトのウィンドウY位置。
 * top     middle     bottom
 * @default bottom
 *
 * @param Default Width
 * @type number
 * @min 0
 * @desc ウィンドウのデフォルト幅。0に設定すると、ウィンドウ幅は画面幅になります。
 * @default 0
 *
 * @param Default Enable
 * @type boolean
 * @on 有効
 * @off 無効
 * @desc デフォルトですべてのスキルを有効にしますか？
 * 無効 - false     有効 - true
 * @default true
 *
 * @param Default Cost
 * @type boolean
 * @on 表示
 * @off 非表示
 * @desc デフォルトでスキルのコストを表示しますか？
 * 非表示 - false     表示 - true
 * @default true
 *
 * @help
 * 翻訳:ムノクラ
 * https://munokura.tk/
 * https://twitter.com/munokura/
 *
 * ============================================================================
 * Introduction
 * ============================================================================
 *
 * このプラグインはスキルのリストを表示するウィンドウを作成します。
 * スキルが選択されると、
 * 選択されたスキルのIDの値を選択された変数に代入するように設定されます。
 *
 * ============================================================================
 * Plugin Commands
 * ============================================================================
 *
 * このプラグインを利用するには、次のプラグインコマンドを使用します。
 *
 * プラグインコマンド
 *
 * MapSelectSkill v a s
 * - マップ選択スキルウィンドウが開きます。
 * 選択したスキルに設定したい変数のIDで'v'を置き換えます。
 * スキルリストを見たいアクターのIDで'a'を置き換えてください。
 * 's'はオプションですが、使用すると、
 * スキルタイプ's'のスキルのみが表示されます。
 * 's'をスキルタイプのID番号に置き換えます。
 *
 * MapSelectSkillColumns x
 * - マップ選択スキルウィンドウの列数をxに設定します。
 *
 * MapSelectSkillRows x
 * - マップ選択スキルウィンドウの行数をxに設定します。
 *
 * MapSelectSkillWidth x
 * - マップ選択スキルウィンドウの幅をxに設定します。
 * 0を使用すると、ウィンドウ幅は画面幅になります。
 *
 * MapSelectSkillX left
 * MapSelectSkillX center
 * MapSelectSkillX right
 * - マップ選択スキルウィンドウを
 * 画面の左側、画面の中央、画面の右側に配置します。
 *
 * MapSelectSkillY top
 * MapSelectSkillY middle
 * MapSelectSkillY bottom
 * - マップ選択スキルウィンドウを
 * 画面の上部、画面の中央、画面の下部に配置します。
 *
 * EnableAllMapSelectSkills
 * - アクターがそれらを使用できるかどうかにかかわらず、
 * リストされているすべてのスキルが選択可能になります。
 *
 * NormalAllMapSelectSkills
 * - アクターがそのスキルを使用できるかどうかに基づいて、
 * リストされているすべてのスキルを有効または無効にします。
 *
 * ShowMapSelectSkillCost
 * - マップ選択スキルウィンドウにスキルのコストを表示します。
 *
 * HideMapSelectSkillCost
 * - マップ選択スキルウィンドウでスキルのコストを隠します。
 *
 * ============================================================================
 * Changelog
 * ============================================================================
 *
 * Version 1.03:
 * - Updated for RPG Maker MV version 1.5.0.
 *
 * Version 1.02:
 * - Optimization update.
 *
 * Version 1.01:
 * - Compatibility update with Self Switches & Variables v1.01.
 *
 * Version 1.00:
 * - Finished Plugin!
 */
//=============================================================================

Yanfly.Parameters = PluginManager.parameters('YEP_MapSelectSkill');
Yanfly.Param = Yanfly.Param || {};

Yanfly.Param.MSSCol = Number(Yanfly.Parameters['Default Columns']);
Yanfly.Param.MSSRow = Number(Yanfly.Parameters['Default Rows']);
Yanfly.Param.MSSPosX = String(Yanfly.Parameters['Default X Position']);
Yanfly.Param.MSSPosY = String(Yanfly.Parameters['Default Y Position']);
Yanfly.Param.MSSWidth = Number(Yanfly.Parameters['Default Width']);
Yanfly.Param.MSSEnable = eval(String(Yanfly.Parameters['Default Enable']));
Yanfly.Param.MSSCost = eval(String(Yanfly.Parameters['Default Cost']));

//=============================================================================
// Game_System
//=============================================================================

Yanfly.MSS.Game_System_initialize = Game_System.prototype.initialize;
Game_System.prototype.initialize = function() {
    Yanfly.MSS.Game_System_initialize.call(this);
    this.initMapSelectSkill();
};

Game_System.prototype.initMapSelectSkill = function() {
    this._mapSelectSkillWindowColumns = Yanfly.Param.MSSCol;
    this._mapSelectSkillWindowRows = Yanfly.Param.MSSRow;
    this._mapSelectSkillWindowPosX = Yanfly.Param.MSSPosX;
    this._mapSelectSkillWindowPosY = Yanfly.Param.MSSPosY;
    this._mapSelectSkillWindowWidth = Yanfly.Param.MSSWidth;
    this._mapSelectSkillWindowEnable = Yanfly.Param.MSSEnable;
    this._mapSelectSkillWindowCosts = Yanfly.Param.MSSCost;
};

Game_System.prototype.getMapSelectSkillColumns = function() {
    if (this._mapSelectSkillWindowColumns === undefined) {
      this.initMapSelectSkill();
    }
    return this._mapSelectSkillWindowColumns;
};

Game_System.prototype.setMapSelectSkillColumns = function(value) {
    if (this._mapSelectSkillWindowColumns === undefined) {
      this.initMapSelectSkill();
    }
    this._mapSelectSkillWindowColumns = value;
};

Game_System.prototype.getMapSelectSkillRows = function() {
    if (this._mapSelectSkillWindowRows === undefined) {
      this.initMapSelectSkill();
    }
    return this._mapSelectSkillWindowRows;
};

Game_System.prototype.setMapSelectSkillRows = function(value) {
    if (this._mapSelectSkillWindowRows === undefined) {
      this.initMapSelectSkill();
    }
    this._mapSelectSkillWindowRows = value;
};

Game_System.prototype.getMapSelectSkillPosX = function() {
    if (this._mapSelectSkillWindowPosX === undefined) {
      this.initMapSelectSkill();
    }
    return this._mapSelectSkillWindowPosX;
};

Game_System.prototype.setMapSelectSkillPosX = function(value) {
    if (this._mapSelectSkillWindowPosX === undefined) {
      this.initMapSelectSkill();
    }
    this._mapSelectSkillWindowPosX = value;
};

Game_System.prototype.getMapSelectSkillPosY = function() {
    if (this._mapSelectSkillWindowPosY === undefined) {
      this.initMapSelectSkill();
    }
    return this._mapSelectSkillWindowPosY;
};

Game_System.prototype.setMapSelectSkillPosY = function(value) {
    if (this._mapSelectSkillWindowPosY === undefined) {
      this.initMapSelectSkill();
    }
    this._mapSelectSkillWindowPosY = value;
};

Game_System.prototype.getMapSelectSkillWidth = function() {
    if (this._mapSelectSkillWindowWidth === undefined) {
      this.initMapSelectSkill();
    }
    return this._mapSelectSkillWindowWidth;
};

Game_System.prototype.setMapSelectSkillWidth = function(value) {
    if (this._mapSelectSkillWindowWidth === undefined) {
      this.initMapSelectSkill();
    }
    this._mapSelectSkillWindowWidth = value;
};

Game_System.prototype.getMapSelectSkillEnable = function() {
    if (this._mapSelectSkillWindowEnable === undefined) {
      this.initMapSelectSkill();
    }
    return this._mapSelectSkillWindowEnable;
};

Game_System.prototype.setMapSelectSkillEnable = function(value) {
    if (this._mapSelectSkillWindowEnable === undefined) {
      this.initMapSelectSkill();
    }
    this._mapSelectSkillWindowEnable = value;
};

Game_System.prototype.getMapSelectSkillCosts = function() {
    if (this._mapSelectSkillWindowCosts === undefined) {
      this.initMapSelectSkill();
    }
    return this._mapSelectSkillWindowCosts;
};

Game_System.prototype.setMapSelectSkillCosts = function(value) {
    if (this._mapSelectSkillWindowCosts === undefined) {
      this.initMapSelectSkill();
    }
    this._mapSelectSkillWindowCosts = value;
};

//=============================================================================
// Game_Interpreter
//=============================================================================

Yanfly.MSS.Game_Interpreter_pluginCommand =
    Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function(command, args) {
  Yanfly.MSS.Game_Interpreter_pluginCommand.call(this, command, args);
  if (command === 'MapSelectSkill') {
    if (SceneManager._scene instanceof Scene_Map) {
      var varId = parseInt(args[0]);
      var actorId = parseInt(args[1]);
      var stypeId = parseInt(args[2] || 0);
      SceneManager._scene.setupMapSelectSkill(varId, actorId, stypeId);
      this.wait(10);
    }
  } else if (command === 'MapSelectSkillColumns') {
    var value = parseInt(args[0]);
    $gameSystem.setMapSelectSkillColumns(value);
  } else if (command === 'MapSelectSkillRows') {
    var value = parseInt(args[0]);
    $gameSystem.setMapSelectSkillRows(value);
  } else if (command === 'MapSelectSkillWidth') {
    var value = parseInt(args[0]);
    $gameSystem.setMapSelectSkillWidth(value);
  } else if (command === 'MapSelectSkillX') {
    var value = String(args[0]).toLowerCase();
    $gameSystem.setMapSelectSkillPosX(value);
  } else if (command === 'MapSelectSkillY') {
    var value = String(args[0]).toLowerCase();
    $gameSystem.setMapSelectSkillPosY(value);
  } else if (command === 'EnableAllMapSelectSkills') {
    $gameSystem.setMapSelectSkillEnable(true);
  } else if (command === 'NormalAllMapSelectSkills') {
    $gameSystem.setMapSelectSkillEnable(false);
  } else if (command === 'ShowMapSelectSkillCost') {
    $gameSystem.setMapSelectSkillCosts(true);
  } else if (command === 'HideMapSelectSkillCost') {
    $gameSystem.setMapSelectSkillCosts(false);
  }
};

//=============================================================================
// Window_MapSelectSkill
//=============================================================================

function Window_MapSelectSkill() {
    this.initialize.apply(this, arguments);
}

Window_MapSelectSkill.prototype = Object.create(Window_SkillList.prototype);
Window_MapSelectSkill.prototype.constructor = Window_MapSelectSkill;

Window_MapSelectSkill.prototype.initialize = function() {
    var width = this.windowWidth();
    var height = this.windowHeight();
    Window_Selectable.prototype.initialize.call(this, 0, 0, width, height);
    this.openness = 0;
};

Window_MapSelectSkill.prototype.windowWidth = function() {
    return this._windowWidth || Graphics.boxWidth;
};

Window_MapSelectSkill.prototype.windowHeight = function() {
    return this._windowHeight || this.fittingHeight(4);
};

Window_MapSelectSkill.prototype.setup = function(varId, actorId, stypeId) {
    if (!varId) return;
    if (!actorId) return;
    this.updateWindowSettings();
    this._varId = varId;
    this.setActor($gameActors.actor(actorId));
    this.setStypeId(stypeId);
    this.refresh();
    this.activate();
    this.open();
    this.select(0);
};

Window_MapSelectSkill.prototype.includes = function(item) {
    if (!this._stypeId) return item;
    return item && item.stypeId === this._stypeId;
};

Window_MapSelectSkill.prototype.maxCols = function() {
    return $gameSystem.getMapSelectSkillColumns() || 1;
};

Window_MapSelectSkill.prototype.updateWindowSettings = function() {
    this.width = $gameSystem.getMapSelectSkillWidth() || Graphics.boxWidth;
    var col = $gameSystem.getMapSelectSkillRows() || 4;
    this.height = this.fittingHeight(col);
    if ($gameSystem.getMapSelectSkillPosX() === 'left') {
      this.x = 0;
    } else if ($gameSystem.getMapSelectSkillPosX() === 'center') {
      this.x = Math.floor((Graphics.boxWidth - this.width) / 2);
    } else {
      this.x = Graphics.boxWidth - this.width;
    }
    if ($gameSystem.getMapSelectSkillPosY() === 'top') {
      this.y = 0;
    } else if ($gameSystem.getMapSelectSkillPosY() === 'middle') {
      this.y = Math.floor((Graphics.boxHeight - this.height) / 2);
    } else {
      this.y = Graphics.boxHeight - this.height;
    }
};

Window_MapSelectSkill.prototype.isEnabled = function(item) {
    if ($gameSystem.getMapSelectSkillEnable()) return true;
    return Window_SkillList.prototype.isEnabled.call(this, item);
};

Window_MapSelectSkill.prototype.drawSkillCost = function(skill, x, y, width) {
    if ($gameSystem.getMapSelectSkillCosts()) {
      var width =
        Window_SkillList.prototype.drawSkillCost.call(this, skill, x, y, width);
    }
};

//=============================================================================
// Scene_Map
//=============================================================================

Yanfly.MSS.Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
Scene_Map.prototype.createAllWindows = function() {
    Yanfly.MSS.Scene_Map_createAllWindows.call(this);
    this.createMapSelectSkillWindow();
};

Scene_Map.prototype.createMapSelectSkillWindow = function() {
    this._mapSelectSkillWindow = new Window_MapSelectSkill();
    this._mapSelectSkillWindow.setHandler('ok', 
      this.onMapSelectSkillOk.bind(this));
    this._mapSelectSkillWindow.setHandler('cancel', 
      this.onMapSelectSkillCancel.bind(this));
    this.addChild(this._mapSelectSkillWindow);
};

Scene_Map.prototype.setupMapSelectSkill = function(varId, actorId, stypeId) {
    this._mapSelectSkillWindow.setup(varId, actorId, stypeId);
    this._active = false;
};

Scene_Map.prototype.onMapSelectSkillOk = function() {
    this._mapSelectSkillWindow.close();
    var skill = this._mapSelectSkillWindow.item();
    var varId = this._mapSelectSkillWindow._varId;
    if (Imported.YEP_SelfSwVar) $gameTemp.clearSelfSwVarEvBridge();
    if (!skill) {
      $gameVariables.setValue(varId, 0);
    } else {
      $gameVariables.setValue(varId, skill.id);
    }
    if (Imported.YEP_SelfSwVar) $gameTemp.clearSelfSwVarEvent();
    this._active = true;
};

Scene_Map.prototype.onMapSelectSkillCancel = function() {
    this._mapSelectSkillWindow.close();
    var varId = this._mapSelectSkillWindow._varId;
    $gameVariables.setValue(varId, 0);
    this._active = true;
};

//=============================================================================
// End of File
//=============================================================================
