
/*:
 * @plugindesc Automatically generates a description of the skill (for debugging to see if it's working)
 * @target MZ
 * @url https://github.com/RYBA549/RYBA_RPGMakerMZ
 * @author RYBA
 *
 * @help AutoSkillDescription.js
 * 
 * MIT License Copyright (c) 2020 RYBA
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
/*:ja
 * @plugindesc スキルの説明文を自動的に生成します（効果があってるかのデバッグ用）
 * @target MZ
 * @url https://github.com/RYBA549/RYBA_RPGMakerMZ
 * @author RYBA(熱帯魚)
 *
 * @help AutoSkillDescription.js
 * 
 * 特殊効果の「逃げる」などは未対応です。
 * 2行目にダメージ式を代入してあります
 * 
 * <AutoDescriptionOff>
 * メモ欄に記載することで自動生成せずにスキルの説明がそのまま表示されます。
 *
 * 利用規約：
 *  これにより、このソフトウェアおよび関連するドキュメントファイル（以下「ソフトウェア」）のコピーを取得するすべての人に対して、
 * 使用、コピー、変更、マージの権利を含むがこれに限定されない制限なしでソフトウェアを扱うことが許可されます。
 * 次の条件に従って、本ソフトウェアのコピーを発行、配布、サブライセンス、および/または販売し、
 * 本ソフトウェアの提供を受けた者がそうすることを許可する。
 *
 * @command StoreSkillDescription
 * @text Store Skill Description
 * @desc Stores the auto-generated description of a skill in a variable.
 *
 * @arg skillIdSource
 * @type string
 * @text Skill ID or Variable
 * @desc Enter a skill ID or $x for variable (e.g. 1 or $1)
 *
 * @arg variableId
 * @type variable
 * @text Store in Variable
 * @desc The variable ID to store the skill description
 */

// BuffControlSelf タグをチェックするヘルパー関数
function checkBuffControlSelfTags(skill) {
    const tags = ['atk', 'mat', 'def', 'mdf'];
    const presentTags = tags.filter(tag => new RegExp(`<BuffControlSelf:\\s*${tag}`, 'i').test(skill.note));
    return presentTags;
}

function RYBA_CreateSkillDescriptionForCommand(skill) {
    if (!skill) {
        return '';
    }

    let description_add = '';
    let description_add_after = '';
    let description_result_after = '';
    const autoDescriptionOff = skill.meta.AutoDescriptionOff;

    // AddState タグの処理
    let addStatePrefix = '';
    const addStateMatch = skill.note.match(/<AddState:\s*(\d+)>/i);
    if (addStateMatch) {
        const stateId = parseInt(addStateMatch[1]);
        const state = $dataStates[stateId];
        if (state) {
            addStatePrefix = `自分に${state.name}を付与し、`;
        }
    }

    // BuffControlSelf タグの処理
    let buffControlDescription = '';
    const presentTags = checkBuffControlSelfTags(skill);
    
    if (presentTags.length === 4) {
        buffControlDescription = '自分の全能力を20%上昇（上限40%）させ、';
    } else if (presentTags.includes('atk') && presentTags.includes('mat')) {
        buffControlDescription = '自分の全攻撃を20%上昇（上限40%）させ、';
    } else if (presentTags.includes('def') && presentTags.includes('mdf')) {
        buffControlDescription = '自分の全防御を20%上昇（上限40%）させ、';
    } else if (presentTags.length > 0) {
        const abilityNames = {
            atk: '攻撃力',
            mat: '魔法力',
            def: '防御力',
            mdf: '魔法防御力'
        };
        const abilities = presentTags.map(tag => abilityNames[tag]).join('と');
        buffControlDescription = `自分の${abilities}を20%上昇（上限40%）させ、`;
    }

    if (skill && skill.note) {
        const noteTagMatch = /<DescriptionAdd:(.+)>/.exec(skill.note);
        if (noteTagMatch) {
            description_add = noteTagMatch[1];
        }
        const noteTagMatch_after = /<DescriptionAddAfter:(.+)>/.exec(skill.note);
        if (noteTagMatch_after) {
            description_add_after = noteTagMatch_after[1];
        }
        const noteTagMatch_result_after = /<DescriptionResultAfter:(.+)>/.exec(skill.note);
        if (noteTagMatch_result_after) {
            description_result_after = noteTagMatch_result_after[1];
        }
    }

    let result = '';
    if (autoDescriptionOff) {
        result = addStatePrefix + buffControlDescription + description_add + skill.description + description_add_after;
    } else {
        const action = new RYBA_SkillCheck(skill);
        let baseDescription = RYBA_CreateSkillDescription(skill);
        let damageInfo = Nakasa_skillFormatText(skill);

        // ダメージ情報を抽出
        let damageValue = damageInfo.match(/:(.*?)(>|\(防御無視\))/);
        damageValue = damageValue ? damageValue[1] : '';

        // 説明文を組み立てる
        if (action.isDamage()) {
            result = `${baseDescription}\\c[10]${damageValue}\\c[0]のダメージ`;
            if ((skill.damage.formula.includes("a.mat")) && (!skill.damage.formula.includes("b.mdf * 2"))) {
                result = `${baseDescription}\\c[10]${damageValue}\\c[0]の防御を無視した魔法ダメージ`;
            } else if (skill.damage.formula.includes("a.mat")) {
                result = `${baseDescription}\\c[10]${damageValue}\\c[0]の魔法ダメージ`;
            } else if (!skill.damage.formula.includes("b.def * 2")) {
                result = `${baseDescription}\\c[10]${damageValue}\\c[0]の防御を無視したダメージ`;
            }
        } else if (action.isDrain()) {
            result = `${baseDescription}\\c[10]${damageValue}\\c[0]の吸収`;
        } else if (action.isRecover()) {
            result = `${baseDescription}\\c[1]${damageValue}\\c[0]の回復`;
        } else {
            result = baseDescription;
        }

        // ステート付与と能力上昇の情報を取得
        const skillEffectData = new RYBA_SkillEffectText(action, false);
        const stateEffects = skillEffectData.getStateEffectText();
        const otherEffects = skillEffectData.getEffectText();
        
        // ステート付与と能力上昇の情報を追加
        if (stateEffects || otherEffects) {
            if (result.endsWith('に')) {
                result = result.slice(0, -1); // 末尾の "に" を削除
            } else if (!result.endsWith('ジ') && !result.endsWith('収') && !result.endsWith('復')) {
                result += RYBA_AndText;
            }
            if (stateEffects) result += ' ' + stateEffects;
            if (otherEffects) result += ' ' + otherEffects;
        }

        const numRepeatsTotal = action.numRepeatsTotal();
        if (numRepeatsTotal > 1) {
            result += '(x' + numRepeatsTotal + ')';
        }

        // 最終的な説明文の組み立て
        result = addStatePrefix + buffControlDescription + description_add + result + description_add_after + '\n' +
                 SkillUnlockRateText(skill) + WarmUpText(skill) + CoolDownText(skill) + 
                 TargetPercentText(skill) + description_result_after;

        // HPダメージかつ会心なしの場合、<会心:なし>を追加
        if (skill.damage.type === 1 && skill.damage.critical === false) {
            result += '<会心:なし>';
        }
    }

    return result;
};

// プラグインコマンドを更新
PluginManager.registerCommand("AutoSkillDescription", "StoreSkillDescription", args => {
    const skillIdSource = String(args.skillIdSource);
    const skillId = skillIdSource.startsWith('$') 
        ? $gameVariables.value(parseInt(skillIdSource.slice(1))) 
        : parseInt(skillIdSource);
    const variableId = parseInt(args.variableId);
    
    const skill = $dataSkills[skillId];
    if (skill) {
        const description = RYBA_CreateSkillDescriptionForCommand(skill);
        $gameVariables.setValue(variableId, description);
    } else {
        console.error(`スキルID ${skillId} が見つかりません。`);
    }
});

function RYBA_SkillCheck(){
    this.initialize(...arguments);
};

RYBA_SkillCheck.prototype.initialize = function(object) {
    this._item = new Game_Item();
    this._item.setObject(object);
    this._effectDic = [];
    
    for (const effect of this.item().effects) {
        this.createItemEffect(this._effectDic, effect);
    }

    this._gainHpValue = this.effectHpCal();
    this._gainMpValue = this.effectMpCal();
    this._gainTpValue = this.effectTpCal();

    this.effectStatesCal();

    this.effectParamCal();
    
};

RYBA_SkillCheck.prototype.effectDir = function(){
    return this._effectDic;
};

RYBA_SkillCheck.prototype.isGainHp = function() {
    return this._gainHpValue > 0;
};

RYBA_SkillCheck.prototype.isGainMp = function() {
    return this._gainMpValue > 0;
};

RYBA_SkillCheck.prototype.isGainTp = function() {
    return this._gainTpValue > 0;
};

RYBA_SkillCheck.prototype.isLossHp = function() {
    return this._gainHpValue < 0;
};

RYBA_SkillCheck.prototype.isLossMp = function() {
    return this._gainMpValue < 0;
};

RYBA_SkillCheck.prototype.isLossTp = function() {
    return this._gainTpValue < 0;
};

RYBA_SkillCheck.prototype.isHpText = function() {
    if(this.isHpEffect()){
        return true;
    }
    if(!this.isRecover()){
       return false;
    }
    return this.isGainHp();
};

RYBA_SkillCheck.prototype.createItemEffect = function (array,effect){
    const code = effect.code;
    const len = array.length;
    for(let i = 0; i < len; ++i ){
        let data = array[i];
        if( !data ){
            continue;
        }
        if( data.code === code ){
            data.effects.push(effect);
            return;
        }
    }
    array.push( {code:code,effects:[effect]} );
};

RYBA_SkillCheck.prototype.effectFindIndex = function (code){
    const len = this._effectDic.length;
    for(let i = 0; i < len; ++i ){
        let data = this._effectDic[i];
        if( !data ){
            continue;
        }
        if( data.code === code ){
            return i;
        }
    }
    return -1;
};

RYBA_SkillCheck.prototype.effectRemoveCode = function (code){
    const index = this.effectFindIndex(code);
    if(index < 0){
        return null;
    }
    const data = this._effectDic[index];
    this._effectDic.splice(index,1);
    return data;
};
RYBA_SkillCheck.prototype.baseEffectPointCal = function (code){
    let result = 0;
    var data = this.effectRemoveCode(code);
    if(!data){
        return result;
    }
    const len = data.effects.length;
    for(let i = 0; i < len; ++i ){
        let effect = data.effects[i];
        if( !effect ){
            continue;
        }
        if(effect.value1 ){
            result += effect.value1;
        }
        if(effect.value2){
            result += effect.value2;
        }
    }
    return result;
};


RYBA_SkillCheck.prototype.effectHpCal = function (){
    return this.baseEffectPointCal(Game_Action.EFFECT_RECOVER_HP);
};

RYBA_SkillCheck.prototype.effectMpCal = function (){
    return this.baseEffectPointCal(Game_Action.EFFECT_RECOVER_MP);
};

RYBA_SkillCheck.prototype.effectTpCal = function (){
    return this.baseEffectPointCal(Game_Action.EFFECT_GAIN_TP);
};

RYBA_SkillCheck.prototype.baseEffectStatesCal = function (array,code){
    this.baseEffectParamCal(array,code);
    // let result = 0;
    // var data = this.effectRemoveCode(code);
    // if(!data){
    //     return;
    // }
    // const len = data.effects.length;
    // for(let i = 0; i < len; ++i ){
    //     let effect = data.effects[i];
    //     if( !effect ){
    //         continue;
    //     }
    //     array.push($dataStates[effect.dataId]);
    // }
};

RYBA_SkillCheck.prototype.effectStatesCal = function() {
    this._addStates = [];
    this._removeStates = [];
    this.baseEffectStatesCal(this._addStates,Game_Action.EFFECT_ADD_STATE);
    this.baseEffectStatesCal(this._removeStates,Game_Action.EFFECT_REMOVE_STATE);
};

RYBA_SkillCheck.prototype.addStatesArray = function(){
    return this._addStates;
};

RYBA_SkillCheck.prototype.removeStatesArray = function(){
    return this._removeStates;
};

RYBA_SkillCheck.prototype.baseEffectParamCal = function (array,code){
    let result = 0;
    var data = this.effectRemoveCode(code);
    if(!data){
        return;
    }
    const len = data.effects.length;
    for(let i = 0; i < len; ++i ){
        let effect = data.effects[i];
        if( !effect ){
            continue;
        }
        let find = false;
        let aryLen = array.length;
        for(let j = 0; j < aryLen; ++j){
            let data = array[j];
            if(data.dataId === effect.dataId){
                find = true;
                data.value += 1;
                break;
            }
        }
        if(!find){
            array.push({dataId:effect.dataId, value:1});
        }
    }
};

RYBA_SkillCheck.prototype.effectParamCal = function() {
    this._addBuff = [];
    this._addDebuff = [];
    this._removeBuff = [];
    this._removeDebuff = [];
    this.baseEffectParamCal(this._addBuff,Game_Action.EFFECT_ADD_BUFF);
    this.baseEffectParamCal(this._addDebuff,Game_Action.EFFECT_ADD_DEBUFF);
    this.baseEffectParamCal(this._removeBuff,Game_Action.EFFECT_REMOVE_BUFF);
    this.baseEffectParamCal(this._removeDebuff,Game_Action.EFFECT_REMOVE_DEBUFF);
};

RYBA_SkillCheck.prototype.addBuffArray = function(){
    return this._addBuff;
};

RYBA_SkillCheck.prototype.addDebuffArray = function(){
    return this._addDebuff;
};

RYBA_SkillCheck.prototype.removeBuffArray = function(){
    return this._removeBuff;
};

RYBA_SkillCheck.prototype.removeDebuffArray = function(){
    return this._removeDebuff;
};


RYBA_SkillCheck.prototype.item = function(){
    return this._item.object();
};

RYBA_SkillCheck.prototype.checkItemScope = function(list) {
    return list.includes(this.item().scope);
};

RYBA_SkillCheck.prototype.isForOpponent = function() {
    return this.checkItemScope([1, 2, 3, 4, 5, 6, 14]);
};

RYBA_SkillCheck.prototype.isForFriend = function() {
    return this.checkItemScope([7, 8, 9, 10, 12, 13, 14]);
};

RYBA_SkillCheck.prototype.isForUser = function() {
    return this.checkItemScope([11]);
};

RYBA_SkillCheck.prototype.isForAliveFriend = function() {
    return this.checkItemScope([7, 8, 11, 14]);
};

RYBA_SkillCheck.prototype.isForDeadFriend = function() {
    return this.checkItemScope([9, 10]);
};

RYBA_SkillCheck.prototype.isForOne = function() {
    return this.checkItemScope([1, 7, 9, 11, 12]);
};

RYBA_SkillCheck.prototype.isForRandom = function() {
    return this.checkItemScope([3, 4, 5, 6]);
};

RYBA_SkillCheck.prototype.isForAll = function() {
    return this.checkItemScope([2, 8, 10, 13, 14]);
};

RYBA_SkillCheck.prototype.numTargets = function() {
    return this.isForRandom() ? this.item().scope - 2 : 0;
};

RYBA_SkillCheck.prototype.numRepeatsTotal = function() {
    let baseCount = this.numTargets() > 0 ? this.numTargets() : 1;
    let repeats = this.item().repeats;
    return repeats * baseCount;
};

RYBA_SkillCheck.prototype.checkDamageType = function(list) {
    return list.includes(this.item().damage.type);
};

RYBA_SkillCheck.prototype.isHpEffect = function() {
    return this.checkDamageType([1, 3, 5]);
};

RYBA_SkillCheck.prototype.isMpEffect = function() {
    return this.checkDamageType([2, 4, 6]);
};

RYBA_SkillCheck.prototype.isDamage = function() {
    return this.checkDamageType([1, 2]);
};

RYBA_SkillCheck.prototype.isRecover = function() {
    return this.checkDamageType([3, 4]);
};

RYBA_SkillCheck.prototype.isDrain = function() {
    return this.checkDamageType([5, 6]);
};

RYBA_SkillCheck.prototype.isHpRecover = function() {
    return this.checkDamageType([3]);
};

RYBA_SkillCheck.prototype.isMpRecover = function() {
    return this.checkDamageType([4]);
};

RYBA_SkillCheck.prototype.isCertainHit = function() {
    return this.item().hitType === Game_Action.HITTYPE_CERTAIN;
};

RYBA_SkillCheck.prototype.isPhysical = function() {
    return this.item().hitType === Game_Action.HITTYPE_PHYSICAL;
};

RYBA_SkillCheck.prototype.isMagical = function() {
    return this.item().hitType === Game_Action.HITTYPE_MAGICAL;
};

RYBA_SkillCheck.prototype.isMagicSkill = function() {
    if (this.isSkill()) {
        return $dataSystem.magicSkills.includes(this.item().stypeId);
    } else {
        return false;
    }
};

//---------------------------------------------------------------------------


function RYBA_SkillEffectText(){
    this.initialize(...arguments);
};

RYBA_SkillEffectText.prototype.initialize = function(act, isNoFomat) {
    const isRecover = act.isRecover();
    this._isNoFomat = isNoFomat;
    this._effectText = '';
    this._stateEffectText = ''; // ステート効果用の新しいプロパティ

    // ステート付与の処理を別途保存
    this._stateEffectText += this.addStatesText(act);
    this._stateEffectText += this.removeStatesText(act);

    // その他の効果を処理
    this._effectText += this.effectLossText(act);
    if(!isRecover){
        this._effectText += this.effectGainText(act);
    }
    this._effectText += this.addBuffText(act) + this.addDebuffText(act) + this.removeBuffText(act) + this.removeDebuffText(act);
};

RYBA_SkillEffectText.prototype.getEffectText = function(){
    return this._effectText;
};

RYBA_SkillEffectText.prototype.getStateEffectText = function(){
    return this._stateEffectText;
};

RYBA_SkillEffectText.prototype.baseEffectPointText = function(isHp,isMp,isTp){
    let flag = false;
    let result = '';
    if(isHp){
        flag = true;
        result += 'HP';
    }
    if(isMp){
        if(flag){
            result += RYBA_AndText;
        }else{
            flag = true;
        }
        result += 'MP';
    }
    if(isTp){
        if(flag){
            result += RYBA_AndText;
        }else{
            flag = true;
        }
        result += 'TP';
    }
    if(result !== ''){
        if(this._isNoFomat){
            result = 'の' + result;
            this._isNoFomat = false;
        }else{
            result = '+' + result;
        }
    }
    return result;
};

RYBA_SkillEffectText.prototype.effectLossText = function(act){
    var beforeNoFomat = this._isNoFomat;
    let result = this.baseEffectPointText(act.isLossHp(),act.isLossMp(),act.isLossTp());
    if(result === ''){
        return '';
    }
    result += '減';
    if(beforeNoFomat){
        result += '少';
    }
    return result;
};

RYBA_SkillEffectText.prototype.effectGainText = function(act){
    var beforeNoFomat = this._isNoFomat;
    let result = this.baseEffectPointText(act.isGainHp(),act.isGainMp(),act.isGainTp());
    if(result === ''){
        return '';
    }
    result += '増';
    if(beforeNoFomat){
        result += '加';
    }
    return result;
};
RYBA_SkillEffectText.prototype.baseEffectStatesText = function(array){
    let flag = false;
    let result = '';

    const len = array.length;
    let conCount = 0;
    for(let i = 0; i < len; ++i ){
        const data = array[i];
        if (data.dataId === 0) {
            continue;
        }
        if( conCount > 0 ){
            result += RYBA_AndText;
        }
        result += $dataStates[data.dataId].name;
        ++conCount;
    }

    if(result !== ''){
        if(this._isNoFomat){
            result = '+' + result;
            this._isNoFomat = false;
        }else{
            result = '+' + result;
        }
    }
    return result;
};

RYBA_SkillEffectText.prototype.addStatesText = function(act){
    var beforeNoFomat = this._isNoFomat;
    let result = this.baseEffectStatesText(act.addStatesArray());
    if(result === ''){
        return '';
    }
    if(beforeNoFomat){
        result += '';
    }
    return result;
};

RYBA_SkillEffectText.prototype.removeStatesText = function(act){
    let result = this.baseEffectStatesText(act.removeStatesArray().filter(state => state.dataId !== 1));  // 戦闘不能（ID: 1）を除外
    if(result === ''){
        return '';
    }
    result += '治療';
    return result;
};

RYBA_SkillEffectText.prototype.baseEffectParamText = function(array){
    let result = '';
    const params = array.map(data => data.dataId);

    // 攻撃力(2)、魔法力(4)、防御力(3)、魔法防御力(5)の全てが含まれているかチェック
    const allMainStats = [2, 3, 4, 5].every(id => params.includes(id));

    if (params.length === 8 || (params.length >= 4 && allMainStats)) {
        result = '全能力';
    } else if (params.includes(2) && params.includes(4) && params.length === 2) {
        result = '全攻撃';
    } else if (params.includes(3) && params.includes(5) && params.length === 2) {
        result = '全防御';
    } else {
        const len = array.length;
        for(let i = 0; i < len; ++i ){
            const data = array[i];
            result += TextManager.param(data.dataId);
        }
    }

    if(result !== ''){
        if(this._isNoFomat){
            result = 'の' + result;
            this._isNoFomat = false;
        }else{
            result = '+' + result;
        }
    }
    return result;
};

RYBA_SkillEffectText.prototype.addBuffText = function(act){
    let result = this.baseEffectParamText(act.addBuffArray());
    if(result === ''){
        return '';
    }
    result += '20%上昇（上限40%）';
    return result;
};
RYBA_SkillEffectText.prototype.addDebuffText = function(act){
    let result = this.baseEffectParamText(act.addDebuffArray());
    if(result === ''){
        return '';
    }
    result += '20%低下（下限40%）';
    return result;
};

RYBA_SkillEffectText.prototype.removeBuffText = function(act){
    let result = this.baseEffectParamText(act.removeBuffArray());
    if(result === ''){
        return '';
    }
    result += '強化解除';
    return result;
};

RYBA_SkillEffectText.prototype.removeDebuffText = function(act){
    let result = this.baseEffectParamText(act.removeDebuffArray());
    if(result === ''){
        return '';
    }
    result += '弱体解除';
    return result;
};

//---------------------------------------------------------------------------

const RYBA_AndText = '･';
function RYBA_GetItemDescription(skill){
    if(!skill){
        return '';
    }
    return skill.autoItemDescriptionText ? skill.autoItemDescriptionText : skill.description;
};
function RYBA_GetItemName(item){
    if(!item){
        return '';
    }
    return item.autoItemName ? item.autoItemName : item.name;
};
Window_Help.prototype.setItem = function(item) {
    this.setText(RYBA_GetItemDescription(item));
};

function RYBA_GetItemIconIndex(item){
    if(!item){
        return '';
    }
    return item.autoItemIconIndex ? item.autoItemIconIndex : item.iconIndex;
}
 //説明文を作る関数
function RYBA_CreateSkillDescription(skill){
    if(!skill){
        return '';
    }
    
    const action = new RYBA_SkillCheck(skill);
    let result = '';
    //ランダム
    if(action.isForRandom()){
        result += 'ランダムな';
    }
   
    //まず陣営
    if(action.isForUser()){
        result += '使用者';
    }else{
        if(action.isForOpponent()){
            result += '敵';
        }else if(action.isForFriend()){
            result += '味方';
        } 
        //数
        if(action.isForOne()){
            result += '単体';
        }else if(action.isForAll()){
            result += '全体';
        }    
    }
    
    //負傷者のみなので復活スキルに違いない
    const deadOnly = (action.isForDeadFriend() && !action.isForAliveFriend());
    if(deadOnly){
        result += 'を'; // 常に "を" を追加
        return result + '蘇生し、蘇生した';
    }

    const isDamage = action.isDamage();
    const isRecover = action.isRecover();
    const isDrain = action.isDrain();
    
    // HP, MP, TP回復時は "の" を使用
    if(isRecover || isDrain){
        result += 'の';
    } else {
        result += 'に';
    }

    let isPointText = false;
    
    if(isDamage){
        // "に" は既に追加されているので何もしない
    }else if(isRecover || isDrain){
        if(action.isHpText()){
            isPointText = true;
            result += 'HPを';
        }
    }
    
    if(action.isMpEffect() || (isRecover && action.isGainMp())){
        if(isPointText){
            result += RYBA_AndText;
        }
        result += 'MPを';
        isPointText = true;
    }

    if(isRecover && action.isGainTp()){
        result += RYBA_AndText + 'TPを';
    }

    if(result === ''){
        return '？？？';
    }

    return result;
};

function RYBA_getHitTypeText(skill){
    if(skill.occasion === 3){
        return '[パッシブ]';
    }else if(skill.hitType === Game_Action.HITTYPE_PHYSICAL){
        return '[物理]';
    }else if(skill.hitType === Game_Action.HITTYPE_MAGICAL || $dataSystem.magicSkills.includes(skill.stypeId)){
        return '[魔法]';
    }else if(skill.hitType === Game_Action.HITTYPE_CERTAIN){
        return '[特殊]';
    }

    return '[なし]';
}

function RYBA_getSkillTypeText(skill){
    let result = '';
    result += RYBA_getHitTypeText(skill);
    if(skill.tpCost === 100){
        result += ' <必殺技>'
    }
    return result;
}

function Nakasa_skillCostText (skill) {
    let result = '';
    if (skill.tpCost > 0) {
        result = 'TP:'+skill.tpCost;
    } else if (skill.mpCost > 0) {
        result = 'MP:'+skill.mpCost;
    }else if (DataManager.getVirtualCost(skill.virtualCostArray,0) > 0) {
        result = 'Ｇ:'+DataManager.getVirtualCost(skill.virtualCostArray,0);
    }else if(DataManager.getVirtualCostMost(skill.virtualCostArray) > 0){
        result = 'VA:'+DataManager.getVirtualCostMost(skill.virtualCostArray);
    }
    return Nakasa_StringPaddingRight(''+result,' ',7);
};

function SkillUnlockRateText(skill) {
    // メタデータから SkillUnlockRate を取得
    const skillunlockrateTag = String(skill.meta.SkillUnlockRate || '');
    
    // <SkillUnlockRate:xx,yy> 形式の xx のみを抽出する正規表現
    const match = skillunlockrateTag.match(/(\d+),\s*\d+/);

    // xx を取得し、アンロック率として返す
    if (match) {
        const rate = match[1];
        return '<\\c[2]選択できる確率\\c[0]:' + rate + '%>';
    } else {
        return '';
    }
}

function WarmUpText (skill) {
    let warmupTag = String(skill.meta.Warmup);
    warmupTag = warmupTag.replace(/ /g, '');
    if ((warmupTag) && (0 < warmupTag)) {
        return '<\\c[23]使用開始\\c[0]:' + warmupTag + 'ターン後>';
    } else {
        return '';
    }
};

function CoolDownText (skill) {
    let cooldownTag = String(skill.meta.Cooldown);
    cooldownTag = cooldownTag.replace(/ /g, '');
    if ((cooldownTag) && (0 < cooldownTag)) {
        return '<\\c[23]再使用可能\\c[0]:' + cooldownTag + 'ターン後>';
    } else {
        return '';
    }
};

function TargetPercentText (skill) {
    let setTargetPercentTag = String(skill.meta.TargetPercent);
    setTargetPercentTag = setTargetPercentTag.replace(/ /g, '');
    if ((setTargetPercentTag) && (0 <= setTargetPercentTag)) {
        return '<\\c[10]狙われる確率\\c[0]:' + setTargetPercentTag + '%>';
    } else if ((setTargetPercentTag) && (setTargetPercentTag === "-1")) {
        return '<\\c[10]自分への敵視解除\\c[0]>';
    } else if ((setTargetPercentTag) && (setTargetPercentTag === "-2")) {
        return '<\\c[10]全ての敵視解除\\c[0]>';
    } else {
        return '';
    }
};

function Nakasa_skillFormatText (skill) {
    const sign = [0].includes(skill.damage.type) ? false : true;
    if(!sign){
        return '';
    }

    const autoDescriptionOff = skill.meta.AutoDescriptionOff;
    const action = new RYBA_SkillCheck(skill);
    const isDamage = action.isDamage();
    const isRecover = action.isRecover();

    let result_word = 'ダメージ';
    if (isRecover) {
        result_word = '回復量';
    } else if (isDamage) {
        
        if ((skill.damage.formula.includes("a.mat")) && (!skill.damage.formula.includes("b.mdf * 2"))) {
            result_word = '防御無視の魔法ダメージ';
        } else if (skill.damage.formula.includes("a.mat")) {
            result_word = '魔法ダメージ';
        // ダメージ計算式に "b.def * 2" が含まれていないかチェック
        } else if (!skill.damage.formula.includes("b.def * 2")) {
            result_word = '防御無視ダメージ';
        }
    }

    let text_a = "";
    let text_b = "相手の";
    let result = String(skill.damage.formula);
    let result_add = '';
    result = result.replace(/ /g, '');
    result = result.replace(/a\./g, text_a);
    result = result.replace(/b\./g, text_b);
    result = result.replace(/mhp/g, TextManager.param(0));
    result = result.replace(/mmp/g, TextManager.param(1));
    result = result.replace(/atk/g, TextManager.param(2));
    result = result.replace(/def/g, TextManager.param(3));
    result = result.replace(/mat/g, TextManager.param(4));
    result = result.replace(/mdf/g, TextManager.param(5));
    result = result.replace(/spd/g, TextManager.param(6));
    result = result.replace(/luk/g, TextManager.param(7));

    result = result.replace("-" + text_b + TextManager.param(3) + "*2", '');
    result = result.replace("-" + text_b + TextManager.param(5) + "*2", '');

    result = result.replace(/Math.min/g, '少ない数値');
    result = result.replace(/Math.min/g, '多い数値');
    result = result.replace(/\*/g, '×');
    result = result.replace(/\//g, '÷');
    result = result.replace(/,/g, 'か');
    result = result.replace(/hp/g, '残HP');
    result = result.replace(/mp/g, '残MP');
    result = result.replace(/tp/g, '残TP');

    if (skill && skill.note) {
        const noteTagMatch = /<DescriptionResult:(.+)>/.exec(skill.note);
        if (noteTagMatch) {
            result = noteTagMatch[1];
        }
    }

    result = '<' + result_word + ':' + result + result_add + '>';

    return result;
};


// 既存のコードはそのまま保持
(() => {
    const Scene_Boot_start = Scene_Boot.prototype.start;
    Scene_Boot.prototype.start = function() {
      Scene_Boot_start.call(this);
      DataManager.processAutoSkillDescription();
    };
    
    DataManager.processAutoSkillDescription = function() {
      for (let i = 1; i < $dataSkills.length; i++) {
        const skill = $dataSkills[i];
        skill.autoItemDescriptionText = RYBA_CreateSkillDescriptionForCommand(skill);
      }
    };
})();
