/*=============================================================================
 EscapeCharacterSe.js
----------------------------------------------------------------------------
 (C)2020 Triacontane
 This software is released under the MIT License.
 http://opensource.org/licenses/mit-license.php
----------------------------------------------------------------------------
 Version
 1.2.1 2024/XX/XX メッセージウィンドウでのみ効果音が鳴るよう修正
 1.2.0 2023/10/14 前回の効果音を停止してから新しい効果音を再生するよう修正
 1.1.0 2023/10/07 制御文字で効果音演奏するときに音量、ピッチ、位相を指定できるよう修正
 1.0.0 2020/09/20 初版
----------------------------------------------------------------------------
 [Blog]   : https://triacontane.blogspot.jp/
 [Twitter]: https://twitter.com/triacontane/
 [GitHub] : https://github.com/triacontane/
=============================================================================*/

/*:
 * @plugindesc EscapeCharacterSePlugin
 * @target MZ
 * @url https://github.com/triacontane/RPGMakerMV/tree/mz_master/EscapeCharacterSe.js
 * @base MaterialBase
 * @author triacontane
 *
 * @param volume
 * @desc volume
 * @default 90
 * @type number
 * @max 100
 *
 * @param pitch
 * @desc pitch
 * @default 100
 * @type number
 * @min 50
 * @max 150
 *
 * @param pan
 * @desc pan
 * @default 0
 * @type number
 * @min -100
 * @max 100
 *
 * @help EscapeCharacterSe.js
 *
 * Play a sound effect when the control characters are analyzed.
 * This is useful for messages and battle log displays.
 *
 * \se[id] // Play the sound effect of the identifier id.
 *
 * This plugin is released under the MIT License.
 */
/*:ja
 * @plugindesc 制御文字の効果音演奏プラグイン
 * @target MZ
 * @url https://github.com/triacontane/RPGMakerMV/tree/mz_master/EscapeCharacterSe.js
 * @base MaterialBase
 * @author トリアコンタン
 *
 * @param volume
 * @text 音量
 * @desc 効果音の音量(制御文字\v[n]を使う場合はテキストタブから入力)
 * @default 90
 * @type number
 * @max 100
 *
 * @param pitch
 * @text ピッチ
 * @desc 効果音のピッチ(制御文字\v[n]を使う場合はテキストタブから入力)
 * @default 100
 * @type number
 * @min 50
 * @max 150
 *
 * @param pan
 * @text 位相
 * @desc 効果音の位相(制御文字\v[n]を使う場合はテキストタブから入力)
 * @default 0
 * @type number
 * @min -100
 * @max 100
 *
 * @param muteSwitch
 * @text 効果音を無効にするスイッチID
 * @desc スイッチがONのとき効果音を再生しません
 * @default 1
 * @type switch
 *
 * @help EscapeCharacterSe.js
 *
 * 制御文字を解析したタイミングで効果音を演奏します。
 * メッセージウィンドウでのみ効果音が再生されます。
 * 公式プラグイン『MaterialBase.js』で素材登録しておいて
 * 制御文字から素材の識別子を指定します。
 * 『MaterialBase.js』は、RPGツクールMZのインストールフォルダ配下の
 * 以下のフォルダに格納されています。
 * dlc/BasicResources/plugins/official
 *
 * \se[id] // 識別子 id の効果音を演奏します。
 * \se[id,40,100,0] // 識別子 id の効果音を音量40、ピッチ100、位相0で演奏します。
 *
 * 利用規約：
 *  作者に無断で改変、再配布が可能で、利用形態（商用、18禁利用等）
 *  についても制限はありません。
 *  このプラグインはもうあなたのものです。
 */

(() => {
    'use strict';
    const script = document.currentScript;
    const param = PluginManagerEx.createParameter(script);

    let currentSe = null;

    // Window_Baseで制御文字を認識してテキストから削除
    const _Window_Base_processEscapeCharacter = Window_Base.prototype.processEscapeCharacter;
    Window_Base.prototype.processEscapeCharacter = function(code, textState) {
        _Window_Base_processEscapeCharacter.apply(this, arguments);
        if (code === 'SE') {
            const seId = this.obtainEscapeParamString(textState);
            // Window_Messageの場合のみ音を再生
            if (seId && this instanceof Window_Message) {
                this.playEscapeAudio(seId);
            }
        }
    };

    Window_Base.prototype.obtainEscapeParamString = function(textState) {
        const arr = /^\[.+?]/.exec(textState.text.slice(textState.index));
        if (arr) {
            textState.index += arr[0].length;
            return arr[0].substring(1, arr[0].length - 1);
        } else {
            return '';
        }
    };

    Window_Base.prototype.playEscapeAudio = function(seId) {
        // プラグインパラメータで指定したスイッチIDを取得
        const switchId = param.muteSwitch || 0; // 'muteSwitch' はプラグインパラメータ名
        if ($gameSwitches.value(switchId)) {
            return; // スイッチがONの場合、効果音を再生しない
        }

        AudioManager.playMaterialAudio(seId);
    };

    // バトルログでの効果音再生部分は削除
    // （Window_BattleLogの拡張部分を削除）

    AudioManager.playMaterialAudio = function(seId) {
        // 前回の効果音が再生中なら停止する
        if (currentSe) {
            this.stopSe(currentSe);
        }

        const params = seId.split(',');
        let fileName = $gameSystem.getMaterialAudio(params[0]);
        if (!fileName) {
            fileName = seId;
        }
        const audio = {
            name: fileName,
            volume: parseInt(params[1]) || param.volume || 90,
            pitch: parseInt(params[2]) || param.pitch || 100,
            pan: parseInt(params[3]) || param.pan || 0
        }
        this.playSe(audio);

        // 現在の効果音を保存
        currentSe = audio;
    };
})();