/*:
 * @target MZ
 * @plugindesc Gain gold equal to the damage dealt times the gain percentage stored in a variable when attacking specific enemies and display gold gained as popup at a fixed position, and store gold in multiple variables.
 * @author スタジオVR
 * 
 * @param offsetX
 * @text Offset X
 * @desc The X offset for the gold popup. (Default: 0)
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 *
 * @param offsetY
 * @text Offset Y
 * @desc The Y offset for the gold popup. (Default: 0)
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 *
 * @param goldVariables
 * @text Gold Variables
 * @desc The variable IDs to store the accumulated gold.
 * @type variable[]
 * @default []
 *
 * @param currentDamageVariable
 * @text Current Damage Variable
 * @desc The variable ID to store the current damage dealt to enemies.
 * @type variable
 * @default 503
 * 
 * @param maxDamageVariable
 * @text Max Damage Variable
 * @desc The variable ID to store the max damage dealt in a single battle.
 * @type variable
 * @default 504
 *
 * @param initialBattlePowerVariable
 * @text Initial Battle Power Variable
 * @desc The variable ID to store the initial battle power.
 * @type variable
 * @default 505
 * 
 *
 * @param goldGainingEnemies
 * @text Gold Gaining Enemies
 * @desc A list of enemy IDs that grant gold when attacked. Leave empty to include all enemies.
 * @type number[]
 * @default []
 *
 * @param gainPercentageVariable
 * @text Gain Percentage Variable
 * @desc The variable ID that stores the gain percentage.
 * @type variable
 * @default 
 *
 * @param percentageIncrease
 * @text Percentage Increase
 * @desc The value added to the gain percentage variable after dealing damage to an enemy.
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @param doubleGoldSwitch
 * @text Double Gold Switch
 * @desc The switch ID that, when turned ON, doubles the gained gold.
 * @type switch
 * @default 0
 * @help GoldGainFromDamage.js
 *
 * This plugin allows you to gain gold equal to the damage dealt times the gain percentage stored in a variable when attacking
 * specific enemies during battles and display the gained gold as popup at a
 * fixed position on the screen, and store the accumulated gold in multiple variables.
 *
 * ============================================================================
 * Changelog
 * ============================================================================
 * Version 1.0:
 * - Initial release
 * Version 1.1:
 * - Added popup display for gained gold at a fixed position on the screen
 * - Added storing accumulated gold in multiple variables
 * Version 1.2:
 * - Added the ability to specify gold gaining enemies via plugin parameters
 * Version 1.3:
 * - Changed the gold gain calculation to use a gain percentage stored in a variable
 */
(() => {
  const pluginParams = PluginManager.parameters("GoldGainFromDamage");
  const doubleGoldSwitch = parseInt(pluginParams["doubleGoldSwitch"]) || 0;
  const goldVariables = JSON.parse(pluginParams["goldVariables"]).map(Number);
  const goldGainingEnemies = JSON.parse(pluginParams["goldGainingEnemies"]).map(Number);
  const gainPercentageVariable = parseInt(pluginParams["gainPercentageVariable"]) || 1;
  const percentageIncrease = parseInt(pluginParams["percentageIncrease"]) || 0;
  const currentDamageVariable = parseInt(pluginParams["currentDamageVariable"]) || 503;
  const maxDamageVariable = parseInt(pluginParams["maxDamageVariable"]) || 504;
  const initialBattlePowerVariable = parseInt(pluginParams["initialBattlePowerVariable"]) || 505;

  const _Game_Action_executeHpDamage = Game_Action.prototype.executeHpDamage;
  Game_Action.prototype.executeHpDamage = function (target, value) {
    _Game_Action_executeHpDamage.call(this, target, value);
    if (this.subject().isActor() && target.isEnemy() && this.isGoldGainingEnemy(target)) {
      const gainPercentage = $gameVariables.value(gainPercentageVariable) / 100;
      const goldMultiplier = $gameSwitches.value(doubleGoldSwitch) ? 2 : 1;
      const goldGain = Math.floor(value * gainPercentage * goldMultiplier);
      goldVariables.forEach(variableId => {
        $gameVariables.setValue(variableId, $gameVariables.value(variableId) + goldGain);
      });

      $gameVariables.setValue(gainPercentageVariable, $gameVariables.value(gainPercentageVariable) + percentageIncrease);
      
      // 累計の与ダメージ（変数503）を更新
      $gameVariables.setValue(currentDamageVariable, $gameVariables.value(currentDamageVariable) + value);

      // 1回の戦闘の合計与ダメージ（変数504）を更新
      const currentBattleDamage = $gameVariables.value(maxDamageVariable) + value;
      $gameVariables.setValue(maxDamageVariable, currentBattleDamage);

      // 初期戦闘力（変数505）を更新
      if ($gameVariables.value(maxDamageVariable) >= $gameVariables.value(initialBattlePowerVariable)) {
        $gameVariables.setValue(initialBattlePowerVariable, $gameVariables.value(initialBattlePowerVariable) + value);
        // 変数504と変数505が同じ数値になるように更新
        $gameVariables.setValue(maxDamageVariable, $gameVariables.value(initialBattlePowerVariable));
      }
    }
  };

  Game_Action.prototype.isGoldGainingEnemy = function (target) {
    if (goldGainingEnemies.length === 0) {
      return true; // goldGainingEnemies が空の場合、すべての敵がゴールド獲得敵とみなされます。
    } else {
      return goldGainingEnemies.includes(target.enemyId()); // goldGainingEnemies に含まれる敵IDのみがゴールド獲得敵とみなされます。
    }
  };

})();

