/*:
 * @target MZ
 * @plugindesc HalfHP_Switch v1.14.0: Activates specified switches when a certain number of actors' HP falls below a specified percentage in battle.
 * @author スタジオVR
 *
 * @param settingsList
 * @text 設定リスト
 * @desc 異なるスイッチの有効化のための設定リスト。
 * @type struct<SwitchSettings>[]
 * @default []
 *
 * @help HalfHP_Switch.js
 *
 * このプラグインは、バトル中に指定した数のアクターのHPが一定の割合以下になったときに、特定のスイッチを有効化します。
 * スイッチID、最小アクター数、およびupdateOnCollapse設定を、各リスト項目のプラグインパラメータで設定できます。
 */

/*~struct~SwitchSettings:
 * @param switchId
 * @text スイッチID
 * @desc 指定した数のアクターのHPが一定の割合以下になったときに有効化するスイッチのID。
 * @type number
 * @min 1
 * @default 1
 *
 * @param minActors
 * @text 最小アクター数
 * @desc スイッチを有効化するために必要な、HPが一定の割合以下のアクターの最小数。
 * @type number
 * @min 1
 * @default 2
 *
 * @param hpThreshold
 * @text HPしきい値
 * @desc スイッチを有効化するためのHPのしきい値をパーセンテージで指定します（0～100）。
 * @type number
 * @min 1
 * @max 100
 * @default 50
 *
 * @param updateOnCollapse
 * @text Update on Collapse
 * @desc trueの場合、死亡したアクターをカウントに含めます。falseの場合、死亡したアクターをカウントから除外します。
 * @type boolean
 * @default true
 */

(() => {
  const pluginName = "HalfHP_Switch";
  const parameters = PluginManager.parameters(pluginName);
  const settingsList = JSON.parse(parameters["settingsList"]).map(setting => {
    const parsedSetting = JSON.parse(setting);
    parsedSetting.updateOnCollapse = parsedSetting.updateOnCollapse === "true";
    return parsedSetting;
  });

  const _Game_Actor_executeDamage = Game_Actor.prototype.executeDamage;
  Game_Actor.prototype.executeDamage = function(value) {
    _Game_Actor_executeDamage.call(this, value);
    this.updateSwitches();
  };

  Game_Actor.prototype.updateSwitches = function() {
    for (const setting of settingsList) {
      let count = 0;
      const party = $gameParty.battleMembers();
      for (const actor of party) {
        if (actor._hp / actor.mhp <= setting.hpThreshold / 100 && (setting.updateOnCollapse || !actor.isDead())) {
          count++;
        }
      }
      $gameSwitches.setValue(setting.switchId, count >= setting.minActors);
    }
  };

  const _Game_Actor_gainHp = Game_Actor.prototype.gainHp;
  Game_Actor.prototype.gainHp = function(value) {
    _Game_Actor_gainHp.call(this, value);
    this.updateSwitches();
  };

  const _Game_Actor_performCollapse = Game_Actor.prototype.performCollapse;
  Game_Actor.prototype.performCollapse = function() {
    _Game_Actor_performCollapse.call(this);
    this.updateSwitches();
  };
})();