/*:
* @plugindesc Map Click Coordinate Plugin for RPG Maker MZ
* @author スタジオVR
*
* @param VariableX
* @text X Coordinate Variable
* @desc The ID of the variable to store the X coordinate.
* @type number
* @default 1
*
* @param VariableY
* @text Y Coordinate Variable
* @desc The ID of the variable to store the Y coordinate.
* @type number
* @default 2
*
* @param VariableRegion
* @text Region ID Variable
* @desc The ID of the variable to store the clicked region ID.
* @type number
* @default 3
*
* @help This plugin allows you to store the map coordinates and region ID of a left-clicked location in specified variables.
* Use plugin commands to enable/disable this feature and to set region to common event mapping.
*
* @command toggleClick
* @text Toggle Click Listener
* @desc Toggles the click listener on and off.
*
* @arg enabled
* @type boolean
* @default true
* @text Enabled
* @desc Enable or disable the click listener.
*
* @command setRegionCommonEvent
* @text Set Region Common Event
* @desc Set the common event triggered by a region ID.
*
* @arg regionId
* @type number
* @default 1
* @text Region ID
* @desc The region ID to trigger a common event.
*
* @arg commonEventId
* @type common_event
* @default 1
* @text Common Event ID
* @desc The common event ID to be triggered by the region ID.
*/

(() => {
    const parameters = PluginManager.parameters('MapClickCoordinate');
    const variableX = Number(parameters['VariableX']) || 1;
    const variableY = Number(parameters['VariableY']) || 2;
    const variableRegion = Number(parameters['VariableRegion']) || 3;

    let clickListenerEnabled = false;
    const regionCommonEvents = new Map();

    function triggerCommonEvent(regionId) {
        const commonEventId = regionCommonEvents.get(regionId);
        if (commonEventId) {
            $gameTemp.reserveCommonEvent(commonEventId);
        }
    }

    function updateCoordinates() {
        if (TouchInput.isPressed() && clickListenerEnabled) { // left click
            const x = $gameMap.canvasToMapX(TouchInput._x);
            const y = $gameMap.canvasToMapY(TouchInput._y);
            const regionId = $gameMap.regionId(Math.floor(x), Math.floor(y));
            $gameVariables.setValue(variableX, Math.floor(x));
            $gameVariables.setValue(variableY, Math.floor(y));
            $gameVariables.setValue(variableRegion, regionId);
            triggerCommonEvent(regionId);
        }
    }

    PluginManager.registerCommand('MapClickCoordinate', 'toggleClick', args => {
        clickListenerEnabled = args.enabled.toLowerCase() === 'true';
    });

    PluginManager.registerCommand('MapClickCoordinate', 'setRegionCommonEvent', args => {
        const regionId = Number(args.regionId);
        const commonEventId = Number(args.commonEventId);
        regionCommonEvents.set(regionId, commonEventId);
    });

    const alias_Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        alias_Scene_Map_update.call(this);
        updateCoordinates();
    };
})();