//=============================================================================
// MouseWheelExtend.js
// ----------------------------------------------------------------------------
// (C)2016 Triacontane
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.5.5 2024/10/31 右クリックでShiftを押す機能をパラメータで設定可能に変更
// 1.5.4 2024/10/31 Ctrlキー押下時は右クリックでShiftを押さないよう修正
// 1.5.3 2024/10/31 右クリックの状態管理を厳密化し、予期しない入力を防止
// 1.5.2 2024/10/31 Shiftキーが連打される問題を修正
// 1.5.1 2024/10/31 会話ログを右クリックで閉じた後も右クリックでShiftキーが効くように修正
// 1.5.0 2024/10/31 右クリックでShiftキー押下機能を追加（選択肢表示中は無効）
// 1.4.2 2024/10/31 Tabキーの押下時間を100ミリ秒に調整
// 1.4.1 2024/10/31 Tabキー押下と解放時のログ出力を追加
// 1.4.0 2024/10/31 マウスホイール奥回転でTabキー押下機能を追加
// 1.3.0 2023/07/05 MZで動作するよう修正
// 1.2.0 2017/05/18 ホイールクリックしたときにスイッチを切り替えられる機能を追加
// 1.1.0 2016/07/04 マウスホイールの状態をスイッチや変数に格納する機能など4種類の機能を追加
// 1.0.0 2016/07/03 初版
// ----------------------------------------------------------------------------
// [Blog]   : http://triacontane.blogspot.jp/
// [Twitter]: https://twitter.com/triacontane/
// [GitHub] : https://github.com/triacontane/
//=============================================================================

/*:
 * @plugindesc マウスホイール拡張プラグイン
 * @target MZ
 * @url https://github.com/triacontane/RPGMakerMV/tree/mz_master/MouseWheelExtend.js
 * @base PluginCommonBase
 * @orderAfter PluginCommonBase
 * @author トリアコンタン
 *
 * @param rightClickShift
 * @text 右クリックでShift
 * @desc 右クリックでShiftキーを押したことにします。選択肢表示中やCtrlキー押下中は無効です。
 * @default true
 * @type boolean
 *
 * @param scrollMessage
 * @text スクロールでメッセージ送り
 * @desc マウスホイールを手前に回転させてメッセージ送りをします。戻すことはできません。
 * @default true
 * @type boolean
 *
 * @param scrollTab
 * @text スクロールでTab押下
 * @desc マウスホイールを奥に回転させてTabキーを押したのと同じ動作をします。
 * @default true
 * @type boolean
 *
 * @param scrollCursor
 * @text スクロールでカーソル移動
 * @desc マウスホイールを動かしてウィンドウのカーソルを移動します。
 * @default true
 * @type boolean
 *
 * @param scrollDirection
 * @text スクロールで方向ボタン操作
 * @desc マウスホイールのスクロールに方向ボタンと同等の機能を持たせます。
 * @default false
 * @type boolean
 *
 * @param scrollVariable
 * @text スクロールで変数代入
 * @desc マウスホイールのスクロールで任意の変数に値を設定します。
 * 下:2 左:4 右:6 上:8　マップ画面でのみ有効です。
 * @default 0
 * @type variable
 *
 * @param click
 * @text クリック操作
 * @desc マウスホイールをクリックしたときの操作を選択します。
 * @default ok
 * @type select
 * @option なし
 * @value
 * @option 決定
 * @value ok
 * @option キャンセル
 * @value cancel
 * @option スイッチ(指定した番号のスイッチON)
 * @value switch
 * @option トグル(指定した番号のスイッチ切り替え)
 * @value toggle
 *
 * @param clickSwitch
 * @text クリックスイッチ番号
 * @desc クリック操作をスイッチ、トグルに指定した場合の対象スイッチ番号です。
 * @default 0
 * @type switch
 *
 * @param sensitivityY
 * @text 感度Y
 * @desc マウスホイールの縦回転の感度です。通常はこのままでOKです。
 * @default 4
 *
 * @param sensitivityX
 * @text 感度X
 * @desc マウスホイールの横回転の感度です。通常はこのままでOKです。
 * @default 4
 */

(()=> {
    'use strict';
    const script = document.currentScript;
    const param = PluginManagerEx.createParameter(script);
    const ButtonName = {
        OK: 'ok',
        CANCEL: 'escape',
        UP: 'up',
        DOWN: 'down',
        RIGHT: 'right',
        LEFT: 'left',
        TAB: 'tab',
        SHIFT: 'shift',
        CONTROL: 'control'
    };

    let isTabProcessing = false;
    let rightButtonPressed = false;
    let shiftState = false;
    const TAB_PRESS_DURATION = 100;

    //-----------------------------------------------------------------------------
    // Input拡張
    //-----------------------------------------------------------------------------
    const _Input_clear = Input.clear;
    Input.clear = function() {
        _Input_clear.apply(this, arguments);
        Input._currentState[ButtonName.SHIFT] = false;
        shiftState = false;
    };

    Input.isControlPressed = function() {
        return this._currentState[ButtonName.CONTROL];
    };

    //-----------------------------------------------------------------------------
    // TouchInput拡張
    //-----------------------------------------------------------------------------
    const _TouchInput_clear = TouchInput.clear;
    TouchInput.clear = function() {
        _TouchInput_clear.apply(this, arguments);
        this._rightButtonPressed = false;
        rightButtonPressed = false;
        Input._currentState[ButtonName.SHIFT] = false;
        shiftState = false;
    };

    const _TouchInput__onRightButtonDown = TouchInput._onRightButtonDown;
    TouchInput._onRightButtonDown = function(event) {
        _TouchInput__onRightButtonDown.apply(this, arguments);
        if (param.rightClickShift && !rightButtonPressed && !$gameMessage.isChoice() && !Input.isControlPressed()) {
            rightButtonPressed = true;
            shiftState = true;
            Input._currentState[ButtonName.SHIFT] = true;
        }
        event.preventDefault();
    };

    const _TouchInput__onRightButtonUp = TouchInput._onRightButtonUp;
    TouchInput._onRightButtonUp = function(event) {
        _TouchInput__onRightButtonUp.apply(this, arguments);
        rightButtonPressed = false;
        shiftState = false;
        Input._currentState[ButtonName.SHIFT] = false;
        event.preventDefault();
    };

    const _TouchInput_update = TouchInput.update;
    TouchInput.update = function() {
        _TouchInput_update.apply(this, arguments);
        this.updateRightButton();
    };

    TouchInput.updateRightButton = function() {
        if (!this._rightButtonPressed && rightButtonPressed) {
            rightButtonPressed = false;
            shiftState = false;
            Input._currentState[ButtonName.SHIFT] = false;
        }
    };

    //-----------------------------------------------------------------------------
    // Scene_Base拡張
    //-----------------------------------------------------------------------------
    const _Scene_Base_update = Scene_Base.prototype.update;
    Scene_Base.prototype.update = function() {
        _Scene_Base_update.apply(this, arguments);
        if (!TouchInput._rightButtonPressed && rightButtonPressed) {
            rightButtonPressed = false;
            shiftState = false;
            Input._currentState[ButtonName.SHIFT] = false;
        }
    };

    //-----------------------------------------------------------------------------
    // Scene_Map拡張
    //-----------------------------------------------------------------------------
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.apply(this, arguments);
        this.updateRightButton();
    };

    Scene_Map.prototype.updateRightButton = function() {
        if (rightButtonPressed && !TouchInput._rightButtonPressed) {
            rightButtonPressed = false;
            shiftState = false;
            Input._currentState[ButtonName.SHIFT] = false;
        }
    };

    //-----------------------------------------------------------------------------
    // Game_Map拡張
    //-----------------------------------------------------------------------------
    const _Game_Map_update = Game_Map.prototype.update;
    Game_Map.prototype.update = function(sceneActive) {
        _Game_Map_update.apply(this, arguments);
        this.updateWheelTrigger();
    };

    Game_Map.prototype.updateWheelTrigger = function() {
        if (TouchInput.isMiddleTriggered()) {
            if (param.click === 'switch') {
                $gameSwitches.setValue(param.clickSwitch, true);
            } else if (param.click === 'toggle') {
                $gameSwitches.setValue(param.clickSwitch, !$gameSwitches.value(param.clickSwitch));
            }
        }
        if (param.scrollVariable) {
            const prevValue = $gameVariables.value(param.scrollVariable);
            let value = 0;
            if (TouchInput.isValidWheelX(false)) value = 4;
            if (TouchInput.isValidWheelX(true)) value = 6;
            if (TouchInput.isValidWheelY(false)) value = 2;
            if (TouchInput.isValidWheelY(true)) value = 8;
            if (prevValue !== value) {
                $gameVariables.setValue(param.scrollVariable, value);
            }
        }
    };

    if (param.scrollMessage || param.scrollTab) {
        const _Window_Message_isTriggered = Window_Message.prototype.isTriggered;
        Window_Message.prototype.isTriggered = function() {
            if (param.scrollTab && TouchInput.isValidWheelY(true) && !isTabProcessing) {
                isTabProcessing = true;
//                console.log('Tab key pressed by mouse wheel');
                Input._currentState['tab'] = true;
                
                setTimeout(() => {
                    Input._currentState['tab'] = false;
//                    console.log('Tab key released by mouse wheel');
                    
                    setTimeout(() => {
                        isTabProcessing = false;
                    }, 50); // 次の入力を受け付けるまでの待機時間
                }, TAB_PRESS_DURATION);
                
                return false;
            }
            return _Window_Message_isTriggered.apply(this, arguments) || 
                   (param.scrollMessage && TouchInput.isValidWheelY(false));
        };
    }

    TouchInput.isValidWheelX = function(minus) {
        return minus ? this.wheelX <= -param.sensitivityX : this.wheelX >= param.sensitivityX;
    }

    TouchInput.isValidWheelY = function(minus) {
        return minus ? this.wheelY <= -param.sensitivityY : this.wheelY >= param.sensitivityY;
    }

    const _TouchInput__onMiddleButtonDown = TouchInput._onMiddleButtonDown;
    TouchInput._onMiddleButtonDown = function(event) {
        _TouchInput__onMiddleButtonDown.apply(this, arguments);
        if (param.click === 'ok') {
            Input._currentState[ButtonName.OK] = true;
        } else if (param.click === 'cancel') {
            Input._currentState[ButtonName.CANCEL] = true;
        }
        this._onMiddleTriggered = true;
    };

    const _TouchInput__onMouseUp = TouchInput._onMouseUp;
    TouchInput._onMouseUp = function(event) {
        _TouchInput__onMouseUp.apply(this, arguments);
        if (event.button === 1) {
            if (param.click === 'ok') {
                Input._currentState[ButtonName.OK] = false;
            } else if (param.click === 'cancel') {
                Input._currentState[ButtonName.CANCEL] = false;
            }
        }
    };

    TouchInput.isMiddleTriggered = function() {
        return this._middleTriggered;
    };

    if (param.scrollDirection) {
        TouchInput._wheelValidFrame = 12;
        
        const _TouchInput__onWheel = TouchInput._onWheel;
        TouchInput._onWheel = function(event) {
            _TouchInput__onWheel.apply(this, arguments);
            if (event.deltaY <= -param.sensitivityY) {
                this._wheelUp = TouchInput._wheelValidFrame;
                Input._currentState[ButtonName.UP] = true;
            }
            if (event.deltaY >= param.sensitivityY) {
                this._wheelDown = TouchInput._wheelValidFrame;
                Input._currentState[ButtonName.DOWN] = true;
            }
            if (event.deltaX <= -param.sensitivityX) {
                this._wheelRight = TouchInput._wheelValidFrame;
                Input._currentState[ButtonName.RIGHT] = true;
            }
            if (event.deltaX >= param.sensitivityX) {
                this._wheelLeft = TouchInput._wheelValidFrame;
                Input._currentState[ButtonName.LEFT] = true;
            }
        };

        const _TouchInput_update2 = TouchInput.update;
        TouchInput.update = function() {
            _TouchInput_update2.apply(this, arguments);
            this.updateWheelDirection();
        };

        TouchInput.updateWheelDirection = function() {
            if (this._wheelUp > 0) {
                this._wheelUp--;
                if (this._wheelUp <= 0) {
                    Input._currentState[ButtonName.UP] = false;
                }
            }
            if (this._wheelDown > 0) {
                this._wheelDown--;
                if (this._wheelDown <= 0) {
                    Input._currentState[ButtonName.DOWN] = false;
                }
            }
            if (this._wheelRight > 0) {
                this._wheelRight--;
                if (this._wheelRight <= 0) {
                    Input._currentState[ButtonName.RIGHT] = false;
                }
            }
            if (this._wheelLeft > 0) {
                this._wheelLeft--;
                if (this._wheelLeft <= 0) {
                    Input._currentState[ButtonName.LEFT] = false;
                }
            }
        };
    }
})();