/*=============================================================================
 PictureGrouping.js
----------------------------------------------------------------------------
 (C)2023 Triacontane
 This software is released under the MIT License.
 http://opensource.org/licenses/mit-license.php
----------------------------------------------------------------------------
 Version
 1.2.3 2024/11/16 最後に呼んだグループ化ピクチャ指定が表示される問題を修正
 1.2.2 2024/11/15 トリミングされた画像に対する倍率処理を修正
 1.2.1 2024/11/15 倍率指定時に切り出し範囲が影響を受けないように修正
 1.2.0 2024/11/15 個別のピクチャに倍率を指定できる機能を追加
 1.1.0 2024/10/30 ピクチャ以外のフォルダから画像を指定する機能と、画像をトリミングして表示する機能を追加
 1.0.4 2024/10/19 グルーピングピクチャを表示しているマップから別のシーンに遷移後、戻ってきたときに画像が一瞬チラつくことがある問題を修正
 1.0.3 2024/03/19 ピクチャのファイル名に制御文字を使用できるよう修正
 1.0.2 2024/01/14 同じピクチャ番号でグループピクチャを再表示したときに一瞬チラつきが発生する現象を修正
 1.0.1 2023/11/12 グループ化していないピクチャを再表示できなくなる問題を修正
 1.0.0 2023/11/12 初版
----------------------------------------------------------------------------
 [Blog]   : https://triacontane.blogspot.jp/
 [Twitter]: https://twitter.com/triacontane/
 [GitHub] : https://github.com/triacontane/
=============================================================================*/

/*:
 * @plugindesc ピクチャのグループ化プラグイン
 * @target MZ
 * @url https://github.com/triacontane/RPGMakerMV/tree/mz_master/PictureGrouping.js
 * @base PluginCommonBase
 * @orderAfter PluginCommonBase
 * @author トリアコンタン
 *
 * @command GROUPING_PICTURE
 * @text グループ化ピクチャ指定
 * @desc グループ化するピクチャファイルを指定します。リストの下が手前に表示されます。
 *
 * @arg pictureList
 * @text ピクチャリスト
 * @desc グループ化するピクチャのリストです。
 * @default []
 * @type struct<Picture>[]
 *
 * @help PictureGrouping.js
 *
 * 複数のピクチャファイルをひとつのピクチャとして表示できます。
 * 座標や拡大率、不透明度などを一括管理できます。
 * プラグインコマンドからグループ化するピクチャを指定してください。
 * その後、イベントコマンド「ピクチャの表示」を実行すると、元のピクチャのうえに
 * グループ化で指定したピクチャがすべて重なって表示されます。
 * 表示後にピクチャを追加削除することはできません。
 *　
 * このプラグインの利用にはベースプラグイン『PluginCommonBase.js』が必要です。
 * 『PluginCommonBase.js』は、RPGツクールMZのインストールフォルダ配下の
 * 以下のフォルダに格納されています。
 * dlc/BasicResources/plugins/official
 *
 * 利用規約：
 *  作者に無断で改変、再配布が可能で、利用形態（商用、18禁利用等）
 *  についても制限はありません。
 *  このプラグインはもうあなたのものです。
 */

/*~struct~Picture:
 *
 * @param FileName
 * @text ファイル名
 * @desc ピクチャのファイル名です。
 * @default
 * @dir img/pictures/
 * @type file
 *
 * @param OtherFileName
 * @text 他のファイル名
 * @desc ピクチャ以外の画像を使いたいときは、こちらを指定してください。
 * @default
 * @dir img
 * @type file
 *
 * @param Rect
 * @text トリミング範囲
 * @desc 画像を切り出して表示したい場合に指定してください。
 * @type struct<Rect>
 *
 * @param X
 * @text X座標
 * @desc ピクチャの相対X座標です。
 * @default 0
 * @type number
 *
 * @param Y
 * @text Y座標
 * @desc ピクチャの相対Y座標です。
 * @default 0
 * @type number
 *
 * @param ScaleX
 * @text X方向の倍率
 * @desc ピクチャのX方向の倍率です。100で等倍です。
 * @default 100
 * @type number
 *
 * @param ScaleY
 * @text Y方向の倍率
 * @desc ピクチャのY方向の倍率です。100で等倍です。
 * @default 100
 * @type number
 *
 */

/*~struct~Rect:
 * @param X
 * @text X座標
 * @desc 切り出し範囲のX座標です。
 * @default 0
 * @type number
 *
 * @param Y
 * @text Y座標
 * @desc 切り出し範囲のY座標です。
 * @default 0
 * @type number
 *
 * @param Width
 * @text 幅
 * @desc 切り出し範囲の幅です。
 * @default 0
 * @type number
 *
 * @param Height
 * @text 高さ
 * @desc 切り出し範囲の高さです。
 * @default 0
 * @type number
 */

(() => {
    'use strict';
    const script = document.currentScript;

    // グループ化ピクチャの登録コマンド
    PluginManagerEx.registerCommand(script, 'GROUPING_PICTURE', args => {
        $gameScreen.registerGroupingPicture(args.pictureList);
    });

    const _Game_System_onAfterLoad = Game_System.prototype.onAfterLoad;
    Game_System.prototype.onAfterLoad = function() {
        _Game_System_onAfterLoad.apply(this, arguments);
        if (!$gameScreen._groupingPictureList) {
            $gameScreen._groupingPictureList = [];
        }
    };

    const _Game_Screen_clear = Game_Screen.prototype.clear;
    Game_Screen.prototype.clear = function() {
        _Game_Screen_clear.apply(this, arguments);
        this._groupingPictureList = [];
    };

    // グループ化ピクチャの登録
    Game_Screen.prototype.registerGroupingPicture = function(pictureList) {
        this._groupingPictureList = pictureList.map(picture => {
            const name = PluginManagerEx.convertEscapeCharacters(picture.FileName);
            const otherName = PluginManagerEx.convertEscapeCharacters(picture.OtherFileName || '');
            const rect = picture.Rect || {};
            
            return {
                fileName: name,
                otherName: otherName,
                x: picture.X || 0,
                y: picture.Y || 0,
                scaleX: picture.ScaleX || 100,
                scaleY: picture.ScaleY || 100,
                rect: {
                    x: rect.X || 0,
                    y: rect.Y || 0,
                    width: rect.Width || 0,
                    height: rect.Height || 0
                }
            };
        });
        
        // 事前に画像を読み込む
        pictureList.forEach(picture => {
            const name = PluginManagerEx.convertEscapeCharacters(picture.FileName);
            if (name) ImageManager.loadPicture(name);
        });
    };

    // 登録済みグループ化ピクチャリストの取得
    Game_Screen.prototype.getGroupingPictureList = function() {
        return this._groupingPictureList || [];
    };

    // ピクチャ表示時の処理拡張
    const _Game_Picture_show = Game_Picture.prototype.show;
    Game_Picture.prototype.show = function(
        name, origin, x, y, scaleX, scaleY, opacity, blendMode
    ) {
        const groupingList = $gameScreen.getGroupingPictureList();
        
        if (groupingList && groupingList.length > 0) {
            // グループピクチャ用の情報を保存
            this._groupingList = groupingList.slice(); // 配列をコピー
            
            // 元のピクチャをリストに追加（一番下に表示）
            if (name) {
                this._groupingList.unshift({
                    fileName: name,
                    x: 0,
                    y: 0,
                    scaleX: 100,
                    scaleY: 100,
                    rect: {x: 0, y: 0, width: 0, height: 0}
                });
            }
            
            // ユニークな名前を設定（タイムスタンプベース）
            arguments[0] = 'grouping_' + Date.now().toString();
            
            // リストをクリア（既に取得済み）
            $gameScreen._groupingPictureList = [];
        }
        
        _Game_Picture_show.apply(this, arguments);
    };

    // グループ化情報の取得
    Game_Picture.prototype.groupingList = function() {
        return this._groupingList;
    };

    // Spriteのビットマップ読み込み処理を拡張
    const _Sprite_Picture_loadBitmap = Sprite_Picture.prototype.loadBitmap;
    Sprite_Picture.prototype.loadBitmap = function() {
        const picture = this.picture();
        
        // グループ化ピクチャの場合
        if (picture && picture.groupingList && picture.groupingList()) {
            this._loadingGroupPicture = true;
            this._groupingList = picture.groupingList();
            this._groupingBitmaps = [];
            
            // 各画像の読み込みを開始
            this._groupingList.forEach(data => {
                const bitmap = this._loadPictureForGrouping(data);
                this._groupingBitmaps.push(bitmap);
            });
            
            // ダミーのビットマップを一時的に設定
            this.bitmap = new Bitmap(1, 1);
        } else {
            // 通常のピクチャ処理
            _Sprite_Picture_loadBitmap.apply(this, arguments);
        }
    };

    // グループ化ピクチャ用の画像読み込み
    Sprite_Picture.prototype._loadPictureForGrouping = function(data) {
        if (data.otherName) {
            const paths = data.otherName.split('/');
            const folder = 'img/' + paths.shift() + '/';
            const name = paths.join('/');
            return ImageManager.loadBitmap(folder, name);
        } else {
            return ImageManager.loadPicture(data.fileName);
        }
    };

    // 更新処理で画像の読み込み状態をチェック
    const _Sprite_Picture_update = Sprite_Picture.prototype.update;
    Sprite_Picture.prototype.update = function() {
        _Sprite_Picture_update.apply(this, arguments);
        
        // グループ化ピクチャの読み込み中かつすべての画像が読み込み完了した場合
        if (this._loadingGroupPicture && 
            this._groupingBitmaps && 
            this._groupingBitmaps.every(bitmap => bitmap.isReady())) {
            
            this._createGroupingBitmap();
            this._loadingGroupPicture = false;
        }
    };

    // グループ化ピクチャのビットマップ作成
    Sprite_Picture.prototype._createGroupingBitmap = function() {
        try {
            if (!this.picture() || !this._groupingList) return;
            
            // 全体の必要なサイズを計算
            let maxWidth = 0;
            let maxHeight = 0;
            
            this._groupingList.forEach((data, index) => {
                const bitmap = this._groupingBitmaps[index];
                if (!bitmap || !bitmap.isReady()) return;
                
                const rect = data.rect || {};
                
                const originalX = Number(data.x) || 0;
                const originalY = Number(data.y) || 0;
                const scaleX = Number(data.scaleX) || 100;
                const scaleY = Number(data.scaleY) || 100;
                
                const sw = rect.width > 0 ? (rect.width * 100 / scaleX) : bitmap.width;
                const sh = rect.height > 0 ? (rect.height * 100 / scaleY) : bitmap.height;
                
                const dw = sw * (scaleX / 100);
                const dh = sh * (scaleY / 100);
                
                const dx = originalX * (scaleX / 100);
                const dy = originalY * (scaleY / 100);
                
                maxWidth = Math.max(maxWidth, dx + dw);
                maxHeight = Math.max(maxHeight, dy + dh);
            });
            
            // 最小サイズ保証
            maxWidth = Math.max(maxWidth, 1);
            maxHeight = Math.max(maxHeight, 1);
            
            // 合成用ビットマップ作成
            const compositeBitmap = new Bitmap(maxWidth, maxHeight);
            
            // リストを逆順に描画（下のものが手前になるように）
            for (let i = this._groupingList.length - 1; i >= 0; i--) {
                const data = this._groupingList[i];
                const bitmap = this._groupingBitmaps[i];
                
                if (!bitmap || !bitmap.isReady()) continue;
                
                const rect = data.rect || {};
                
                const originalX = Number(data.x) || 0;
                const originalY = Number(data.y) || 0;
                const scaleX = Number(data.scaleX) || 100;
                const scaleY = Number(data.scaleY) || 100;
                
                const sx = Number(rect.x) || 0;
                const sy = Number(rect.y) || 0;
                const sw = rect.width > 0 ? (rect.width * 100 / scaleX) : bitmap.width;
                const sh = rect.height > 0 ? (rect.height * 100 / scaleY) : bitmap.height;
                
                const dx = originalX * (scaleX / 100);
                const dy = originalY * (scaleY / 100);
                
                const dw = sw * (scaleX / 100);
                const dh = sh * (scaleY / 100);
                
                compositeBitmap.blt(bitmap, sx, sy, sw, sh, dx, dy, dw, dh);
            }
            
            // 作成したビットマップを設定
            this.bitmap = compositeBitmap;
            
            // 不要になったリソースを解放
            this._clearGroupingResources();
            
        } catch (e) {
            console.error("Error creating grouping bitmap:", e);
        }
    };

    // グループ化ピクチャのリソース解放
    Sprite_Picture.prototype._clearGroupingResources = function() {
        this._groupingList = null;
        this._groupingBitmaps = null;
    };
    
    // スプライト破棄時の処理
    const _Sprite_Picture_destroy = Sprite_Picture.prototype.destroy;
    Sprite_Picture.prototype.destroy = function(options) {
        this._clearGroupingResources();
        _Sprite_Picture_destroy.apply(this, arguments);
    };
})();