//=============================================================================
// Ranking.js
// ----------------------------------------------------------------------------
// Copyright (c) 2023-2023 Hangen
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.0.0 2023/05/01 初版
// 1.0.1 2023/10/16 ランキング終了後メニュー表示禁止を解除
// 1.0.2 2023/12/28 ランキング情報を配列形式で取得する機能「GetRankingData」、IDからランキング情報を取得する機能「SearchRankIdToValue」を追加、その他パラメータを追加
// ----------------------------------------------------------------------------
// [Twitter]: https://twitter.com/hangen_saimin
//=============================================================================
/*:ja
 * @target MV MZ
 * @plugindesc ランキングプラグイン
 * @author 半減
 * 
 * @help ランキング機能を提供するプラグインです。
 * 
 * 概要:
 * 各ユーザーごとに獲得したスコアをGoodleスプレッドシートに保存し、取り出してツクールの画面に表示します。
 * その他、スコアをランキングに送信する機能、ID振り分け機能、名前設定機能を有しています。
 * 
 * 注意点:
 * 1.ID振り分け機能が実行されていない場合、スコアをランキングに送信する機能は使用できません。
 * 2.プラグインパラメータ[doGetURL][gasURL][playerId][playerName]は必須項目です。必ず値を設定してください。
 * 
 * プラグインコマンド:
 *   イベントコマンド「プラグインコマンド」から実行。
 *   （引数の間は半角スペースで区切る）
 *   "\v[0]"のように変数を指定することも可能
 * 
 * 
 * DisplayRanking [読み込むランキングのボードID(1～50)] [新規スコア表示区分(ON/OFF)] [ランキング表示順(ASK/DESK)] [処理ステータス格納変数番号(省略可能)]
 * 例：DisplayRanking 1 ON DESK 99
 * 上記例の場合、1番目のGoogleスプレッドシートを読み込み、ランキング表示順は降順(スコアが高いデータが上位)、新規スコアがあった場合は上部に表示し、
 * 変数番号99に処理ステータスを格納します。
 * 処理ステータスはランキング表示処理終了後に、ランキング表示に成功した場合は[1]、失敗した場合は[9999]が格納されます。
 * プラグインコマンド実行後にループ処理を入れ、ループ中断の条件に「処理ステータスが初期値以外の場合」を入れることでランキング表示までの間、別の処理が入らないようにすることができます。
 * また、処理ステータス格納変数番号を設定していない場合は、適宜ウェイトを入れて、ランキング表示までの間を調節することを推奨します。
 * 
 * SendRanking [書き込むランキングのボードID(1～50)] [書き込む値] [書き込み判定基準(UP/DOWN/NON)] [処理ステータス格納変数番号(省略可能)]
 * 例：SendRanking 1 100 UP 99
 * 上記例の場合、1番目のGoogleスプレッドシートに書き込み、書き込む値は100、書き込む条件は登録済みの値に対して、それよりも大きい値の場合、
 * 変数番号99に処理ステータスを格納します。
 * 書き込む判定基準はUPが　「登録済みの値に対して、それよりも大きい値の場合」に上書き。
 * 書き込む判定基準はDOWNが「登録済みの値に対して、それよりも小さい値の場合」に上書き。
 * 書き込む判定基準はNONが 「条件なし」で必ず上書き。
 * 処理ステータスは書き込み処理終了後に、書き込みに成功した場合は[1]、失敗した場合は[9999]が格納されます。
 * プラグインコマンド実行後にループ処理を入れ、ループ中断の条件に「処理ステータスが初期値以外の場合」を入れることで書き込み完了までの間、別の処理が入らないようにすることができます。
 * また、処理ステータス格納変数番号を設定していない場合は、適宜ウェイトを入れて、書き込みまでの間を調節することを推奨します。
 * 注意点として、後述のRankIdSetを実行していない場合はエラーとなります。
 * 
 * RankIdSet
 * 例：RankIdSet
 * プラグインパラメータ[playerId]に設定した変数番号にプレイヤーごとに一意となる番号を設定します。
 * この設定が実行されていない場合、プラグインコマンド[SendRanking]はエラーとなります。
 * また、引数に[UPD]を設定することで、IDの上書きを行うことも可能です(RankIdSet UPD)
 * 
 * RankNameSetActor [アクターID]
 * 例：RankNameSetActor 1
 * 上記例の場合、アクターIDを参照し、アクターの名前をプラグインパラメータ[playerName]に設定した変数番号に設定します。
 * 
 * RankNameSet [登録名]
 * 例：RankNameSet ハロルド
 * 上記例の場合、引数に設定された登録名をプラグインパラメータ[playerName]に設定した変数番号に設定します。
 * 上記プラグインコマンドを使用せず、直接プラグインパラメータ[playerName]に設定した変数番号に設定しても問題ありません。
 * 
 * GetRankingData [取得するランキングのボードID(1～50)] [ランキングソート順(ASK/DESK)] [ID一覧格納変数番号] [順位一覧格納変数番号] [名前一覧格納変数番号] [値一覧格納変数番号] [処理ステータス格納変数番号(省略可能)]
 * 例：RankNameSet 1 DESK 10 11 12 13 99
 * 上記例の場合、1番目のGoogleスプレッドシートを読み込み、ランキング表示順は降順(スコアが高いデータが上位)、
 * 変数番号10にID一覧を格納、変数番号11に順位一覧を格納、変数番号12に名前一覧を格納、変数番号13に値一覧を格納、変数番号99に処理ステータスを格納します。
 * 処理ステータスはランキング取得処理終了後に、ランキング取得に成功した場合は[1]、失敗した場合は[9999]が格納されます。
 * プラグインコマンド実行後にループ処理を入れ、ループ中断の条件に「処理ステータスが初期値以外の場合」を入れることでランキング取得までの間、別の処理が入らないようにすることができます。
 * また、処理ステータス格納変数番号を設定していない場合は、適宜ウェイトを入れて、ランキング取得までの間を調節することを推奨します。
 * ※追加情報
 * 　取得したデータは配列形式で変数に格納されます。
 * 　データを取得する場合は「$gameVariables.value(変数番号)[数字]」のようにすることで、配列内の任意のデータを取得可能です。
 * 　配列はランキングと同様にASK(昇順)、またはDESK(降順)で格納されているため、配列0番は1位、1番は2位……といった内容になっています。
 * 　例１：gameVariables.value(10)[0]
 * 　上記例の場合、1位のIDを取得します。
 * 　例２：gameVariables.value(11)[1]
 * 　上記例の場合、2位の名前を取得します。
 * 
 * SearchRankIdToValue [検索するランキングのボードID(1～50)] [検索ID] [値格納用変数] [処理ステータス格納変数番号(省略可能)] [配列設定OFF(省略可能)]
 * 例：SearchRankIdToValue 2 2302271212147891234 20 99
 * 上記例の場合、1番目のGoogleスプレッドシートからID「2302271212147891234」を検索し、取得した値(value)を変数20番に格納、変数番号99に処理ステータスを格納します。
 * 処理ステータスは検索処理終了後に、検索に成功した場合は[1]、失敗した場合は[9999]が格納されます。
 * プラグインコマンド実行後にループ処理を入れ、ループ中断の条件に「処理ステータスが初期値以外の場合」を入れることで検索終了までの間、別の処理が入らないようにすることができます。
 * また、処理ステータス格納変数番号を設定していない場合は、適宜ウェイトを入れて、検索終了までの間を調節することを推奨します。
 * ※追加情報
 * 　上記で取得したValueが「123|234|345|456|567」のように"|"で区切った文字列の場合、配列形式で格納されます。
 * 　配列形式の場合は「$gameVariables.value(変数番号)[数字]」のようにすることで、配列内の任意のデータを取得可能です。
 * 　配列は左から順に格納されたデータとなっています。
 * 　例１：gameVariables.value(20)[0]
 * 　上記例の場合、123を取得します。
 * 　例２：gameVariables.value(20)[3]
 * 　上記例の場合、456を取得します。
 * 
 * 
 * 利用規約：
 *  再配布、改変、商用利用、R18作品での使用、全て許可します。
 *  作者への連絡等も不要です。
 *  クレジットへの記載もご自由にどうぞ。
 * 
 * @param gasURL
 * @text AppsScriptのデプロイURL
 * @type string
 * @desc GoogleスプレッドシートのデプロイURL
 * @default
 * 
 * @param playerId
 * @text プレイヤーID格納用変数番号
 * @type variable
 * @desc プレイヤーごとに一意に触れらるIDを格納する変数番号
 * @default
 * 
 * @param playerName
 * @text プレイヤー名格納用変数番号
 * @type variable
 * @desc ランキングに表示する名前を格納する
 * @default
 * 
 * @param rankingFontSize
 * @text ランキング情報フォントサイズ
 * @type number
 * @desc  ランキング情報（順位、スコア、名前）のフォントサイズ
 * @default 13
 * 
 * @param rankingTitleFontSize
 * @text ランキングタイトルフォントサイズ
 * @type number
 * @desc ランキングタイトルのフォントサイズ
 * @default 16
 * 
 * @param ranktextTitle
 * @text ランキングタイトル
 * @type string
 * @desc ランキングタイトルに表示する文字列
 * @default ランキングボード
 * 
 * @param recordTitle
 * @text 今回のスコアタイトル
 * @type string
 * @desc 今回スコアが存在する場合に表示するタイトル
 * @default 今回のスコア
 * 
 * @param recordScoreName
 * @text 今回のスコア-スコア名称
 * @type string
 * @desc 今回スコアのスコア名称
 * @default スコア
 * 
 * @param recordRankName
 * @text 今回のスコア-順位名称
 * @type string
 * @desc 今回スコアの順位名称
 * @default 順位
 * 
 * @param rankEndStr
 * @text 順位の末尾文字
 * @type string
 * @desc 順位の末尾に表示する文字
 * @default 位
 * 
 * @param valueEndStr
 * @text スコアの末尾文字
 * @type string
 * @desc スコアの末尾に表示する文字
 * @default pt
 * 
 * @param nameEndStr
 * @text 名前の末尾文字
 * @type string
 * @desc 名前の末尾に表示する文字
 * @default さん
 * 
 * @param fontFamily
 * @text フォント種類
 * @type string
 * @desc 文字列のフォントを指定
 * @default Arial
 * 
 * @param noName
 * @text 名前未設定時名称
 * @type string
 * @desc 名前が設定されていない場合のデフォルト登録名
 * @default ゲスト
 * 
 * @param batsuString
 * @text 閉じるボタン文字
 * @type string
 * @desc 閉じるボタンの表記
 * @default ✕
 * 
 * @param rankingMenuDispFlg
 * @text ランキング表示終了後メニュー表示フラグ
 * @type boolean
 * @on メニュー表示禁止を解除する
 * @off 解除しない
 * @desc ランキング表示終了後メニュー表示禁止を解除するかしないか
 * @default true
 * 
 * @param 色変更
 * 
 * @param bgColor
 * @parent 色変更
 * @text 背景色
 * @type string
 * @desc 背景色のカラーコード
 * @default 000000
 * 
 * @param bgAlpha
 * @parent 色変更
 * @text 背景色-不透明度
 * @type number
 * @min 0.0
 * @max 1.0
 * @desc 背景色の不透明度
 * @decimals 1
 * @default 0.5
 * 
 * @param bgRankingColor
 * @parent 色変更
 * @text ランキング背景色
 * @type string
 * @desc ランキングの背景色のカラーコード
 * @default f8f8f8
 * 
 * @param bgRankingAlpha
 * @parent 色変更
 * @text ランキング背景色-不透明度
 * @type number
 * @min 0.0
 * @max 1.0
 * @desc ランキング背景色の不透明度
 * @decimals 1
 * @default 1.0
 * 
 * @param rankingTitleColor
 * @parent 色変更
 * @text ランキングタイトル色
 * @type string
 * @desc ランキングタイトル色のカラーコード
 * @default 45434a
 * 
 * @param rankColor
 * @parent 色変更
 * @text 順位色
 * @type string
 * @desc 順位色のカラーコード
 * @default 45434a
 * 
 * @param rankBgColor
 * @parent 色変更
 * @text 順位背景色
 * @type string
 * @desc 順位背景色のカラーコード
 * @default FFFFFFF
 * 
 * @param rankBgAlpha
 * @parent 色変更
 * @text 順位背景色-不透明度
 * @type number
 * @min 0.0
 * @max 1.0
 * @desc 順位背景色の不透明度
 * @decimals 1
 * @default 1.0
 * 
 * @param scoreColor
 * @parent 色変更
 * @text スコア色
 * @type string
 * @desc スコア色のカラーコード
 * @default 45434a
 * 
 * @param nameColor
 * @parent 色変更
 * @text 名前色
 * @type string
 * @desc 名前色のカラーコード
 * @default 45434a
 * 
 * @param thisScoreTitleColor
 * @parent 色変更
 * @text 今回のスコアタイトル色
 * @type string
 * @desc 今回のスコアタイトル色のカラーコード
 * @default 45434a
 * 
 * @param thisScoreSubTitleColor
 * @parent 色変更
 * @text 今回のスコア-スコアサブタイトル色
 * @type string
 * @desc スコアサブタイトル色のカラーコード
 * @default 45434a
 * 
 * @param thisRankSubTitleColor
 * @parent 色変更
 * @text 今回のスコア-順位サブタイトル色
 * @type string
 * @desc 順位サブタイトル色のカラーコード
 * @default 45434a
 * 
 * @param thisScoreColor
 * @parent 色変更
 * @text 今回のスコア-スコア色
 * @type string
 * @desc 今回のスコア-スコア色のカラーコード
 * @default 45434a
 * 
 * @param thisRankColor
 * @parent 色変更
 * @text 今回のスコア-順位色
 * @type string
 * @desc 今回のスコア-順位色のカラーコード
 * @default 45434a
 * 
 * @param batsuButtonFontColor
 * @parent 色変更
 * @text 閉じるボタン-バツ色
 * @type string
 * @desc 閉じるボタン-バツ色のカラーコード
 * @default 000000
 * 
 * @param batsuButtonBgColor
 * @parent 色変更
 * @text 閉じるボタン-背景色
 * @type string
 * @desc 閉じるボタン-背景色のカラーコード
 * @default C1C1C1
 * 
 * @param batsuButtonBgAlpha
 * @parent 色変更
 * @text 閉じるボタン背景色-不透明度
 * @type number
 * @min 0.0
 * @max 1.0
 * @desc 閉じるボタン背景色の不透明度
 * @decimals 1
 * @default 1.0
 * 
 * 
 * @param サイズ変更
 * 
 * @param bgSizeWidth
 * @parent サイズ変更
 * @text ランキング背景サイズ横
 * @type number
 * @min -10000
 * @max 10000
 * @desc ランキング背景サイズ横のサイズ調整を行う数値。正数で広げる。負数で狭める。
 * @default 0
 * 
 * @param bgSizeHeight
 * @parent サイズ変更
 * @text ランキング背景サイズ縦
 * @type number
 * @min -10000
 * @max 10000
 * @desc ランキング背景サイズ縦のサイズ調整を行う数値。正数で広げる。負数で狭める。
 * @default 0
 * 
 * 
 * @param 位置調整
 * 
 * @param bgMoveX
 * @parent 位置調整
 * @text ランキング背景横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc ランキング背景の位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param bgMoveY
 * @parent 位置調整
 * @text ランキング背景縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc ランキングタイトルの位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 * @param rankingTitleMoveX
 * @parent 位置調整
 * @text ランキングタイトル横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc ランキングタイトルの位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param rankingTitleMoveY
 * @parent 位置調整
 * @text ランキングタイトル縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc ランキングタイトルの位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 * @param rankMoveX
 * @parent 位置調整
 * @text 順位横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 順位の位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param rankMoveY
 * @parent 位置調整
 * @text 順位縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 順位の位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 * @param scoreMoveX
 * @parent 位置調整
 * @text スコア横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc スコアの位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param scoreMoveY
 * @parent 位置調整
 * @text スコア縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc スコアの位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 * @param nameMoveX
 * @parent 位置調整
 * @text 名前横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 名前の位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param nameMoveY
 * @parent 位置調整
 * @text 名前縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 名前の位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 * @param thisScoreTitleMoveX
 * @parent 位置調整
 * @text 今回のスコアタイトル横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 今回のスコアの位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param thisScoreTitleMoveY
 * @parent 位置調整
 * @text 今回のスコアタイトル縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 今回のスコアの位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 * @param thisScoreSubTitleMoveX
 * @parent 位置調整
 * @text 今回のスコア-スコアサブタイトル横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc スコアサブタイトルの位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param thisScoreSubTitleMoveY
 * @parent 位置調整
 * @text 今回のスコア-スコアサブタイトル縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc スコアサブタイトルの位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 * @param thisRankSubTitleMoveX
 * @parent 位置調整
 * @text 今回のスコア-順位サブタイトル横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 順位サブタイトルの位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param thisRankSubTitleMoveY
 * @parent 位置調整
 * @text 今回のスコア-順位サブタイトル縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 順位サブタイトルの位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 * @param thisScoreMoveX
 * @parent 位置調整
 * @text 今回のスコア-スコア横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 今回のスコア-スコアの位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param thisScoreMoveY
 * @parent 位置調整
 * @text 今回のスコア-スコア縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 今回のスコア-スコアの位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 * @param thisRankMoveX
 * @parent 位置調整
 * @text 今回のスコア-順位横位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 今回のスコア-順位の位置調整を行う数値。正数で右に、負数で左に移動。
 * @default 0
 * 
 * @param thisRankMoveY
 * @parent 位置調整
 * @text 今回のスコア-順位縦位置調整
 * @type number
 * @min -10000
 * @max 10000
 * @desc 今回のスコア-順位の位置調整を行う数値。正数で下に、負数で上に移動。
 * @default 0
 * 
 */

(function(){
  'use strict';

  var gasURL = PluginManager.parameters("Ranking")["gasURL"];
  var playerId = PluginManager.parameters("Ranking")["playerId"];
  var playerName = PluginManager.parameters("Ranking")["playerName"];
  var rankingFontSize = PluginManager.parameters("Ranking")["rankingFontSize"];
  var rankingTitleFontSize = PluginManager.parameters("Ranking")["rankingTitleFontSize"];
  var ranktextTitle = String(PluginManager.parameters("Ranking")["ranktextTitle"]);
  var recordTitle = String(PluginManager.parameters("Ranking")["recordTitle"]);
  var recordScoreName = String(PluginManager.parameters("Ranking")["recordScoreName"]);
  var recordRankName = String(PluginManager.parameters("Ranking")["recordRankName"]);
  var rankEndStr = String(PluginManager.parameters("Ranking")["rankEndStr"]);
  var valueEndStr = String(PluginManager.parameters("Ranking")["valueEndStr"]);
  var nameEndStr = String(PluginManager.parameters("Ranking")["nameEndStr"]);
  var fontFamily = String(PluginManager.parameters("Ranking")["fontFamily"]);
  var noName = String(PluginManager.parameters("Ranking")["noName"]);
  var batsuString = String(PluginManager.parameters("Ranking")["batsuString"]);
  var rankingMenuDispFlg = PluginManager.parameters("Ranking")["rankingMenuDispFlg"];

  // 色変更用
  var bgColor = '0x' + String(PluginManager.parameters("Ranking")["bgColor"]);
  var bgAlpha = PluginManager.parameters("Ranking")["bgAlpha"];
  var bgRankingColor = '0x' + String(PluginManager.parameters("Ranking")["bgRankingColor"]);
  var bgRankingAlpha = PluginManager.parameters("Ranking")["bgRankingAlpha"];
  var rankingTitleColor = '0x' + String(PluginManager.parameters("Ranking")["rankingTitleColor"]);
  var rankColor = '0x' + String(PluginManager.parameters("Ranking")["rankColor"]);
  var rankBgColor = '0x' + String(PluginManager.parameters("Ranking")["rankBgColor"]);
  var rankBgAlpha = PluginManager.parameters("Ranking")["rankBgAlpha"];
  var scoreColor = '0x' + String(PluginManager.parameters("Ranking")["scoreColor"]);
  var nameColor = '0x' + String(PluginManager.parameters("Ranking")["nameColor"]);
  var thisScoreTitleColor = '0x' + String(PluginManager.parameters("Ranking")["thisScoreTitleColor"]);
  var thisScoreSubTitleColor = '0x' + String(PluginManager.parameters("Ranking")["thisScoreSubTitleColor"]);
  var thisRankSubTitleColor = '0x' + String(PluginManager.parameters("Ranking")["thisRankSubTitleColor"]);
  var thisScoreColor = '0x' + String(PluginManager.parameters("Ranking")["thisScoreColor"]);
  var thisRankColor = '0x' + String(PluginManager.parameters("Ranking")["thisRankColor"]);
  var batsuButtonFontColor = '0x' + String(PluginManager.parameters("Ranking")["batsuButtonFontColor"]);
  var batsuButtonBgColor = '0x' + String(PluginManager.parameters("Ranking")["batsuButtonBgColor"]);
  var batsuButtonBgAlpha = PluginManager.parameters("Ranking")["batsuButtonBgAlpha"];

  // サイズ変更
  var bgSizeWidth = Number(PluginManager.parameters("Ranking")["bgSizeWidth"]);
  var bgSizeHeight = Number(PluginManager.parameters("Ranking")["bgSizeHeight"]);

  // 位置調整用
  var bgMoveX = Number(PluginManager.parameters("Ranking")["bgMoveX"]);
  var bgMoveY = Number(PluginManager.parameters("Ranking")["bgMoveY"]);
  var rankingTitleMoveX = Number(PluginManager.parameters("Ranking")["rankingTitleMoveX"]);
  var rankingTitleMoveY = Number(PluginManager.parameters("Ranking")["rankingTitleMoveY"]);
  var thisScoreTitleMoveX = Number(PluginManager.parameters("Ranking")["thisScoreTitleMoveX"]);
  var thisScoreTitleMoveY = Number(PluginManager.parameters("Ranking")["thisScoreTitleMoveY"]);
  var thisScoreSubTitleMoveX = Number(PluginManager.parameters("Ranking")["thisScoreSubTitleMoveX"]);
  var thisScoreSubTitleMoveY = Number(PluginManager.parameters("Ranking")["thisScoreSubTitleMoveY"]);
  var thisScoreMoveX = Number(PluginManager.parameters("Ranking")["thisScoreMoveX"]);
  var thisScoreMoveY = Number(PluginManager.parameters("Ranking")["thisScoreMoveY"]);
  var thisRankSubTitleMoveX = Number(PluginManager.parameters("Ranking")["thisRankSubTitleMoveX"]);
  var thisRankSubTitleMoveY = Number(PluginManager.parameters("Ranking")["thisRankSubTitleMoveY"]);
  var thisRankMoveX = Number(PluginManager.parameters("Ranking")["thisRankMoveX"]);
  var thisRankMoveY = Number(PluginManager.parameters("Ranking")["thisRankMoveY"]);
  var rankMoveX = Number(PluginManager.parameters("Ranking")["rankMoveX"]);
  var rankMoveY = Number(PluginManager.parameters("Ranking")["rankMoveY"]);
  var scoreMoveX = Number(PluginManager.parameters("Ranking")["scoreMoveX"]);
  var scoreMoveY = Number(PluginManager.parameters("Ranking")["scoreMoveY"]);
  var nameMoveX = Number(PluginManager.parameters("Ranking")["nameMoveX"]);
  var nameMoveY = Number(PluginManager.parameters("Ranking")["nameMoveY"]);

  var addChildSceneList=[];
  var addChildSceneListBgOnly=[];

  var rankingDispFlg = false;
  var rankingDispOutputFlg = false;
  var recordDispFlg = false;
  var rankingData;
  var scrollPosition = 0;
  var prevVal = "";
  var ranktextRank = "";
  var ranktextName = "";
  var ranktextValue = "";
  var strByteRankMax = 0;
  var strByteNameMax = 0;
  var strByteValueMax = 0;
  var recordScore = "";
  var recordRank = "";
  var background2;
  var rankPlusVal = 1;
  var args0 = "";
  var args1 = "";
  var args2 = "";
  var args3 = "";
  var args4 = "";
  var args5 = "";
  var args6 = "";
  var widthDefault;
  var yDefault;
  var yRecordDefault;
  var screenDispLimit;
  var scrollLowerLimit;
  var batsuFlg = false;
  // 今回のスコア用
  var nowScore = "";

  // シーンに子要素を追加
  const obj = {
    addChild(disp) {
      SceneManager._scene.addChild(disp);
      addChildSceneList.push(disp);
    },
    addChildBg(disp) {
      SceneManager._scene.addChild(disp);
      addChildSceneListBgOnly.push(disp);
    }
  }

  // プラグインコマンドの定義
  var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;

  Game_Interpreter.prototype.pluginCommand = function(command, args){
    _Game_Interpreter_pluginCommand.call(this, command, args);
      
      // プラグインコマンドの確認
        if (command.indexOf('DisplayRanking') != -1) {
          rankingDispFlg = true;

          // 引数を設定
          if (args.length > 0){
            args0 = args[0];
            // 変数を指定している場合は分解して変数から取得
            if (args0.substring(1, 3) == "v[" ) {
              args0 = $gameVariables.value(args0.substring(3, args0.length - 1));
            }
            if (args.length > 1){
              args1 = args[1];
              // 変数を指定している場合は分解して変数から取得
              if (args1.substring(1, 3) == "v[" ) {
                args1 = $gameVariables.value(args1.substring(3, args1.length - 1));
              }
              if (args.length > 2) {
                args2 = args[2];
                // 変数を指定している場合は分解して変数から取得
                if (args2.substring(1, 3) == "v[" ) {
                  args2 = $gameVariables.value(args2.substring(3, args2.length - 1));
                }
                if (args.length > 3) {
                  args3 = args[3];
                  // 変数を指定している場合は分解して変数から取得
                  if (args3.substring(1, 3) == "v[" ) {
                    args3 = $gameVariables.value(args3.substring(3, args3.length - 1));
                  }
                }
              }
            }
          }

          $gameSystem.disableMenu();
          
          fetch(gasURL + "?board=" + args0, {method: "GET",})
              .then(response => response.json())
              .then(data => {
                  widthDefault = Graphics.width / 3 + 40;
                  yDefault = 80;
                  yRecordDefault = yDefault;
                  rankingDispOutputFlg = true;
                  rankingData = data;
                  batsuFlg = true;
                  var background1 = new PIXI.Graphics();
                  background2 = new PIXI.Graphics();
                  // 第２引数がONの場合は新規値を上部に設定する
                  if (args1 == 'ON' && recordDispFlg){
                    yDefault += 100;
                  }
                  // 画面表示限界数
                  screenDispLimit = Math.floor((Graphics.height - yDefault) / ((rankingFontSize * 2) + 6));

                  background1.beginFill(bgColor, 1);
                  background1.alpha = bgAlpha;
                  background1.drawRect(0,0,Graphics.width, Graphics.height); 
                  background1.endFill();
                  obj.addChildBg(background1);

                  // 第３引数がAskの場合は昇順、未設定またはDeskの場合は降順
                  if (args2 == "ASK") {
                    rankingData.sort(compareAsk);  
                  } else {
                    rankingData.sort(compareDesk);
                  }
                  run.rankDispMain();

              })
              .catch((err) => {
                // 第４引数が設定されている場合は処理ステータス(false)を返却
                if (args3 != "") $gameVariables.setValue(args3, 9999);
                alert("通信に失敗しました");
              });

        } else if (command.indexOf('SendRanking') != -1) {

          // 引数を設定
          if (args.length > 0) {
            args0 = args[0];
            // 変数を指定している場合は分解して変数から取得
            if (args0.substring(1, 3) == "v[" ) {
              args0 = $gameVariables.value(args0.substring(3, args0.length - 1));
            }
            if (args.length > 1) {
              args1 = args[1];
              // 変数を指定している場合は分解して変数から取得
              if (args1.substring(1, 3) == "v[" ) {
                args1 = $gameVariables.value(args1.substring(3, args1.length - 1));
              }
              if (args.length > 2) {
                args2 = args[2];
                // 変数を指定している場合は分解して変数から取得
                if (args2.substring(1, 3) == "v[" ) {
                  args2 = $gameVariables.value(args2.substring(3, args2.length - 1));
                }
                if (args.length > 3) {
                  args3 = args[3];
                  // 変数を指定している場合は分解して変数から取得
                  if (args3.substring(1, 3) == "v[" ) {
                    args3 = $gameVariables.value(args3.substring(3, args3.length - 1));
                  }
                }
              }
            }
          }

          // ランキングに登録するIDが未設定の場合
          if ("" != $gameVariables.value(playerId)) {
            var form = document.createElement('form');
            form.action = '/hoge';
            form.method = 'POST';
            var formData = new FormData(form);
            formData.append("id", $gameVariables.value(playerId));
            // 名前が設定されていない場合は名称未設定用の名称を設定
            var name = noName;
            if ("" != $gameVariables.value(playerName)) {
              name = $gameVariables.value(playerName);
            }
            formData.append("name", name);
            formData.append("board", args0);
            formData.append("value", args1);
            // 第3引数が無くても動くように
            var updateFlg = "UP";
            if (args2 != "") {
              updateFlg = args2;
            }
            formData.append("updateFlg", updateFlg);
            fetch(gasURL, {method: "POST",body: formData,})
            .then()
            .catch((err) => {
              // 第４引数が設定されている場合は処理ステータス(false)を返却
              if (args3 != "") $gameVariables.setValue(args3, 9999);
              alert("通信に失敗しました");
            });
            rankingDispFlg = false;
            recordDispFlg = true;
            // 今回のスコア用に格納
            nowScore = args1;
            // 第４引数が設定されている場合は処理ステータス(true)を返却
            if (args3 != "") $gameVariables.setValue(args3, 1);
            
          } else {
            // 第４引数が設定されている場合は処理ステータス(false)を返却
            if (args3 != "") $gameVariables.setValue(args3, 9999);
            alert("ランキング登録用のIDが設定されていません。\n製作者にご連絡をお願いします。");
          }

        } else if (command.indexOf('RankIdSet') != -1) {

          // 引数を設定
          if (args.length > 0){
            args0 = args[0];
            // 変数を指定している場合は分解して変数から取得
            if (args0.substring(1, 3) == "v[" ) {
              args0 = $gameVariables.value(args0.substring(3, args0.length - 1));
            }
            if (args.length > 1){
              args1 = args[1];
              // 変数を指定している場合は分解して変数から取得
              if (args1.substring(1, 3) == "v[" ) {
                args1 = $gameVariables.value(args1.substring(3, args1.length - 1));
              }
              if (args.length > 2){
                args2 = args[2];
                // 変数を指定している場合は分解して変数から取得
                if (args2.substring(1, 3) == "v[" ) {
                  args2 = $gameVariables.value(args2.substring(3, args2.length - 1));
                }
              }
            }
          }

          // IDがすでに設定済みの場合は基本的に起動しない
          if("" == $gameVariables.value(playerId) || args0 == "UPD") {
            var date = new Date();
            var rand = String(Math.floor( Math.random() * (9999 + 1 - 1) ) + 1).padStart(4, '0');
            var dateYyyyMmDdHhMmSssss = String(date.getFullYear()).substring(2,4) + ('0' + (date.getMonth() + 1)).slice(-2) + ('0' + date.getDate()).slice(-2) + ('0' + date.getHours()).slice(-2) + ('0' + date.getMinutes()).slice(-2) + ('0' + date.getSeconds()).slice(-2) + date.getMilliseconds();
            var id = String(dateYyyyMmDdHhMmSssss) + String(rand);
            $gameVariables.setValue(playerId, id);
          }

        } else if (command.indexOf('RankNameSetActor') != -1){

          // 引数を設定
          if (args.length > 0) {
            args0 = args[0];
            // 変数を指定している場合は分解して変数から取得
            if (args0.substring(1, 3) == "v[" ) {
              args0 = $gameVariables.value(args0.substring(3, args0.length - 1));
            }
            if (args.length > 1){
              args1 = args[1];
              // 変数を指定している場合は分解して変数から取得
              if (args1.substring(1, 3) == "v[" ) {
                args1 = $gameVariables.value(args1.substring(3, args1.length - 1));
              }
              if (args.length > 2){
                args2 = args[2];
                // 変数を指定している場合は分解して変数から取得
                if (args2.substring(1, 3) == "v[" ) {
                  args2 = $gameVariables.value(args2.substring(3, args2.length - 1));
                }
              }
            }
          }

          $gameVariables.setValue(playerName, $gameActors.actor(args0).name());
        } else if (command.indexOf('RankNameSet') != -1) {

          // 引数を設定
          if (args.length > 0) {
            args0 = args[0];
            // 変数を指定している場合は分解して変数から取得
            if (args0.substring(1, 3) == "v[" ) {
              args0 = $gameVariables.value(args0.substring(3, args0.length - 1));
            }
            if (args.length > 1) {
              args1 = args[1];
              // 変数を指定している場合は分解して変数から取得
              if (args1.substring(1, 3) == "v[" ) {
                args1 = $gameVariables.value(args1.substring(3, args1.length - 1));
              }
              if (args.length > 2) {
                args2 = args[2];
                // 変数を指定している場合は分解して変数から取得
                if (args2.substring(1, 3) == "v[" ) {
                  args2 = $gameVariables.value(args2.substring(3, args2.length - 1));
                }
              }
            }
          }

          $gameVariables.setValue(playerName, args0);
        } else if (command.indexOf('GetRankingData') != -1) {

          // 引数を設定
          if (args.length > 0) {
            args0 = args[0];
            // 変数を指定している場合は分解して変数から取得
            if (args0.substring(1, 3) == "v[" ) {
              args0 = $gameVariables.value(args0.substring(3, args0.length - 1));
            }
            if (args.length > 1) {
              args1 = args[1];
              // 変数を指定している場合は分解して変数から取得
              if (args1.substring(1, 3) == "v[" ) {
                args1 = $gameVariables.value(args1.substring(3, args1.length - 1));
              }
              if (args.length > 2) {
                args2 = args[2];
                // 変数を指定している場合は分解して変数から取得
                if (args2.substring(1, 3) == "v[" ) {
                  args2 = $gameVariables.value(args2.substring(3, args2.length - 1));
                }
                if (args.length > 3) {
                  args3 = args[3];
                  // 変数を指定している場合は分解して変数から取得
                  if (args3.substring(1, 3) == "v[" ) {
                    args3 = $gameVariables.value(args3.substring(3, args3.length - 1));
                  }
                  if (args.length > 4) {
                    args4 = args[4];
                    // 変数を指定している場合は分解して変数から取得
                    if (args4.substring(1, 3) == "v[" ) {
                      args4 = $gameVariables.value(args4.substring(3, args4.length - 1));
                    }
                    if (args.length > 5) {
                      args5 = args[5];
                      // 変数を指定している場合は分解して変数から取得
                      if (args5.substring(1, 3) == "v[" ) {
                        args5 = $gameVariables.value(args5.substring(3, args5.length - 1));
                      }
                      if (args.length > 6) {
                        args6 = args[6];
                        // 変数を指定している場合は分解して変数から取得
                        if (args6.substring(1, 3) == "v[" ) {
                          args6 = $gameVariables.value(args6.substring(3, args6.length - 1));
                        }
                      }
                    }
                  }
                }
              }
            }
          }

          // 引数１に設定したランキングボードからIDに合致するデータを取得する
          fetch(gasURL + "?board=" + args0, {method: "GET",})
              .then(response => response.json())
              .then(data => {
                rankingData = data;
                // 第２引数がAskの場合は昇順、未設定またはDeskの場合は降順
                if (args1 == "ASK") {
                  rankingData.sort(compareAsk);  
                } else {
                  rankingData.sort(compareDesk);
                }
                var listRankVal = 0;
                var listRankPlusVal = 1;
                var listRanktextId = [];
                var listRanktextRank = [];
                var listRanktextName = [];
                var listRanktextValue = [];
                var listPrevVal = "";
                
                for(let todoObj of rankingData) {
                  // 同値順位設定処理
                  if (listPrevVal == todoObj.value) {
                    listRankPlusVal++;
                  } else {
                    listRankVal += listRankPlusVal;
                    listRankPlusVal = 1;
                    listPrevVal = todoObj.value;
                  }
                  // 順位が未設定の場合のみ1を設定
                  if (0 == listRankVal) {
                    listRankVal = 1;
                  }
                  
                  // 配列を作成
                  listRanktextId.push(todoObj.id);
                  listRanktextRank.push(listRankVal + rankEndStr);
                  listRanktextName.push(todoObj.name + " " + nameEndStr);
                  listRanktextValue.push(todoObj.value.toLocaleString() + " " + valueEndStr);
                }
                // 変数に設定
                if (args2 != "") $gameVariables.setValue(args2, listRanktextId);
                if (args3 != "") $gameVariables.setValue(args3, listRanktextRank);
                if (args4 != "") $gameVariables.setValue(args4, listRanktextName);
                if (args5 != "") $gameVariables.setValue(args5, listRanktextValue);
                // 第７引数が設定されている場合は処理ステータス(true)を返却
                if (args6 != "") $gameVariables.setValue(args6, 1);
              })
              .catch((err) => {
                // 第７引数が設定されている場合は処理ステータス(false)を返却
                if (args6 != "") $gameVariables.setValue(args6, 9999);
                alert("通信に失敗しました");
              });
        } else if (command.indexOf('SearchRankIdToValue') != -1) {

          // 引数を設定
          if (args.length > 0) {
            args0 = args[0];
            // 変数を指定している場合は分解して変数から取得
            if (args0.substring(1, 3) == "v[" ) {
              args0 = $gameVariables.value(args0.substring(3, args0.length - 1));
            }
            if (args.length > 1) {
              args1 = args[1];
              // 変数を指定している場合は分解して変数から取得
              if (args1.substring(1, 3) == "v[" ) {
                args1 = $gameVariables.value(args1.substring(3, args1.length - 1));
              }
              if (args.length > 2) {
                args2 = args[2];
                // 変数を指定している場合は分解して変数から取得
                if (args2.substring(1, 3) == "v[" ) {
                  args2 = $gameVariables.value(args2.substring(3, args2.length - 1));
                }
                if (args.length > 3) {
                  args3 = args[3];
                  // 変数を指定している場合は分解して変数から取得
                  if (args3.substring(1, 3) == "v[" ) {
                    args3 = $gameVariables.value(args3.substring(3, args3.length - 1));
                  }
                  if (args.length > 4) {
                    args4 = args[4];
                    // 変数を指定している場合は分解して変数から取得
                    if (args4.substring(1, 3) == "v[" ) {
                      args4 = $gameVariables.value(args4.substring(3, args4.length - 1));
                    }
                  }
                }
              }
            }
          }

          // 引数１に設定したランキングボードからIDに合致するデータを取得する
          fetch(gasURL + "?board=" + args0, {method: "GET",})
              .then(response => response.json())
              .then(data => {
                rankingData = data;
                
                for(let todoObj of rankingData) {
                  // 第２引数とIDが合致するデータの値を取得
                  if (args1 == todoObj.id) {
                    // 第３引数の変数番号に値をセットする
                    // その際に、データ内に|が存在する場合は配列にする
                    // ただし第４引数がOFFの場合は|があっても配列にしない
                    if (todoObj.value.indexOf('|') != -1 && args4 != "OFF") {
                      $gameVariables.setValue(args2, todoObj.value.split('|'));
                    } else {
                      $gameVariables.setValue(args2, todoObj.value);
                    }
                    break;
                  }
                }
                // 第４引数が設定されている場合は処理ステータス(true)を返却
                if (args3 != "") $gameVariables.setValue(args3, 1);
              })
              .catch((err) => {
                // 第４引数が設定されている場合は処理ステータス(false)を返却
                if (args3 != "") $gameVariables.setValue(args3, 9999);
                alert("通信に失敗しました");
              });

        }
  }

  const run = {
    rankDispMain(){
      // すでに表示されているシーン情報を削除
      addChildSceneList.forEach(element => SceneManager._scene.removeChild(element));
      addChildSceneList = [];

      var rankVal = 0;
      var roopCnt = 0;
      background2 = new PIXI.Graphics();
      ranktextRank = "";
      ranktextName = "";
      ranktextValue = "";
      recordScore = "";
      recordRank = "";
      rankPlusVal = 1;
      
      // スクロールポジション分表示を変更
      var positionTmp = scrollPosition;

      for(let todoObj of rankingData) {
        // 同値順位設定処理
        if (prevVal == todoObj.value) {
          rankPlusVal++;
        } else {
          rankVal += rankPlusVal;
          rankPlusVal = 1;
          prevVal = todoObj.value;
        }
        // 順位が未設定の場合のみ1を設定
        if (0 == rankVal) {
          rankVal = 1;
        }
        
        // スクロールポジション分表示を変更
        if (positionTmp < 1) {
          ranktextRank = ranktextRank.concat(rankVal + rankEndStr + "\n\n");
          ranktextName = ranktextName.concat(todoObj.name + " " + nameEndStr + "\n\n");
          ranktextValue = ranktextValue.concat(todoObj.value.toLocaleString() + " " + valueEndStr + "\n\n");
        } else {
          positionTmp--;
        }
        
        // 今回登録データかつIDが合致したデータを取得
        if (args2 == "ASK") {
          if (args1 == 'ON' && recordDispFlg && nowScore != "" && recordScore == "") {
            if (Number(nowScore) <= todoObj.value) {
              recordScore = Number(nowScore).toLocaleString();
              if (Number(nowScore) == todoObj.value) {
                recordRank = rankVal + rankEndStr;
              } else {
                if (rankVal == 1) {
                  recordRank = rankVal - 1 + rankEndStr;
                } else {
                  recordRank = rankVal + rankEndStr;
                }
              }
            }
          }
        } else {
          if (args1 == 'ON' && recordDispFlg && nowScore != "" && recordScore == "") {
            if (Number(nowScore) >= todoObj.value) {
              recordScore = Number(nowScore).toLocaleString();
              recordRank = rankVal + rankEndStr;
              if (Number(nowScore) == todoObj.value) {
                recordRank = rankVal + rankEndStr;
              } else {
                if (rankVal == 1) {
                  recordRank = rankVal - 1 + rankEndStr;
                } else {
                  recordRank = rankVal + rankEndStr;
                }
              }
            }
          }
        }

        if (strByteRankMax < String(rankVal + rankEndStr).bytes()){
          strByteRankMax = String(rankVal + rankEndStr).bytes();
        }
        if (strByteNameMax < (todoObj.name + " " + nameEndStr).bytes()){
          strByteNameMax = (todoObj.name + " " + nameEndStr).bytes();
        }
        if (strByteValueMax < (todoObj.value.toLocaleString() + " " + valueEndStr).bytes()){
          strByteValueMax = (todoObj.value.toLocaleString() + " " + valueEndStr).bytes();
        }
        roopCnt++;
      }
      // 最下位の場合
      if (args1 == 'ON' && recordDispFlg && nowScore != "" && recordScore == "") {
          recordScore = Number(nowScore).toLocaleString();
          recordRank = rankVal + 1 + rankEndStr;
      }
      // レコードランクが0の場合は1を設定
      if (recordRank == 0) {
        recordRank = 1;
      }
      // スクロール下限値を設定
      scrollLowerLimit = roopCnt - screenDispLimit;

      // 横軸の基本設定
      var xStart1 = widthDefault;
      var xStart2 = widthDefault + (strByteRankMax * rankingFontSize) + Number(rankingFontSize);
      var xStart3 = widthDefault + (strByteRankMax * rankingFontSize) + (strByteValueMax * rankingFontSize / 2) + (rankingFontSize * 2) + (rankingFontSize * 2);
      var widthData = widthDefault + (strByteRankMax * rankingFontSize) + (strByteValueMax * rankingFontSize) + (strByteNameMax * rankingFontSize);
      // 縦軸の基本設定
      var yStart = yDefault - Math.floor((rankingFontSize / 10));

      // ランキング背景
      background2.beginFill(bgRankingColor, 1);
      background2.alpha = bgRankingAlpha;
      background2.drawRect((Graphics.width / 3) + bgMoveX, bgMoveY, (widthData / 2) + bgSizeWidth, Graphics.height + bgSizeHeight); 
      background2.endFill();
      obj.addChildBg(background2);

      // 第２引数がONの場合は今回のスコアを上部に設定する
      if (args1 == 'ON' && recordDispFlg) {
        var recordTitleDisp = new PIXI.Text(recordTitle, {
          fontFamily: fontFamily,
          fontSize: Number(rankingTitleFontSize),
          fill: thisScoreTitleColor,
          align: 'center',
          fontWeight: 'bold',
        });
        var recordScoreTitleDisp = new PIXI.Text(recordScoreName, {
          fontFamily: fontFamily,
          fontSize: Number(rankingFontSize),
          fill: thisScoreSubTitleColor,
          align: 'center',
          fontWeight: 'bold',
        });
        var recordRankTitleDisp = new PIXI.Text(recordRankName, {
          fontFamily: fontFamily,
          fontSize: Number(rankingFontSize),
          fill: thisRankSubTitleColor,
          align: 'center',
          fontWeight: 'bold',
        });
        var recordScoreDisp = new PIXI.Text(recordScore + " " + valueEndStr, {
          fontFamily: fontFamily,
          fontSize: Number(rankingFontSize),
          fill: thisScoreColor,
          align: 'left',
        });
        var recordRankDisp = new PIXI.Text(recordRank, {
          fontFamily: fontFamily,
          fontSize: Number(rankingFontSize),
          fill: thisRankColor,
          align: 'left',
        });

      // 位置を設定
        var recordTitleLength = (ranktextTitle.bytes() - recordTitle.bytes()) * (rankingTitleFontSize / 3);
        recordTitleDisp.x = (Graphics.width / 3) + (widthData / 6) + recordTitleLength + thisScoreTitleMoveX;
        recordTitleDisp.y = yRecordDefault - 50 + thisScoreTitleMoveY;
        recordScoreTitleDisp.x = widthDefault - 10 + thisScoreSubTitleMoveX;
        recordScoreTitleDisp.y = yRecordDefault - 20 + thisScoreSubTitleMoveY;
        recordRankTitleDisp.x = widthDefault - 10 + thisRankSubTitleMoveX;
        recordRankTitleDisp.y = yRecordDefault + 10 + thisRankSubTitleMoveY;
        recordScoreDisp.x = widthDefault + (rankingFontSize * 3) + thisScoreMoveX;
        recordScoreDisp.y = yRecordDefault - 20 + thisScoreMoveY;
        recordRankDisp.x = widthDefault + (rankingFontSize * 3) + thisRankMoveX;
        recordRankDisp.y = yRecordDefault + 10 + thisRankMoveY;

        obj.addChild(recordTitleDisp);
        obj.addChild(recordScoreTitleDisp);
        obj.addChild(recordRankTitleDisp);
        obj.addChild(recordScoreDisp);
        obj.addChild(recordRankDisp);
        
      }

      // 背景白色設定
      for(var i = 0;i < roopCnt;i++) {
        // 順位の背景白設定
        var bgRank1 = new PIXI.Graphics();
        bgRank1.beginFill(rankBgColor, 1);
        bgRank1.alpha = rankBgAlpha;
        bgRank1.drawRect(xStart1 + rankMoveX,yStart - 5 + rankMoveY,strByteRankMax * rankingFontSize, rankingFontSize * 2);
        bgRank1.endFill();
        obj.addChild(bgRank1);

        // スコアの背景白設定
        var bgRank2 = new PIXI.Graphics();
        bgRank2.beginFill(rankBgColor, 1);
        bgRank2.alpha = rankBgAlpha;
        bgRank2.drawRect(xStart2 + scoreMoveX,yStart - 5 + scoreMoveY,(strByteValueMax * rankingFontSize / 2) + (rankingFontSize * 2), rankingFontSize * 2);
        bgRank2.endFill();
        obj.addChild(bgRank2);

        // 名前の背景白設定
        var bgRank3 = new PIXI.Graphics();
        bgRank3.beginFill(rankBgColor, 1);
        bgRank3.alpha = rankBgAlpha;
        bgRank3.drawRect(xStart3 + nameMoveX,yStart - 5 + nameMoveY,(strByteNameMax * rankingFontSize / 2) + (rankingFontSize * 2), rankingFontSize * 2);
        bgRank3.endFill();
        obj.addChild(bgRank3);

        yStart += (rankingFontSize * 2) + 6;
      }

      // タイトルの書式設定
      var ranktextTitleDisp = new PIXI.Text(ranktextTitle, {
        fontFamily: fontFamily,
        fontSize: Number(rankingTitleFontSize),
        fill: rankingTitleColor,
        align: 'center',
        fontWeight: 'bold',
      });
      // ランクの書式設定
      var ranktextRankDisp = new PIXI.Text(ranktextRank, {
        fontFamily: fontFamily,
        fontSize: Number(rankingFontSize),
        fill: rankColor,
        align: 'right',
      });
      // 名前の書式設定
      var ranktextNameDisp = new PIXI.Text(ranktextName, {
        fontFamily: fontFamily,
        fontSize: Number(rankingFontSize),
        fill: scoreColor,
        align: 'left',
      });
      // 値の書式設定
      var ranktextValueDisp = new PIXI.Text(ranktextValue, {
        fontFamily: fontFamily,
        fontSize: Number(rankingFontSize),
        fill: nameColor,
        align: 'right',
      });

      // 位置を設定
      ranktextTitleDisp.x = (Graphics.width / 3) + (widthData / 6) + rankingTitleMoveX;
      ranktextTitleDisp.y = yDefault - 50 + rankingTitleMoveY;
      ranktextRankDisp.x = widthDefault + 10 + rankMoveX;
      ranktextRankDisp.y = yDefault + rankMoveY;
      ranktextValueDisp.x = widthDefault + (strByteRankMax * rankingFontSize) + (rankingFontSize * 2) + scoreMoveX;
      ranktextValueDisp.y = yDefault + scoreMoveY;
      ranktextNameDisp.x = widthDefault + (strByteRankMax * rankingFontSize) + (strByteValueMax * rankingFontSize / 2) + (rankingFontSize * 2) + (rankingFontSize * 3) + nameMoveX;
      ranktextNameDisp.y = yDefault + nameMoveY;

      obj.addChild(ranktextTitleDisp);
      obj.addChild(ranktextRankDisp);
      obj.addChild(ranktextValueDisp);
      obj.addChild(ranktextNameDisp);

      // バツボタン作成
      var batsuButton = new PIXI.Graphics();
      batsuButton.beginFill(batsuButtonBgColor, 1);
      batsuButton.alpha = batsuButtonBgAlpha;
      batsuButton.drawRect((Graphics.width / 3) + (widthData / 2) - 35 + bgMoveX + bgSizeWidth,5 + bgMoveY,30, 30); 
      batsuButton.interactive = true;
      batsuButton.buttonMode = true;
      var batsuButtonText = new PIXI.Text(batsuString, {
        fontFamily: fontFamily,
        fontSize: 26,
        fill: batsuButtonFontColor,
        align: 'center',
        fontWeight: 'bold',
      });
      batsuButtonText.x = (Graphics.width / 3) + (widthData / 2) - 31 + bgMoveX + bgSizeWidth
      batsuButtonText.y = 5 + bgMoveY
      batsuButton.on('pointerupoutside', function(e) {
        if (rankingDispOutputFlg) {
          if (batsuFlg) {
            rankHidden();
          }
        }
      });
      obj.addChildBg(batsuButton);
      obj.addChildBg(batsuButtonText);
      // 第４引数が設定されている場合は処理ステータス(true)を返却
      if (args3 != "") $gameVariables.setValue(args3, 1);
    }

  }

  // json形式を小さい順に並べ替え
  function compareAsk( a, b ) {
    var r = 0;
    if( a.value < b.value ){ r = -1; }
    else if( a.value > b.value ){ r = 1; }

    return r;
  }
  // json形式を大きさ順に並べ替え
  function compareDesk( a, b ) {
    var r = 0;
    if( a.value < b.value ){ r = -1; }
    else if( a.value > b.value ){ r = 1; }

    return ( -1 * r );
  }

  // バイト数の長さを取得
  String.prototype.bytes = function () {
    var length = 0;
    for (var i = 0; i < this.length; i++) {
      var c = this.charCodeAt(i);
      if ((c >= 0x0 && c < 0x81) || (c === 0xf8f0) || (c >= 0xff61 && c < 0xffa0) || (c >= 0xf8f1 && c < 0xf8f4)) {
        length += 1;
      } else {
        length += 2;
      }
    }
    return length;
  };
  
  // 何かしらのキーを押下するとランキングを非表示
  document.addEventListener('keypress', function(event) {
    if (rankingDispOutputFlg) {
      rankHidden();
    }
  });

  // ホイールでランキングをスクロール
  document.addEventListener('wheel', function(event) {
    if (rankingDispOutputFlg) {
      if (event.deltaY > 0) {
        scrollPosition++;
      } else if (event.deltaY < 0) {
        scrollPosition--;
      }
      if (scrollPosition < 0) {
        scrollPosition = 0;
      } else if (scrollPosition > scrollLowerLimit) {
        scrollPosition--;
      }
      run.rankDispMain();
    }
  });
  
  // ランキングを非表示
  function rankHidden(){
    Input._currentState['ok'] = false;
    Input._currentState['cancel'] = false;
    addChildSceneList.forEach(element => SceneManager._scene.removeChild(element));
    addChildSceneList = [];
    addChildSceneListBgOnly.forEach(element => SceneManager._scene.removeChild(element));
    addChildSceneListBgOnly = [];
    const _Game_Interpreter_prototype = Game_Interpreter.prototype;
    _Game_Interpreter_prototype._waitCount = 60;
    // ランキング終了後にメニュー禁止を解除するかしないか
    if (rankingMenuDispFlg == "true") {
      // フラグがONの場合のみ解除する
      $gameSystem.enableMenu();
    }
    rankingDispFlg = false;
    rankingDispOutputFlg = false;
    batsuFlg = false;
  }

  // ランキング表示中の移動やメニュー表示を禁止
  const _Ranking_Game_Player_executeMove = Game_Player.prototype.executeMove;
  Game_Player.prototype.executeMove = function(e) {
      if(!rankingDispFlg) {
        _Ranking_Game_Player_executeMove.call(this, e);
      }
  };
  const _Ranking_Game_Player_triggerAction = Game_Player.prototype.triggerAction;
  Game_Player.prototype.triggerAction = function() {
      if(!rankingDispFlg) {
        _Ranking_Game_Player_triggerAction.call(this);
      }
      return false;
  };
  const _Ranking_Game_Player_checkEventTriggerHere = Game_Player.prototype.checkEventTriggerHere;
  Game_Player.prototype.checkEventTriggerHere = function(e) {
      if(!rankingDispFlg){
        _Ranking_Game_Player_checkEventTriggerHere.apply(this, arguments);
      }
      return false;
  };
  const _Ranking_Game_Player_checkEventTriggerThere = Game_Player.prototype.checkEventTriggerThere;
  Game_Player.prototype.checkEventTriggerThere = function(e) {
      if(!rankingDispFlg){
        _Ranking_Game_Player_checkEventTriggerThere.apply(this, arguments);
      }
      return false;
  };
    
})();