//=============================================================================
// SoR_BattleBGMManager_MZ.js
// SoR License (C) 2020 蒼竜, REQUIRED User Registration on Dragon Cave
// http://dragonflare.blue/dcave/license.php
// ----------------------------------------------------------------------------
// Latest version v1.02 (2021/05/26)
//=============================================================================
/*:ja
@plugindesc ＜戦闘BGM制御＞ v1.02
@author 蒼竜
@target MZ
@orderAfter SoR_BattleTeminationRefactor_MZ.js
@url http://dragonflare.blue/dcave/
@help 戦闘BGMなど、ゲーム進行に応じた頻繁な変更が
想定されるBGM設定を、プラグインパラメータでデータベース化し、
デフォルトのエディタによる逐次ファイル名指定をせずに
切り替えが行えるようにします。

※導入すると、データベース上の関連BGM/ME設定は無視されます。
※通常のイベントコマンドによる一時的な変更は可能です。
※勝利・敗北時設定(BGM)を使用する場合には別途、
57.「戦闘終了処理ルーチン細分化」
(SoR_BattleTeminationRefactor_MZ)が必要となります。

@param ---BGM設定---
@param NormalBattleBGMs
@default []
@desc 使用する通常戦闘BGMのリスト
@type struct<BGMDATA>[]
@param BossBattleBGMs
@default []
@desc 使用するボス等イベント戦闘BGMのリスト
@type struct<BGMDATA>[]
@param BattleVictoryBGMs
@default []
@desc   
@type struct<BGMDATA>[]
@param BattleVictoryMEs
@default []
@desc 使用する戦闘勝利MEのリスト
@type struct<MEDATA>[]
@param BattleDefeatedBGMs
@default []
@desc 使用する戦闘敗北BGMのリスト
@type struct<BGMDATA>[]
@param BattleDefeatedMEs
@default []
@desc 使用する戦闘敗北MEのリスト
@type struct<MEDATA>[]
@param ---初期BGM設定---
@param Default_NormalBattleBGM
@desc 初期通常戦闘BGM指定(0でBGMなし)
@type number
@default 1
@min 0
@param Default_BossBattleBGM
@desc 初期ボス等イベント戦闘BGM指定(0でBGMなし)
@type number
@default 1
@min 0
@param Default_BattleVictoryBGM
@desc 初期戦闘勝利BGM指定(0でBGMなし)
@type number
@default 0
@min 0
@param Default_BattleVictoryME
@desc 初期戦闘勝利ME指定(0でMEなし)
@type number
@default 1
@min 0
@param Default_BattleDefeatedBGM
@desc 初期戦闘敗北BGM指定(0でBGMなし)
@type number
@default 0
@min 0
@param Default_BattleDefeatedME
@desc 初期戦闘敗北ME指定(0でMEなし)
@type number
@default 1
@min 0
@param RestoreBGM_AfterBattleEnd
@desc 戦闘からマップへ戻る時、マップBGM再生再開をマップ復帰完了まで遅らせます (default: true)
@default true
@type boolean

@command SwitchBattleBGM
@text 状況パターン変更[戦闘BGM制御]
@desc 戦闘BGMを、通常/ボス戦用で切り替えます。
@arg situation
@type select
@option 通常戦闘
@value 0
@option ボス戦
@value 1
@default 0
@command ChangeNormalBattleBGM
@text 通常戦闘BGM変更[戦闘BGM制御]
@desc プラグインパラメータのIDを指定して、通常戦闘BGMを変更します
@arg bgmID
@type number
@default 1
@min 0
@command ChangeBossBattleBGM
@text ボス戦闘BGM変更[戦闘BGM制御]
@desc プラグインパラメータのIDを指定して、ボス・イベント戦闘BGMを変更します
@arg bgmID
@type number
@default 1
@min 0
@command ChangeBattleVictorySound
@text 戦闘勝利サウンド変更[戦闘BGM制御]
@desc プラグインパラメータで設定したIDを指定して、戦闘勝利時のサウンド(ME/BGM)を変更します
@arg sound
@type select
@option BGM
@value 0
@option ME
@value 1
@desc 再生に使用するサウンド種類を指定します。指定しなかった方のサウンドは自動で無効化(=0)されます。
@default 1
@arg bgmID
@type number
@default 1
@desc 指定したサウンド種類のプラグインパラメータ(サウンド配列)で設定したサウンドのID
@min 0
@command ChangeBattleDefeatedSound
@text 戦闘敗北サウンド変更[戦闘BGM制御]
@desc プラグインパラメータで設定したIDを指定して、戦闘敗北時のサウンド(ME/BGM)を変更します
@arg sound
@type select
@option BGM
@value 0
@option ME
@value 1
@default 1
@desc 再生に使用するサウンド種類を指定します。指定しなかった方のサウンドは自動で無効化(=0)されます。
@arg bgmID
@type number
@default 1
@desc 指定したサウンド種類のプラグインパラメータ(サウンド配列)で設定したサウンドのID
@min 0

@command ContinueMapBGM_Battle
@text マップBGM継続[戦闘BGM制御]
@desc 特殊なゲーム状況下などで、マップBGMを戦闘BGMに代用する設定を行います。
@arg flag
@type boolean
@default true
@desc true: 戦闘BGMを再生しない(設定に依らずマップBGMを継続)  false: 通常に戻す
@command ContinueBattleBGM_Result
@text 戦闘BGM継続[戦闘BGM制御]
@desc 特殊なゲーム状況下などで、戦闘BGMを勝利サウンドに代用する設定を行います。
@arg flag
@type boolean
@default true
@desc true: 勝利サウンドを再生しない(設定に依らず戦闘BGMを継続)  false: 通常に戻す
*/
/*~struct~BGMDATA:
@type string
@param name
@dir audio/bgm/
@type file
@desc BGM
@param volume
@desc 音量 [0...100]
@type number
@default 100
@min 0
@max 100
@param pitch
@desc ピッチ [50...150]
@type number
@default 100
@min 50
@max 150
@param pan
@desc パン(位相) [-50...50]
@type number
@default 0
@min -50
@max 50
*/
/*~struct~MEDATA:
@type string
@param name
@dir audio/me/
@type file
@desc ME
@param volume
@desc 音量 [0...100]
@type number
@default 100
@min 0
@max 100
@param pitch
@desc ピッチ [50...150]
@type number
@default 100
@min 50
@max 150
@param pan
@desc パン(位相) [-50...50]
@type number
@default 0
@min -50
@max 50
*/
/*:
@plugindesc <Battle BGM Manager> v1.02
@author Soryu
@target MZ
@orderAfter SoR_BattleTeminationRefactor_MZ.js
@url http://dragonflare.blue/dcave/index_e.php
@help This plugin creates a database of Battle BGMs 
which are changed frequently during games to switch them easily
without designating the file name in the default event command.

!!Default Battle BGMs and MEs are ignored.
!!Temporal BGM changes by the default event command is still possible.
!!To configure battle victory and defeated BGM/SE settings,
57. SoR_BattleTeminationRefactor_MZ is required.
@param ---BGM Setting---
@param NormalBattleBGMs
@default []
@desc A list of Normal Battle BGMs
@type struct<BGMDATAE>[]
@param BossBattleBGMs
@default []
@desc A list of Event Battle BGMs including boss battles
@type struct<BGMDATAE>[]
@param BattleVictoryBGMs
@default []
@desc A list of Battle Victory BGMs
@type struct<BGMDATAE>[]
@param BattleVictoryMEs
@default []
@desc A list of Battle Victory MEs
@type struct<MEDATAE>[]
@param BattleDefeatedBGMs
@default []
@desc A list of Battle Defeated BGMs
@type struct<BGMDATAE>[]
@param BattleDefeatedMEs
@default []
@desc A list of Battle Defeated MEs
@type struct<MEDATAE>[]

@param ---Initail BGM Setting---
@param Default_NormalBattleBGM
@desc ID of Initial Normal Battle BGM (Set 0 for No BGM)
@type number
@default 1
@min 0
@param Default_BossBattleBGM
@desc ID of Initial Event Battle BGM (Set 0 for No BGM)
@type number
@default 1
@min 0
@param Default_BattleVictoryBGM
@desc ID of Initial Victory BGM (Set 0 for No BGM)
@type number
@default 0
@min 0
@param Default_BattleVictoryME
@desc ID of Initial Victory ME (Set 0 for No ME)
@type number
@default 1
@min 0
@param Default_BattleDefeatedBGM
@desc ID of Initial Defeated BGM (Set 0 for No BGM)
@type number
@default 0
@min 0
@param Default_BattleDefeatedME
@desc ID of Initial Defeated ME (Set 0 for No ME)
@type number
@default 1
@min 0
@param RestoreBGM_AfterBattleEnd
@desc If true, the timing of restarting map BGM when the battle ends is delayed. (default: true)
@default true
@type boolean

@command SwitchBattleBGM
@text Change Battle Situation [Battle BGM Manager]
@desc Switch Battle BGM between Normal and Event battles.
@arg situation
@type select
@option Normal Battles
@value 0
@option Event (Boss) Battles
@value 1
@default 0
@command ChangeNormalBattleBGM
@text Change Normal Battle BGM [Battle BGM Manager]
@desc Set ID in the Normal Battle BGM database to change the battle BGM
@arg bgmID
@type number
@default 1
@min 0
@command ChangeBossBattleBGM
@text Change Event Battle BGM [Battle BGM Manager]
@desc Set ID in the Event Battle BGM database to change the event (boss) battle BGM
@arg bgmID
@type number
@default 1
@min 0
@command ChangeBattleVictorySound
@text Change Victory Sound [Battle BGM Manager]
@desc Set ID in the Victory sound database to change the victory BGM or ME.
@arg sound
@type select
@option BGM
@value 0
@option ME
@value 1
@desc Sound format you use. The item you does not use is automatically disabled (set 0).
@default 1
@arg bgmID
@type number
@default 1
@desc Sound ID in the database BGM/ME (which you chose)
@min 0
@command ChangeBattleDefeatedSound
@text Change Defeated Sound [Battle BGM Manager]
@desc Set ID in the Defeated sound database to change the defeated BGM or ME.
@arg sound
@type select
@option BGM
@value 0
@option ME
@value 1
@default 1
@desc Sound format you use. The item you does not use is automatically disabled (set 0).
@arg bgmID
@type number
@default 1
@desc Sound ID in the database BGM/ME (which you chose)
@min 0

@command ContinueMapBGM_Battle
@text Continue Map BGM for Battle [Battle BGM Manager]
@desc Temporary Replace battle BGM with map BGM under special situation in the game.
@arg flag
@type boolean
@default true
@desc true: Replace battle BGM with map BGM (continue map BGM), false: revert the change
@command ContinueBattleBGM_Result
@text Continue Battle BGM for Result [Battle BGM Manager]
@desc Temporary Replace Victory sound with battle BGM under special situation in the game.
@arg flag
@type boolean
@default true
@desc true: Replace Victory sound with BGM (continue battle BGM), false: revert the change
*/
/*~struct~BGMDATAE:
@type string
@param name
@dir audio/bgm/
@type file
@desc BGM
@param volume
@desc Volume of BGM [0...100]
@type number
@default 100
@min 0
@max 100
@param pitch
@desc Pitch of BGM [50...150]
@type number
@default 100
@min 50
@max 150
@param pan
@desc Pan of BGM [-50...50]
@type number
@default 0
@min -50
@max 50
*/
/*~struct~MEDATAE:
@type string
@param name
@dir audio/me/
@type file
@desc ME
@param volume
@desc Volume of ME [0...100]
@type number
@default 100
@min 0
@max 100
@param pitch
@desc Pitch of ME [50...150]
@type number
@default 100
@min 50
@max 150
@param pan
@desc Pan of ME [-50...50]
@type number
@default 0
@min -50
@max 50
*/

(function() {
const pluginName = "SoR_BattleBGMManager_MZ";
const Param = PluginManager.parameters(pluginName);

const NormalBattleBGMs = convertJsonParams(Param['NormalBattleBGMs']) || '';
const BossBattleBGMs = convertJsonParams(Param['BossBattleBGMs']) || '';
const BattleVictoryBGMs = convertJsonParams(Param['BattleVictoryBGMs']) || '';
const BattleVictoryMEs = convertJsonParams(Param['BattleVictoryMEs']) || '';
const BattleDefeatedBGMs = convertJsonParams(Param['BattleDefeatedBGMs']) || '';
const BattleDefeatedMEs = convertJsonParams(Param['BattleDefeatedMEs']) || '';

const Default_NormalBattleBGM = Number(Param['Default_NormalBattleBGM']) || 0; 
const Default_BossBattleBGM = Number(Param['Default_BossBattleBGM']) || 0; 
const Default_BattleVictoryBGM = Number(Param['Default_BattleVictoryBGM']) || 0; 
const Default_BattleVictoryME = Number(Param['Default_BattleVictoryME']) || 0; 
const Default_BattleDefeatedBGM = Number(Param['Default_BattleDefeatedBGM']) || 0; 
const Default_BattleDefeatedME = Number(Param['Default_BattleDefeatedME']) || 0; 

const RestoreBGM_AfterBattleEnd = Boolean(Param['RestoreBGM_AfterBattleEnd'] === 'true' || false);

function convertJsonParams(param) {
    if (param == undefined) return [];
    let arr = [];
        JSON.parse(param).map(function(param) {
        const obj = JSON.parse(param);
            obj.volume = Number(obj.volume);
            obj.pan = Number(obj.pan);
            obj.pitch = Number(obj.pitch);
            arr.push(obj);
        });
    return arr;
}

NormalBattleBGMs.unshift({name: "", volume: 100, pitch: 100, pan: 0});
BossBattleBGMs.unshift({name: "", volume: 100, pitch: 100, pan: 0});
BattleVictoryBGMs.unshift({name: "", volume: 100, pitch: 100, pan: 0});
BattleVictoryMEs.unshift({name: "", volume: 100, pitch: 100, pan: 0});
BattleDefeatedBGMs.unshift({name: "", volume: 100, pitch: 100, pan: 0});
BattleDefeatedMEs.unshift({name: "", volume: 100, pitch: 100, pan: 0});

PluginManager.registerCommand(pluginName, "SwitchBattleBGM", args => { 
    $gameSystem._battleBGMSituation = Number(args.situation);
    if($gameSystem._battleBGMSituation==0)  $gameSystem._battleBgm = NormalBattleBGMs[$gameSystem.battleBGMIds[0]];
    else if($gameSystem._battleBGMSituation==1) $gameSystem._battleBgm = BossBattleBGMs[$gameSystem.battleBGMIds[1]];
});
PluginManager.registerCommand(pluginName, "ChangeNormalBattleBGM", args => { 
    $gameSystem.battleBGMIds[0] = Number(args.bgmID);
    if($gameSystem._battleBGMSituation==0) $gameSystem._battleBgm = NormalBattleBGMs[$gameSystem.battleBGMIds[0]];
});
PluginManager.registerCommand(pluginName, "ChangeBossBattleBGM", args => { 
    $gameSystem.battleBGMIds[1] = Number(args.bgmID);
    if($gameSystem._battleBGMSituation==1) $gameSystem._battleBgm = BossBattleBGMs[$gameSystem.battleBGMIds[1]];
});

PluginManager.registerCommand(pluginName, "ChangeBattleVictorySound", args => { 
    const id = Number(args.sound);
    $gameSystem.battleBGMIds[2+id] = Number(args.bgmID);
    $gameSystem.battleBGMIds[2+(1-id)] = 0;
    $gameSystem.updateVictorySound();
});
PluginManager.registerCommand(pluginName, "ChangeBattleDefeatedSound", args => { 
    const id = Number(args.sound);
    $gameSystem.battleBGMIds[4+id] = Number(args.bgmID);
    $gameSystem.battleBGMIds[4+(1-id)] = 0;
    $gameSystem.updateDefeatedSound();
});

//v1.02
PluginManager.registerCommand(pluginName, "ContinueMapBGM_Battle", args => { 
    $gameSystem.mapBGMcontinue_bat = Boolean(args.flag === 'true' || false);
});
PluginManager.registerCommand(pluginName, "ContinueBattleBGM_Result", args => { 
    $gameSystem.battleBGMcontinue_res = Boolean(args.flag === 'true' || false);
});

////////////////////////////////////////////////////////////////
const SoR_BBM_GS_initialize = Game_System.prototype.initialize;
Game_System.prototype.initialize = function() {
    SoR_BBM_GS_initialize.call(this);
    this.setInitialBGMs();
}

//overwrite BGM settings
Game_System.prototype.setInitialBGMs = function() {
    if(!this._battleBGMSituation) this._battleBGMSituation = 0;
    if(!this.battleBGMIds) this.battleBGMIds = [Default_NormalBattleBGM, Default_BossBattleBGM, Default_BattleVictoryBGM, Default_BattleVictoryME, Default_BattleDefeatedBGM, Default_BattleDefeatedME];
    if(!this._battleBgm){
        if(this._battleBGMSituation==0) this._battleBgm = NormalBattleBGMs[this.battleBGMIds[0]];
        else if(this._battleBGMSituation==1) this._battleBgm = BossBattleBGMs[this.battleBGMIds[1]];
    }
    if(!this._victoryBgm) this._victoryBgm = BattleVictoryBGMs[this.battleBGMIds[2]];
    if(this._victoryMe != null) this._victoryMe = BattleVictoryMEs[this.battleBGMIds[3]];

    if(!this._defeatBgm) this._defeatBgm = BattleDefeatedBGMs[this.battleBGMIds[4]];
    if(this._defeatMe != null) this._defeatMe = BattleDefeatedMEs[this.battleBGMIds[5]];

    this.mapBGMcontinue_bat = false;
    this.battleBGMcontinue_res = false;
}

Game_System.prototype.resetInitialBGMs = function() {
    this._battleBGMSituation = 0;
    this.battleBGMIds = [Default_NormalBattleBGM, Default_BossBattleBGM, Default_BattleVictoryBGM, Default_BattleVictoryME, Default_BattleDefeatedBGM, Default_BattleDefeatedME];

    if(this._battleBGMSituation==0) this._battleBgm = NormalBattleBGMs[this.battleBGMIds[0]];
    else if(this._battleBGMSituation==1) this._battleBgm = BossBattleBGMs[this.battleBGMIds[1]];

    this._victoryBgm = BattleVictoryBGMs[this.battleBGMIds[2]];
    this._victoryMe = BattleVictoryMEs[this.battleBGMIds[3]];

    this._defeatBgm = BattleDefeatedBGMs[this.battleBGMIds[4]];
    this._defeatMe = BattleDefeatedMEs[this.battleBGMIds[5]];

    this.mapBGMcontinue_bat = false;
    this.battleBGMcontinue_res = false;
}

Game_System.prototype.updateVictorySound = function() {
    this._victoryBgm = BattleVictoryBGMs[this.battleBGMIds[2]];
    this._victoryMe = BattleVictoryMEs[this.battleBGMIds[3]];
}
Game_System.prototype.updateDefeatedSound = function() {
    this._defeatBgm = BattleDefeatedBGMs[this.battleBGMIds[4]];
    this._defeatMe = BattleDefeatedMEs[this.battleBGMIds[5]];
}


const SoR_BBM_SM_stopAudioOnBattleStart = Scene_Map.prototype.stopAudioOnBattleStart;
Scene_Map.prototype.stopAudioOnBattleStart = function() {
    if($gameSystem.mapBGMcontinue_bat) return;
    SoR_BBM_SM_stopAudioOnBattleStart.call(this);
}

const SoR_BBM_BM_playBattleBgm = BattleManager.playBattleBgm;
BattleManager.playBattleBgm = function() {
    if(typeof $gameSystem.mapBGMcontinue_bat === "undefined"){
        $gameSystem.mapBGMcontinue_bat = false;
        $gameSystem.battleBGMcontinue_res = false;
    }
    if($gameSystem.mapBGMcontinue_bat) return;

    SoR_BBM_BM_playBattleBgm.call(this);
}



if(PluginManager._scripts.includes("SoR_BattleTeminationRefactor_MZ")){
    const SoR_BBM_BM_playVictoryMe = BattleManager.playVictoryMe;
    BattleManager.playVictoryMe = function() {
        if(!$gameSystem.battleBGMcontinue_res){
            if($gameSystem.battleBGMIds[2]<=0) SoR_BBM_BM_playVictoryMe.call(this);
            else AudioManager.playBgm($gameSystem._victoryBgm);
        }
    }

    //me
    const SoR_BBM_BM_ProcessWin_AfterSounds = BattleManager.ProcessWin_AfterSounds;
    BattleManager.ProcessWin_AfterSounds = function(){
        if($gameSystem.battleBGMIds[2]<=0) SoR_BBM_BM_ProcessWin_AfterSounds.call(this);
    }

    const SoR_BBM_BM_playDefeatMe = BattleManager.playDefeatMe;
    BattleManager.playDefeatMe = function() {
        if($gameSystem.battleBGMIds[4]<=0) SoR_BBM_BM_playDefeatMe.call(this);//regular me
        else AudioManager.playBgm($gameSystem._defeatBgm);
    }

    const SoR_BBM_BM_ProcessDefeated_AfterSounds_gameover = BattleManager.ProcessDefeated_AfterSounds_gameover;
    BattleManager.ProcessDefeated_AfterSounds_gameover = function(){
        if($gameSystem.battleBGMIds[4]<=0) SoR_BBM_BM_ProcessDefeated_AfterSounds_gameover.call(this);//regular me
    }

    //bgm
    const SoR_BBM_BM_CreanUp_BattleTerminate = BattleManager.CreanUp_BattleTerminate;
    BattleManager.CreanUp_BattleTerminate = function(){
        //alternative bgm
        BattleManager.CreanUp_BGMTransition();
        SoR_BBM_BM_CreanUp_BattleTerminate.call(this);
    }

    
    BattleManager.CreanUp_BGMTransition = function(){
        if($gameSystem.mapBGMcontinue_bat) return;
        if(!RestoreBGM_AfterBattleEnd){
            if($gameSystem.battleBGMIds[2]>0) this.replayBgmAndBgs();
            else if(this._canLose && $gameSystem.battleBGMIds[4]>0) this.replayBgmAndBgs();
        }
    }

    const SoR_BBM_BM_ProcessDefeated_AfterSounds_canLose = BattleManager.ProcessDefeated_AfterSounds_canLose;
    BattleManager.ProcessDefeated_AfterSounds_canLose = function(){
        if(RestoreBGM_AfterBattleEnd) return;
        SoR_BBM_BM_ProcessDefeated_AfterSounds_canLose.call(this);
    }

    const SoR_BBM_BM_Process_EndBattleScene = BattleManager.Process_EndBattleScene;
    BattleManager.Process_EndBattleScene = function() { //deadend
        if(RestoreBGM_AfterBattleEnd) $gameTemp.restoreMapBGM_rightAfterBattles = true;
        SoR_BBM_BM_Process_EndBattleScene.call(this);
    }

    const SoR_BBM_SM_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        SoR_BBM_SM_start.call(this);

        if($gameTemp.restoreMapBGM_rightAfterBattles == true){
            BattleManager.replayBgmAndBgs();
            $gameTemp.restoreMapBGM_rightAfterBattles = false;
        }
    }

    const SoR_BBM_AM_onTransferEnd = AudioManager.saveBgm;
    AudioManager.saveBgm = function() {
        if($gameTemp.restoreMapBGM_rightAfterBattles == true) return BattleManager._mapBgm;
        return SoR_BBM_AM_onTransferEnd.call(this);
    }

}

})();