/*:
 * @target MZ
 * @plugindesc ステート発動時に、指定した味方が指定したステートにかかっている場合にスイッチをオンにするプラグインです。必要な味方の数も指定可能です。
 * @author スタジオVR
 *
 * @param settingsList
 * @text 設定リスト
 * @desc スイッチをオンにするための設定リスト。それぞれのステートとスイッチのIDを指定してください。
 * @type struct<StateSwitchSettings>[]
 * @default []
 *
 * @param requiredAllies
 * @text 必要な味方数(Global設定)
 * @desc 指定したステートにかかっている味方の数がこの数以上になった場合にスイッチがオンになります。
 * @type number
 * @min 1
 * @default 2
 *
 * @help StateSwitch.js
 *
 * このプラグインを使うと、ステートにかかっている場合に指定したスイッチをオンにすることができます。
 * ステートIDとスイッチIDは、プラグインパラメータの設定リストで指定してください。
 *
 * 必要な味方数はプラグインパラメータで指定できます。指定した数以上の味方がステートにかかっている場合、スイッチがオンになります。
 */

/*~struct~StateSwitchSettings:ja
 * @param stateIds
 * @text ステートID
 * @desc モニターするステートのIDを指定してください。複数指定する場合はカンマで区切ってください。
 * @type string
 * @default 1
 *
 * @param switchId
 * @text スイッチID
 * @desc ステートが指定された場合にオンにするスイッチIDを指定してください。
 * @type number
 * @min 1
 * @default 1
 *
 * @param requiredAllies
 * @text 必要な味方数
 * @desc 指定したステートにかかっている味方の数がこの数以上になった場合にスイッチがオンになります。空の場合はGlobalが反映されます。
 * @type number
 * @min 1
 * @default 2
 */

(() => {
    const pluginName = 'StateSwitch';
    const parameters = PluginManager.parameters(pluginName);
    const settingsList = JSON.parse(parameters['settingsList'] || '[]').map(setting => {
        const parsedSetting = JSON.parse(setting);
        parsedSetting.stateIds = parsedSetting.stateIds.split(',').map(Number);
        return parsedSetting;
    });
    const requiredAllies = Number(parameters['requiredAllies'] || 2);

    const _Game_Battler_addState = Game_Battler.prototype.addState;
    Game_Battler.prototype.addState = function (stateId) {
        _Game_Battler_addState.call(this, stateId);
        if (this.isActor()) {
            settingsList.forEach(setting => {
                const stateIds = setting.stateIds;
                if (stateIds.includes(stateId)) {
                    const alliesWithState = $gameParty.battleMembers().filter(member => {
                        return stateIds.some(id => member.isStateAffected(id));
                    });
                    if (alliesWithState.length >= setting.requiredAllies) {
                        $gameSwitches.setValue(Number(setting.switchId), true);
                    }
                }
            });
        }
    };

    const _Game_Battler_removeState = Game_Battler.prototype.removeState;
    Game_Battler.prototype.removeState = function (stateId) {
        _Game_Battler_removeState.call(this, stateId);
        if (this.isActor()) {
            settingsList.forEach(setting => {
                const stateIds = setting.stateIds;
                if (stateIds.includes(stateId)) {
                    const alliesWithState = $gameParty.battleMembers().filter(member => {
                        return stateIds.some(id => member.isStateAffected(id));
                    });
                    if (alliesWithState.length < setting.requiredAllies) {
                        const otherSettings = settingsList.filter(s => s !== setting);
                        const alliesWithOtherSettings = otherSettings.filter(otherSetting => {
                            return otherSetting.stateIds.some(id => this.isStateAffected(id));
                        }).reduce((allies, otherSetting) => {
                            return allies.concat($gameParty.battleMembers().filter(member => {
                                return otherSetting.stateIds.some(id => member.isStateAffected(id));
                            }));
                        }, []);
                        if (alliesWithOtherSettings.length < requiredAllies) {
                            $gameSwitches.setValue(Number(setting.switchId), false);
                        }
                    }
                }
            });
        }
    };
})();
