//=============================================================================
// Svr_BattlePowerScaling.js
//=============================================================================
/*:ja
 * @target MZ
 * @plugindesc v2.0 戦闘ターン経過による強化値加算プラグイン（拡張版）
 * @author スタジオVR
 * 
 * @param PowerUpSettings
 * @text 強化設定
 * @type struct<PowerUpRule>[]
 * @desc 敵グループごとの戦闘力上昇設定
 * @default []
 * 
 * @param AllyPowerRateVariableId
 * @text 味方強化値変数ID
 * @type variable
 * @desc 味方の現在の強化値を格納する変数ID
 * @default 1
 * 
 * @param EnemyPowerRateVariableId
 * @text 敵強化値変数ID
 * @type variable
 * @desc 敵の現在の強化値を格納する変数ID
 * @default 2
 * 
 * @param PopupSettings
 * @text ポップアップ設定
 * @type struct<PopupStyle>
 * @desc ポップアップの表示設定
 * @default {"textColor":"255,255,0","backgroundColor":"0,0,128","fontSize":"28","duration":"60","padding":"8","opacity":"192","xPosition":"center","yPosition":"middle","xOffset":"0","yOffset":"0"}
 * 
 * @param DespairSettings
 * @text ブーストフェイズ設定
 * @type struct<DespairMode>
 * @desc ブーストフェイズ演出の設定
 * @default {"modeName":"ブーストフェイズ","modeMessage":"限界突破...戦闘力が急上昇！","textColor":"255,0,0","effectDuration":"120","flashStrength":"160","flashColor":"255,0,0","shakeStrength":"5","useSound":"true","soundName":"Bell3","soundPitch":"70","soundVolume":"90","xPosition":"center","yPosition":"middle","xOffset":"0","yOffset":"0"}
 * 
 * @param CountdownSettings
 * @text カウントダウン設定
 * @type struct<CountdownStyle>
 * @desc カウントダウン表示設定
 * @default {"enabled":"true","startCount":"3","textColor":"255,128,128","warningColor":"255,0,0","fontSize":"24","soundEnabled":"true","soundName":"Decision2","warningName":"Bell2","soundVolume":"70","xPosition":"center","yPosition":"middle","xOffset":"0","yOffset":"0"}
 * 
 * @param EnableDebug
 * @text デバッグモード
 * @type boolean
 * @desc デバッグログを出力するかどうか
 * @default true
 */

/*~struct~PowerUpRule:ja
 * @param groupIdStart
 * @text グループID開始
 * @type number
 * @min 1
 * @max 999
 * @default 1
 * 
 * @param groupIdEnd
 * @text グループID終了
 * @type number
 * @min 1
 * @max 999
 * @default 1
 * 
 * @param scaleType
 * @text 強化タイプ
 * @type select
 * @option 定期的
 * @value interval
 * @option 指定ターン以降毎ターン
 * @value continuous
 * @default interval
 * 
 * @param turnInterval
 * @text ターン間隔/開始ターン
 * @type number
 * @min 1
 * @max 99
 * @desc 定期的: 何ターンごとに強化するか / 指定ターン以降: 何ターン目から強化開始するか
 * @default 1
 * 
 * @param powerUpValueAlly
 * @text 味方強化値
 * @type number
 * @decimals 1
 * @min 0
 * @max 999.9
 * @default 1
 * 
 * @param powerUpValueEnemy
 * @text 敵強化値
 * @type number
 * @decimals 1
 * @min 0
 * @max 999.9
 * @default 1
 */

/*~struct~PopupStyle:ja
 * @param textColor
 * @text テキスト色
 * @type string
 * @desc カンマ区切りのRGB値 (例: 255,255,0)
 * @default 255,255,0
 * 
 * @param backgroundColor
 * @text 背景色
 * @type string
 * @desc カンマ区切りのRGB値 (例: 0,0,128)
 * @default 0,0,128
 * 
 * @param fontSize
 * @text フォントサイズ
 * @type number
 * @min 12
 * @max 72
 * @default 28
 * 
 * @param duration
 * @text 表示時間
 * @type number
 * @min 1
 * @max 999
 * @default 60
 * 
 * @param padding
 * @text パディング
 * @type number
 * @min 0
 * @max 32
 * @default 8
 * 
 * @param opacity
 * @text 背景の不透明度
 * @type number
 * @min 0
 * @max 255
 * @default 192
 * 
 * @param xPosition
 * @text X座標位置
 * @type select
 * @option 左
 * @value left
 * @option 中央
 * @value center
 * @option 右
 * @value right
 * @default center
 * 
 * @param yPosition
 * @text Y座標位置
 * @type select
 * @option 上
 * @value top
 * @option 中央
 * @value middle
 * @option 下
 * @value bottom
 * @default middle
 * 
 * @param xOffset
 * @text X座標オフセット
 * @type number
 * @min -1000
 * @max 1000
 * @default 0
 * 
 * @param yOffset
 * @text Y座標オフセット
 * @type number
 * @min -1000
 * @max 1000
 * @default 0
 */

/*~struct~DespairMode:ja
 * @param modeName
 * @text モード名
 * @type string
 * @desc ブーストフェイズの表示名
 * @default ブーストフェイズ
 * 
 * @param modeMessage
 * @text 発動メッセージ
 * @type string
 * @desc モード発動時のメッセージ
 * @default 限界突破...戦闘力が急上昇！
 * 
 * @param textColor
 * @text テキスト色
 * @type string
 * @desc カンマ区切りのRGB値 (例: 255,0,0)
 * @default 255,0,0
 * 
 * @param effectDuration
 * @text 演出時間
 * @type number
 * @min 30
 * @max 300
 * @default 120
 * 
 * @param flashStrength
 * @text フラッシュ強度
 * @type number
 * @min 0
 * @max 255
 * @default 160
 * 
 * @param flashColor
 * @text フラッシュ色
 * @type string
 * @desc カンマ区切りのRGB値 (例: 255,0,0)
 * @default 255,0,0
 * 
 * @param shakeStrength
 * @text 画面振動強度
 * @type number
 * @min 0
 * @max 9
 * @default 5
 * 
 * @param useSound
 * @text SE使用
 * @type boolean
 * @default true
 * 
 * @param soundName
 * @text SE名
 * @type file
 * @dir audio/se/
 * @default Bell3
 * 
 * @param soundPitch
 * @text SEピッチ
 * @type number
 * @min 50
 * @max 150
 * @default 70
 * 
 * @param soundVolume
 * @text SE音量
 * @type number
 * @min 0
 * @max 100
 * @default 90
 * 
 * @param xPosition
 * @text X座標位置
 * @type select
 * @option 左
 * @value left
 * @option 中央
 * @value center
 * @option 右
 * @value right
 * @default center
 * 
 * @param yPosition
 * @text Y座標位置
 * @type select
 * @option 上
 * @value top
 * @option 中央
 * @value middle
 * @option 下
 * @value bottom
 * @default middle
 * 
 * @param xOffset
 * @text X座標オフセット
 * @type number
 * @min -1000
 * @max 1000
 * @default 0
 * 
 * @param yOffset
 * @text Y座標オフセット
 * @type number
 * @min -1000
 * @max 1000
 * @default 0
 */

/*~struct~CountdownStyle:ja
 * @param enabled
 * @text カウントダウン有効
 * @type boolean
 * @desc カウントダウンを表示するかどうか
 * @default true
 * 
 * @param startCount
 * @text 開始カウント
 * @type number
 * @min 1
 * @max 10
 * @desc 何ターン前からカウントダウンを開始するか
 * @default 3
 * 
 * @param displayDuration
 * @text 表示時間
 * @type number
 * @min 30
 * @max 300
 * @desc カウントダウン表示の合計時間（フレーム数）
 * @default 90
 * 
 * @param fadeInDuration
 * @text フェードイン時間
 * @type number
 * @min 5
 * @max 60
 * @desc フェードインにかかる時間（フレーム数）
 * @default 20
 * 
 * @param fadeOutDuration
 * @text フェードアウト時間
 * @type number
 * @min 5
 * @max 60
 * @desc フェードアウトにかかる時間（フレーム数）
 * @default 20
 * 
 * @param normalWobbleSettings
 * @text 通常時の揺れ設定
 * @type struct<WobbleSettings>
 * @desc 通常時の文字の揺れ設定
 * @default {"enabled":"true","strength":"0.02","speed":"1.0"}
 * 
 * @param warningWobbleSettings
 * @text 警告時の揺れ設定
 * @type struct<WobbleSettings>
 * @desc 残り1ターン時の文字の揺れ設定
 * @default {"enabled":"true","strength":"0.05","speed":"2.0"}
 * 
 * @param textColor
 * @text 通常テキスト色
 * @type string
 * @desc カンマ区切りのRGB値 (例: 255,128,128)
 * @default 255,128,128
 * 
 * @param warningColor
 * @text 警告テキスト色
 * @type string
 * @desc カンマ区切りのRGB値 (例: 255,0,0)
 * @default 255,0,0
 * 
 * @param fontSize
 * @text フォントサイズ
 * @type number
 * @min 16
 * @max 48
 * @default 24
 * 
 * @param soundEnabled
 * @text SE有効
 * @type boolean
 * @default true
 * 
 * @param soundName
 * @text 通常SE
 * @type file
 * @dir audio/se/
 * @default Decision2
 * 
 * @param warningName
 * @text 警告SE
 * @type file
 * @dir audio/se/
 * @default Bell2
 * 
 * @param soundVolume
 * @text SE音量
 * @type number
 * @min 0
 * @max 100
 * @default 70
 * 
 * @param xPosition
 * @text X座標位置
 * @type select
 * @option 左
 * @value left
 * @option 中央
 * @value center
 * @option 右
 * @value right
 * @default center
 * 
 * @param yPosition
 * @text Y座標位置
 * @type select
 * @option 上
 * @value top
 * @option 中央
 * @value middle
 * @option 下
 * @value bottom
 * @default middle
 * 
 * @param xOffset
 * @text X座標オフセット
 * @type number
 * @min -1000
 * @max 1000
 * @default 0
 * 
 * @param yOffset
 * @text Y座標オフセット
 * @type number
 * @min -1000
 * @max 1000
 * @default 0
 */

/*~struct~WobbleSettings:ja
 * @param enabled
 * @text 揺れ有効
 * @type boolean
 * @desc 文字を揺らすかどうか
 * @default true
 * 
 * @param strength
 * @text 揺れの強さ
 * @type number
 * @decimals 3
 * @min 0
 * @max 0.5
 * @desc 文字の揺れの大きさ（0～0.5）
 * @default 0.02
 * 
 * @param speed
 * @text 揺れの速さ
 * @type number
 * @decimals 1
 * @min 0.1
 * @max 5.0
 * @desc 文字の揺れの速さ（0.1～5.0）
 * @default 1.0
 */

(() => {
    'use strict';

    const pluginName = 'Svr_BattlePowerScaling';
    
    //-----------------------------------------------------------------------------
    // Debug Class
    //-----------------------------------------------------------------------------
    class Debug {
        static log(...args) {
            if (Debug.enabled) {
                console.log(`[${pluginName}]`, ...args);
            }
        }
    }
    Debug.enabled = true;

    //-----------------------------------------------------------------------------
    // Parameter Loading
    //-----------------------------------------------------------------------------
    const params = PluginManager.parameters(pluginName);
    
    // Parse popup settings
    let popupStyle;
    try {
        popupStyle = JSON.parse(params.PopupSettings || '{"textColor":"255,255,0","backgroundColor":"0,0,128","fontSize":"28","duration":"60","padding":"8","opacity":"192","xPosition":"center","yPosition":"middle","xOffset":"0","yOffset":"0"}');
        Debug.log("Popup style:", popupStyle);
    } catch (e) {
        console.error("Popup style parse error:", e);
        popupStyle = {
            textColor: "255,255,0",
            backgroundColor: "0,0,128",
            fontSize: "28",
            duration: "60",
            padding: "8",
            opacity: "192",
            xPosition: "center",
            yPosition: "middle",
            xOffset: "0",
            yOffset: "0"
        };
    }

    // Parse power up settings
    let powerUpSettings = [];
    try {
        const rawSettings = JSON.parse(params.PowerUpSettings || '[]');
        powerUpSettings = rawSettings.map(setting => {
            const parsed = JSON.parse(setting);
            return {
                groupIdStart: Number(parsed.groupIdStart),
                groupIdEnd: Number(parsed.groupIdEnd),
                scaleType: String(parsed.scaleType || 'interval'),
                turnInterval: Number(parsed.turnInterval),
                powerUpValueAlly: Number(parsed.powerUpValueAlly),
                powerUpValueEnemy: Number(parsed.powerUpValueEnemy)
            };
        });
    } catch (e) {
        console.error("Power up settings parse error:", e);
        powerUpSettings = [];
    }

    // Parse despair mode settings
    let despairSettings;
    try {
        despairSettings = JSON.parse(params.DespairSettings || '{}');
        if (typeof despairSettings === 'string') {
            despairSettings = JSON.parse(despairSettings);
        }
        Debug.log("Despair settings:", despairSettings);
    } catch (e) {
        console.error("Despair settings parse error:", e);
        despairSettings = {
            modeName: "ブーストフェイズ",
            modeMessage: "限界突破...戦闘力が急上昇！",
            textColor: "255,0,0",
            effectDuration: "120",
            flashStrength: "160",
            flashColor: "255,0,0",
            shakeStrength: "5",
            useSound: "true",
            soundName: "Bell3",
            soundPitch: "70",
            soundVolume: "90"
        };
    }

    // Parse countdown settings
    let countdownSettings;
    try {
        countdownSettings = JSON.parse(params.CountdownSettings || '{}');
        if (typeof countdownSettings === 'string') {
            countdownSettings = JSON.parse(countdownSettings);
        }
        Debug.log("Countdown settings:", countdownSettings);
    } catch (e) {
        console.error("Countdown settings parse error:", e);
        countdownSettings = {
            enabled: true,
            startCount: 3,
            textColor: "255,128,128",
            warningColor: "255,0,0",
            fontSize: "24",
            soundEnabled: true,
            soundName: "Decision2",
            warningName: "Bell2",
            soundVolume: "70"
        };
    }

    const allyPowerRateVariableId = Number(params.AllyPowerRateVariableId) || 1;
    const enemyPowerRateVariableId = Number(params.EnemyPowerRateVariableId) || 2;
    Debug.enabled = (params.EnableDebug || 'true') === 'true';

//-----------------------------------------------------------------------------
    // Sprite_PowerUpPopup
    //-----------------------------------------------------------------------------
    function Sprite_PowerUpPopup() {
        this.initialize(...arguments);
    }

    Sprite_PowerUpPopup.prototype = Object.create(Sprite.prototype);
    Sprite_PowerUpPopup.prototype.constructor = Sprite_PowerUpPopup;

    Sprite_PowerUpPopup.prototype.initialize = function(allyValue, enemyValue) {
        Sprite.prototype.initialize.call(this);
        this._duration = Number(popupStyle.duration);
        this._flashColor = [0, 0, 0, 0];
        this._flashDuration = 0;
        this._state = 'entering';
        this._slideDistance = 160;
        this._slideDuration = 15;
        this._showDuration = this._duration - (this._slideDuration * 2);
        this.opacity = 0;
        this.createBitmap(allyValue, enemyValue);
        this.setInitialPosition();
        this.update();
    };

    Sprite_PowerUpPopup.prototype.createBitmap = function(allyValue, enemyValue) {
        const fontSize = Number(popupStyle.fontSize);
        const smallFontSize = Math.floor(fontSize * 0.8);
        const padding = Number(popupStyle.padding || 8);
        const lineSpacing = 5; // 行間の追加スペース
        const extraPadding = 20; // テキスト用の追加パディング
        
        // 強化値の取得とフェーズ計算
        const totalAllyPowerRate = $gameVariables.value(allyPowerRateVariableId);
        const totalEnemyPowerRate = $gameVariables.value(enemyPowerRateVariableId);
        const phaseNumber = Math.floor(totalAllyPowerRate / 50);
        
        // 表示テキストの準備
        const mainText = `ブーストフェイズ：PHASE ${phaseNumber || 1}`;
        const allyText  = `味方：戦力+${allyValue}%（累計${totalAllyPowerRate}%）`;
        const enemyText = `敵　：戦力+${enemyValue}%（累計${totalEnemyPowerRate}%）`;
        
        // テキストサイズを計測
        const temp = new Bitmap(1, 1);
        temp.fontSize = fontSize;
        const mainWidth = temp.measureTextWidth(mainText);
        temp.fontSize = smallFontSize;
        const statusWidth = Math.max(
            temp.measureTextWidth(allyText),
            temp.measureTextWidth(enemyText)
        );
        
        // 背景サイズの計算
        const width = Math.max(mainWidth, statusWidth) + (padding * 2) + extraPadding;
        const height = fontSize + (smallFontSize * 2) + (padding * 4) + lineSpacing;
        
        this.bitmap = new Bitmap(width, height);
        
        // グラデーション背景の描画
        const bgColor = popupStyle.backgroundColor.split(',').map(Number);
        const opacity = Number(popupStyle.opacity || 192);
        const context = this.bitmap._context;
        const gradient = context.createLinearGradient(0, 0, width, 0);
        
        // 左から右へのグラデーション（透明度も変化）
        gradient.addColorStop(0, `rgba(${bgColor[0]},${bgColor[1]},${bgColor[2]},${opacity/255})`);
        gradient.addColorStop(0.85, `rgba(${bgColor[0]},${bgColor[1]},${bgColor[2]},${(opacity * 0.7)/255})`);
        gradient.addColorStop(1, `rgba(${bgColor[0]},${bgColor[1]},${bgColor[2]},0)`);
        
        context.fillStyle = gradient;
        context.fillRect(0, 0, width, height);
        this.bitmap._baseTexture.update();
        
        // テキストカラーの設定
        const textColor = popupStyle.textColor.split(',').map(Number);
        this.bitmap.textColor = `rgb(${textColor[0]},${textColor[1]},${textColor[2]})`;
        
        let currentY = padding;
        
        // PHASEの描画（左揃え）
        this.bitmap.fontSize = fontSize;
        this.bitmap.drawText(
            mainText,
            padding,
            currentY,
            width - (padding * 2),
            fontSize,
            'left'
        );
        currentY += fontSize + padding;
        
        // 味方と敵の強化値表示（左揃え）
        this.bitmap.fontSize = smallFontSize;
        this.bitmap.drawText(
            allyText,
            padding,
            currentY,
            width - (padding * 2),
            smallFontSize,
            'left'
        );
        currentY += smallFontSize + lineSpacing;
        
        this.bitmap.drawText(
            enemyText,
            padding,
            currentY,
            width - (padding * 2),
            smallFontSize,
            'left'
        );
    };

    Sprite_PowerUpPopup.prototype.setInitialPosition = function() {
        switch (popupStyle.xPosition) {
            case 'left':
                this.x = 0 - this.width - this._slideDistance;
                break;
            case 'right':
                this.x = Graphics.width - this.width - this._slideDistance;
                break;
            case 'center':
            default:
                this.x = (Graphics.width / 2 - this.width / 2) - this._slideDistance;
                break;
        }
        
        switch (popupStyle.yPosition) {
            case 'top':
                this.y = 0;
                break;
            case 'bottom':
                this.y = Graphics.height - this.height;
                break;
            case 'middle':
            default:
                this.y = Graphics.height / 2 - this.height / 2;
                break;
        }

        this.x += Number(popupStyle.xOffset);
        this.y += Number(popupStyle.yOffset);
        this._targetX = this.x + this._slideDistance;
    };

    Sprite_PowerUpPopup.prototype.update = function() {
        Sprite.prototype.update.call(this);
        if (this._duration > 0) {
            this._duration--;
            this.updateState();
            this.updatePosition();
        }
        if (this._duration === 0) {
            this.parent?.removeChild(this);
        }
    };

    Sprite_PowerUpPopup.prototype.updateState = function() {
        const totalDuration = this._slideDuration * 2 + this._showDuration;
        
        if (this._state === 'entering' && this._duration <= totalDuration - this._slideDuration) {
            this._state = 'showing';
        } else if (this._state === 'showing' && this._duration <= this._slideDuration) {
            this._state = 'exiting';
        }
    };

    Sprite_PowerUpPopup.prototype.updatePosition = function() {
        switch (this._state) {
            case 'entering':
                const enterProgress = (this._slideDuration - (this._duration - (this._showDuration + this._slideDuration))) / this._slideDuration;
                this.x = this._targetX - this._slideDistance * (1 - this.easeOutCubic(enterProgress));
                this.opacity = 255 * enterProgress;
                break;
            
            case 'showing':
                this.x = this._targetX;
                this.opacity = 255;
                break;
            
            case 'exiting':
                const exitProgress = this._duration / this._slideDuration;
                this.x = this._targetX + this._slideDistance * (1 - exitProgress);
                this.opacity = 255 * exitProgress;
                break;
        }
    };

    Sprite_PowerUpPopup.prototype.easeOutCubic = function(x) {
        return 1 - Math.pow(1 - x, 3);
    };

    //-----------------------------------------------------------------------------
    // Sprite_DespairModePopup
    //-----------------------------------------------------------------------------
    function Sprite_DespairModePopup() {
        this.initialize(...arguments);
    }
    
    Sprite_DespairModePopup.prototype = Object.create(Sprite.prototype);
    Sprite_DespairModePopup.prototype.constructor = Sprite_DespairModePopup;
    
    Sprite_DespairModePopup.prototype.initialize = function() {
        Sprite.prototype.initialize.call(this);
        this._duration = Number(despairSettings.effectDuration);
        this._flashDuration = this._duration / 2;
        this._state = 'entering';
        this._fadeInDuration = 30; // フェードイン時間
        this._stayDuration = this._duration - 60; // 表示継続時間
        this._fadeOutDuration = 30; // フェードアウト時間
        this.opacity = 0;
        this.scale.x = 0;
        this.scale.y = 0;
        this.createBitmap();
        this.setInitialPosition();
        this.update();
    };
    
    Sprite_DespairModePopup.prototype.createBitmap = function() {
        const width = Graphics.width * 0.8;
        const height = 72;
        this.bitmap = new Bitmap(width, height);
        
        const textColor = despairSettings.textColor.split(',').map(Number);
        this.bitmap.textColor = `rgb(${textColor[0]},${textColor[1]},${textColor[2]})`;
        this.bitmap.outlineColor = 'rgba(0, 0, 0, 0.8)';
        this.bitmap.outlineWidth = 4;
        this.bitmap.fontSize = 42;
        
        const modeName = despairSettings.modeName;
        const modeMessage = despairSettings.modeMessage;
        
        this.bitmap.drawText(modeName, 0, 0, width, 42, 'center');
        this.bitmap.fontSize = 28;
        this.bitmap.drawText(modeMessage, 0, 42, width, 30, 'center');
    };
    
    Sprite_DespairModePopup.prototype.setInitialPosition = function() {
        // X座標の設定
        switch (despairSettings.xPosition) {
            case 'left':
                this.x = 0;
                break;
            case 'right':
                this.x = Graphics.width - this.width;
                break;
            case 'center':
            default:
                this.x = Graphics.width / 2 - this.width / 2;
                break;
        }
        
        // Y座標の設定
        switch (despairSettings.yPosition) {
            case 'top':
                this.y = 0;
                break;
            case 'bottom':
                this.y = Graphics.height - this.height;
                break;
            case 'middle':
            default:
                this.y = Graphics.height / 3 - this.height / 2;
                break;
        }
        
        // オフセットの適用
        this.x += Number(despairSettings.xOffset || 0);
        this.y += Number(despairSettings.yOffset || 0);
    };
    
    Sprite_DespairModePopup.prototype.update = function() {
        Sprite.prototype.update.call(this);
        if (this._duration > 0) {
            this._duration--;
            this.updateEffect();
        }
        if (this._duration === 0) {
            this.parent?.removeChild(this);
        }
    };
    
    Sprite_DespairModePopup.prototype.updateEffect = function() {
        const totalDuration = this._fadeInDuration + this._stayDuration + this._fadeOutDuration;
        
        // フェードイン
        if (this._duration > totalDuration - this._fadeInDuration) {
            const progress = (totalDuration - this._duration) / this._fadeInDuration;
            this.opacity = 255 * this.easeOutCubic(progress);
            const scale = this.easeOutElastic(progress);
            this.scale.x = scale;
            this.scale.y = scale;
        }
        // 表示維持（微かに動く）
        else if (this._duration > this._fadeOutDuration) {
            this.opacity = 255;
            const wobbleProgress = (this._duration % 60) / 60;
            const wobbleScale = 1 + Math.sin(wobbleProgress * Math.PI * 2) * 0.02;
            this.scale.x = wobbleScale;
            this.scale.y = wobbleScale;
        }
        // フェードアウト
        else {
            const progress = this._duration / this._fadeOutDuration;
            this.opacity = 255 * this.easeInCubic(progress);
            const scale = 1 + (1 - progress) * 0.2;
            this.scale.x = scale;
            this.scale.y = scale;
        }
    };
    
    // イージング関数
    Sprite_DespairModePopup.prototype.easeOutElastic = function(x) {
        const c4 = (2 * Math.PI) / 3;
        return x === 0 ? 0 : x === 1 ? 1 :
            Math.pow(2, -10 * x) * Math.sin((x * 10 - 0.75) * c4) + 1;
    };
    
    Sprite_DespairModePopup.prototype.easeOutCubic = function(x) {
        return 1 - Math.pow(1 - x, 3);
    };
    
    Sprite_DespairModePopup.prototype.easeInCubic = function(x) {
        return x * x * x;
    };

//-----------------------------------------------------------------------------
// Sprite_CountdownPopup
//-----------------------------------------------------------------------------
function Sprite_CountdownPopup() {
    this.initialize(...arguments);
}

Sprite_CountdownPopup.prototype = Object.create(Sprite.prototype);
Sprite_CountdownPopup.prototype.constructor = Sprite_CountdownPopup;

Sprite_CountdownPopup.prototype.initialize = function(turnsLeft) {
    Sprite.prototype.initialize.call(this);
    this._turnsLeft = turnsLeft;
    
    // パラメータから各種時間を取得
    this._duration = Number(countdownSettings.displayDuration || 90);
    this._fadeInDuration = Number(countdownSettings.fadeInDuration || 20);
    this._fadeOutDuration = Number(countdownSettings.fadeOutDuration || 20);
    this._stayDuration = this._duration - (this._fadeInDuration + this._fadeOutDuration);
    
    // 揺れの設定を取得
    this.initializeWobbleSettings();
    
    this.opacity = 0;
    this.scale.x = 0;
    this.scale.y = 0;
    this.createBitmap();
    this.setInitialPosition();
    this.update();
};

Sprite_CountdownPopup.prototype.initializeWobbleSettings = function() {
    let normalSettings, warningSettings;
    
    try {
        normalSettings = JSON.parse(countdownSettings.normalWobbleSettings);
        if (typeof normalSettings === 'string') {
            normalSettings = JSON.parse(normalSettings);
        }
    } catch (e) {
        normalSettings = {
            enabled: true,
            strength: 0.02,
            speed: 1.0
        };
    }
    
    try {
        warningSettings = JSON.parse(countdownSettings.warningWobbleSettings);
        if (typeof warningSettings === 'string') {
            warningSettings = JSON.parse(warningSettings);
        }
    } catch (e) {
        warningSettings = {
            enabled: true,
            strength: 0.05,
            speed: 2.0
        };
    }
    
    this._normalWobble = {
        enabled: normalSettings.enabled === 'true',
        strength: Number(normalSettings.strength),
        speed: Number(normalSettings.speed)
    };
    
    this._warningWobble = {
        enabled: warningSettings.enabled === 'true',
        strength: Number(warningSettings.strength),
        speed: Number(warningSettings.speed)
    };
};

Sprite_CountdownPopup.prototype.createBitmap = function() {
    const width = Graphics.width * 0.6;
    const height = 72;
    this.bitmap = new Bitmap(width, height);
    
    const isWarning = this._turnsLeft === 1;
    const fontSize = Number(countdownSettings.fontSize);
    const textColor = isWarning ? 
        countdownSettings.warningColor.split(',').map(Number) :
        countdownSettings.textColor.split(',').map(Number);
    
    this.bitmap.textColor = `rgb(${textColor[0]},${textColor[1]},${textColor[2]})`;
    this.bitmap.outlineColor = 'rgba(0, 0, 0, 0.8)';
    this.bitmap.outlineWidth = 4;
    
    // カウントダウン数字を大きく表示
    this.bitmap.fontSize = fontSize * 2;
    this.bitmap.drawText(
        String(this._turnsLeft),
        0,
        0,
        width,
        height - 24,
        'center'
    );
    
    // メッセージを下部に表示
    this.bitmap.fontSize = fontSize;
    const message = isWarning ?
        "ブーストフェイズまで残りターン！" :
        "ブーストフェイズまで残りターン";
    this.bitmap.drawText(
        message,
        0,
        height - 28,
        width,
        28,
        'center'
    );
};

Sprite_CountdownPopup.prototype.setInitialPosition = function() {
    // X座標の設定
    switch (countdownSettings.xPosition) {
        case 'left':
            this.x = 0;
            break;
        case 'right':
            this.x = Graphics.width - this.width;
            break;
        case 'center':
        default:
            this.x = Graphics.width / 2 - this.width / 2;
            break;
    }
    
    // Y座標の設定
    switch (countdownSettings.yPosition) {
        case 'top':
            this.y = 0;
            break;
        case 'bottom':
            this.y = Graphics.height - this.height;
            break;
        case 'middle':
        default:
            this.y = Graphics.height / 3 - this.height / 2;
            break;
    }
    
    // オフセットの適用
    this.x += Number(countdownSettings.xOffset || 0);
    this.y += Number(countdownSettings.yOffset || 0);
};

Sprite_CountdownPopup.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (this._duration > 0) {
        this._duration--;
        this.updateEffect();
    }
    if (this._duration === 0) {
        this.parent?.removeChild(this);
    }
};

Sprite_CountdownPopup.prototype.updateEffect = function() {
    const totalDuration = this._fadeInDuration + this._stayDuration + this._fadeOutDuration;
    
    // フェードイン
    if (this._duration > totalDuration - this._fadeInDuration) {
        const progress = (totalDuration - this._duration) / this._fadeInDuration;
        this.opacity = 255 * this.easeOutCubic(progress);
        const scale = this.easeOutElastic(progress);
        this.scale.x = scale;
        this.scale.y = scale;
    }
    // 表示維持
    else if (this._duration > this._fadeOutDuration) {
        this.opacity = 255;
        
        // 警告時（残り1ターン）の処理
        if (this._turnsLeft === 1) {
            if (this._warningWobble.enabled) {
                const flashProgress = (this._duration % (60 / this._warningWobble.speed)) / (60 / this._warningWobble.speed);
                this.opacity = 255 * (0.7 + Math.sin(flashProgress * Math.PI * 2) * 0.3);
                if (this._warningWobble.strength > 0) {
                    const wobbleScale = 1 + Math.sin(flashProgress * Math.PI * 2) * this._warningWobble.strength;
                    this.scale.x = wobbleScale;
                    this.scale.y = wobbleScale;
                }
            }
        }
        // 通常時の処理
        else if (this._normalWobble.enabled) {
            const wobbleProgress = (this._duration % (60 / this._normalWobble.speed)) / (60 / this._normalWobble.speed);
            if (this._normalWobble.strength > 0) {
                const wobbleScale = 1 + Math.sin(wobbleProgress * Math.PI * 2) * this._normalWobble.strength;
                this.scale.x = wobbleScale;
                this.scale.y = wobbleScale;
            }
        }
    }
    // フェードアウト
    else {
        const progress = this._duration / this._fadeOutDuration;
        this.opacity = 255 * this.easeInCubic(progress);
        const scale = 1 + (1 - progress) * 0.2;
        this.scale.x = scale;
        this.scale.y = scale;
    }
};

Sprite_CountdownPopup.prototype.easeOutElastic = function(x) {
    const c4 = (2 * Math.PI) / 3;
    return x === 0 ? 0 : x === 1 ? 1 :
        Math.pow(2, -10 * x) * Math.sin((x * 10 - 0.75) * c4) + 1;
};

Sprite_CountdownPopup.prototype.easeOutCubic = function(x) {
    return 1 - Math.pow(1 - x, 3);
};

Sprite_CountdownPopup.prototype.easeInCubic = function(x) {
    return x * x * x;
};

    //-----------------------------------------------------------------------------
    // Spriteset_Battle の拡張
    //-----------------------------------------------------------------------------
    const _Spriteset_Battle_initialize = Spriteset_Battle.prototype.initialize;
    Spriteset_Battle.prototype.initialize = function() {
        _Spriteset_Battle_initialize.call(this);
        this._powerUpPopups = [];
    };

    Spriteset_Battle.prototype.createPowerUpPopup = function(allyValue, enemyValue) {
        const sprite = new Sprite_PowerUpPopup(allyValue, enemyValue);
        this._powerUpPopups.push(sprite);
        this.addChild(sprite);
    };

    const _Spriteset_Battle_update = Spriteset_Battle.prototype.update;
    Spriteset_Battle.prototype.update = function() {
        _Spriteset_Battle_update.call(this);
        this._powerUpPopups = this._powerUpPopups.filter(popup => popup.parent);
    };

    //-----------------------------------------------------------------------------
    // BattleManager の拡張
    //-----------------------------------------------------------------------------
    const _BattleManager_initMembers = BattleManager.initMembers;
    BattleManager.initMembers = function() {
        _BattleManager_initMembers.call(this);
        this._allyPowerUpValue = 0;
        this._enemyPowerUpValue = 0;
        this._turnCount = 0;
        this._inDespairMode = false;
    };

    const _BattleManager_startBattle = BattleManager.startBattle;
    BattleManager.startBattle = function() {
        _BattleManager_startBattle.call(this);
        this._allyPowerUpValue = 0;
        this._enemyPowerUpValue = 0;
        this._turnCount = 0;
        this._inDespairMode = false;
        $gameVariables.setValue(allyPowerRateVariableId, this._allyPowerUpValue);
        $gameVariables.setValue(enemyPowerRateVariableId, this._enemyPowerUpValue);
    };

    const _BattleManager_endTurn = BattleManager.endTurn;
    BattleManager.endTurn = function() {
        _BattleManager_endTurn.call(this);
        this._turnCount++;
        this.updatePowerValue();
        this.checkDespairCountdown();
    };

    BattleManager.getDespairTurnThreshold = function() {
        const troopId = $gameTroop._troopId;
        let threshold = -1;

        powerUpSettings.forEach(rule => {
            if (rule.scaleType === 'continuous' && 
                troopId >= rule.groupIdStart && 
                troopId <= rule.groupIdEnd) {
                threshold = rule.turnInterval;
            }
        });

        return threshold;
    };

    BattleManager.checkDespairCountdown = function() {
        if (!countdownSettings.enabled || this._inDespairMode) return;

        const threshold = this.getDespairTurnThreshold();
        if (threshold === -1) return;

        const turnsLeft = threshold - this._turnCount;
        const startCount = Number(countdownSettings.startCount);

        if (turnsLeft > 0 && turnsLeft <= startCount) {
            // カウントダウンポップアップ表示
            const spriteset = SceneManager._scene._spriteset;
            if (spriteset) {
                const sprite = new Sprite_CountdownPopup(turnsLeft);
                spriteset.addChild(sprite);
            }

            // SE再生
            if (countdownSettings.soundEnabled === 'true') {
                AudioManager.playSe({
                    name: turnsLeft === 1 ? countdownSettings.warningName : countdownSettings.soundName,
                    pan: 0,
                    pitch: turnsLeft === 1 ? 90 : 100,
                    volume: Number(countdownSettings.soundVolume)
                });
            }
        }
    };

    BattleManager.activateDespairMode = function() {
        // 画面効果
        if (Number(despairSettings.flashStrength) > 0) {
            const flashColor = despairSettings.flashColor.split(',').map(Number);
            $gameScreen.startFlash(flashColor.concat([Number(despairSettings.flashStrength)]), 30);
        }
        
        if (Number(despairSettings.shakeStrength) > 0) {
            $gameScreen.startShake(Number(despairSettings.shakeStrength), 9, 30);
        }
        
        // SE再生
        if (despairSettings.useSound === 'true') {
            AudioManager.playSe({
                name: despairSettings.soundName,
                pan: 0,
                pitch: Number(despairSettings.soundPitch),
                volume: Number(despairSettings.soundVolume)
            });
        }
        
        // ポップアップ表示
        const spriteset = SceneManager._scene._spriteset;
        if (spriteset) {
            const sprite = new Sprite_DespairModePopup();
            spriteset.addChild(sprite);
        }
    };

    BattleManager.updatePowerValue = function() {
        const troopId = $gameTroop._troopId;
        let allyPowerUpThisTurn = 0;
        let enemyPowerUpThisTurn = 0;

        powerUpSettings.forEach(rule => {
            if (troopId >= rule.groupIdStart && troopId <= rule.groupIdEnd) {
                let shouldPowerUp = false;
                
                // 強化タイプに応じた判定
                if (rule.scaleType === 'interval') {
                    shouldPowerUp = this._turnCount % rule.turnInterval === 0;
                } else if (rule.scaleType === 'continuous') {
                    if (this._turnCount === rule.turnInterval) {
                        this._inDespairMode = true;
                        this.activateDespairMode();
                    }
                    shouldPowerUp = this._turnCount >= rule.turnInterval;
                }

                if (shouldPowerUp) {
                    this._allyPowerUpValue += rule.powerUpValueAlly;
                    this._enemyPowerUpValue += rule.powerUpValueEnemy;
                    allyPowerUpThisTurn += rule.powerUpValueAlly;
                    enemyPowerUpThisTurn += rule.powerUpValueEnemy;
                    
                    $gameVariables.setValue(allyPowerRateVariableId, this._allyPowerUpValue);
                    $gameVariables.setValue(enemyPowerRateVariableId, this._enemyPowerUpValue);

                    // 戦闘メンバー全員のステータスを更新
                    $gameParty.battleMembers().forEach(actor => {
                        actor.refresh();
                    });

                    // 敵全員のステータスを更新
                    $gameTroop.members().forEach(enemy => {
                        enemy.refresh();
                    });

                    // デバッグログ出力
                    if (Debug.enabled) {
                        Debug.log("=== ステータス更新 ===");
                        Debug.log(`ターン: ${this._turnCount}`);
                        Debug.log(`味方強化値: ${$gameVariables.value(allyPowerRateVariableId)}`);
                        Debug.log(`敵強化値: ${$gameVariables.value(enemyPowerRateVariableId)}`);
                        Debug.log(`強化タイプ: ${rule.scaleType}`);
                        
                        Debug.log("--- 味方パーティ ---");
                        $gameParty.battleMembers().forEach(actor => {
                            Debug.log(`${actor.name()} - 攻撃力: ${actor.atk}, 魔法力: ${actor.mat}`);
                        });
                        
                        Debug.log("--- 敵パーティ ---");
                        $gameTroop.members().forEach(enemy => {
                            Debug.log(`${enemy.name()} - 攻撃力: ${enemy.atk}, 魔法力: ${enemy.mat}`);
                        });
                    }
                }
            }
        });

        if (allyPowerUpThisTurn > 0 || enemyPowerUpThisTurn > 0) {
            const spriteset = SceneManager._scene._spriteset;
            if (spriteset && spriteset.createPowerUpPopup) {
                spriteset.createPowerUpPopup(allyPowerUpThisTurn, enemyPowerUpThisTurn);
            }
        }
    };

    //-----------------------------------------------------------------------------
    // Game_BattlerBase の拡張
    //-----------------------------------------------------------------------------
    const _Game_BattlerBase_paramRate = Game_BattlerBase.prototype.paramRate;
    Game_BattlerBase.prototype.paramRate = function(paramId) {
        let rate = _Game_BattlerBase_paramRate.call(this, paramId);
        
        // 戦闘中のみ強化値を適用
        if ($gameParty.inBattle()) {
            const powerRate = this.isActor() ? 
                $gameVariables.value(allyPowerRateVariableId) : 
                $gameVariables.value(enemyPowerRateVariableId);
            
            // パラメータIDごとに強化値を適用（0:HP, 1:MP, 2:ATK, 3:DEF, 4:MAT, 5:MDF, 6:AGI, 7:LUK）
            if ([2, 3, 4, 5, 6].includes(paramId)) {  // ATK, DEF, MAT, MDF, AGIに適用
                rate *= (1 + powerRate / 100);
            }
        }
        
        return rate;
    };

})();