//=============================================================================
// Svr_CalcEquipItemPerRate.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc スキルや武器タイプボーナスからのパーセンテージ増加を考慮して、最適な装備を計算します。
 * @author スタジオVR
 * 
 * @param switchId
 * @desc 装備固定モードを表すスイッチを設定します。
 * @text 装備固定スイッチ
 * @type switch
 * @default 0
 * 
 * @param OptimizeEquipTypes
 * @text 最強装備する装備タイプ
 * @desc 最強装備を行う装備タイプのリスト（カンマ区切り）
 * @default 1,2,3,4,5
 * 
 * @param ClearEquipTypes
 * @text 全て外す装備タイプ
 * @desc 装備を全て外す装備タイプのリスト（カンマ区切り）
 * @default 1,2,3,4,5
 * 
 * @param ClearEquipSE
 * @text 全て外すSE
 * @desc 全て外す時に再生するSE
 * @default Equip2
 *
 * @param CriticalRateValue
 * @text 会心率の評価値
 * @desc 会心率+1毎の評価値
 * @default 1
 * 
 * @param EvasionRateValue
 * @text 回避率の評価値
 * @desc 回避率+1毎の評価値
 * @default 2
 * 
 * @param MagicEvasionRateValue
 * @text 魔法回避率の評価値
 * @desc 魔法回避率+1毎の評価値
 * @default 1
 * 
 * @param CounterAttackRateValue
 * @text 反撃率の評価値
 * @desc 反撃率+1毎の評価値
 * @default 3
 * 
 * @param MagicReflectionRateValue
 * @text 魔法反射率の評価値
 * @desc 魔法反射率+1毎の評価値
 * @default 2
 * 
 * @command setActorOptimizeType
 * @text アクター最強装備タイプ設定
 * @desc アクター毎の最強装備時の重視タイプを設定します
 *
 * @arg actorId
 * @type actor
 * @text アクター
 * @desc 設定するアクター
 *
 * @arg optimizeType
 * @type select
 * @option 通常
 * @value normal
 * @option 攻撃重視
 * @value attack
 * @option 魔法重視
 * @value magic
 * @option 防御重視
 * @value defense
 * @text 重視タイプ
 * @desc 最強装備時に重視するパラメータ
 * @default normal
 * 
 * @command getActorOptimizeType
 * @text アクター最強装備タイプ取得
 * @desc アクターの最強装備タイプを変数に格納します
 *
 * @arg actorId
 * @type actor
 * @text アクター
 * @desc タイプを取得するアクター
 *
 * @arg variableId
 * @type variable
 * @text 格納変数
 * @desc タイプを格納する変数ID
 * 
 * @command optimizePartyEquipments
 * @text パーティ装備最適化
 * @desc パーティ全員の装備を最適化します
 * 
 * @command optimizeBattleMembersEquipments
 * @text 戦闘メンバー装備最適化
 * @desc 戦闘メンバーの装備を最適化します
 * 
 * @command openOptimizeTypeSelector
 * @text 最強装備タイプ設定画面
 * @desc パーティーメンバーの最強装備タイプを設定する画面を開きます
 * 
 * @help
 * このプラグインは、スキルやウェポンタイプボーナスによるパーセンテージ増加を
 * 考慮して最適な装備を計算します。
 * 
 * 【職業のメモ欄タグ】
 * <OptimizeType:type>
 *   職業の初期の最強装備タイプを設定します。
 *   type に指定できる値:
 *   - normal  : 通常
 *   - attack  : 攻撃重視
 *   - magic   : 魔法重視
 *   - defense : 防御重視
 * 
 * 例：
 * <OptimizeType:attack>  # 攻撃重視で最強装備
 * <OptimizeType:magic>   # 魔法重視で最強装備
 * <OptimizeType:defense> # 防御重視で最強装備
 * 
 * プラグインコマンド：
 * 1. setActorOptimizeType
 *    - アクター毎の最強装備タイプを設定します
 *    - 設定可能なタイプ：通常、攻撃重視、魔法重視、防御重視
 * 
 * 2. optimizePartyEquipments
 *    - パーティ全員の装備を最適化します
 *    - 各アクターに設定されたタイプに従って最適化を行います
 * 
 * 3. optimizeBattleMembersEquipments
 *    - 戦闘メンバーの装備を最適化します
 *    - 各アクターに設定されたタイプに従って最適化を行います
 * 
 * 最強装備を押したとき、以下の選択肢が表示されます：
 * - 最強装備に変更する
 * - 最強装備に変更する（攻撃力を重視）
 * - 最強装備に変更する（魔法力を重視）
 * - 最強装備に変更する（防御力を重視）
 * - やめる
 * 
 * プラグインコマンド：
 * 1. setActorOptimizeType
 *    - アクター毎の最強装備タイプを設定します
 *    - 設定可能なタイプ：通常、攻撃重視、魔法重視、防御重視
 * 
 * 2. optimizePartyEquipments
 *    - パーティ全員の装備を最適化します
 *    - 各アクターに設定されたタイプに従って最適化を行います
 * 
 * スクリプトコマンド：
 * $gameSystem.setActorOptimizeType(actorId, type) - アクターの装備タイプを設定
 * $gameSystem.getActorOptimizeType(actorId) - アクターの装備タイプを取得
 * $gameParty.optimizeEquipmentsAll() - パーティ全員の装備を最適化
 *
 * プラグインパラメータ：
 * - OptimizeEquipTypes: 最強装備を行う装備タイプのリストを指定します。
 * - ClearEquipTypes: 装備を全て外す装備タイプのリストを指定します。
 * - CriticalRateValue: 会心率+1毎の評価値を指定します。
 * - EvasionRateValue: 回避率+1毎の評価値を指定します。
 * - MagicEvasionRateValue: 魔法回避率+1毎の評価値を指定します。
 * - CounterAttackRateValue: 反撃率+1毎の評価値を指定します。
 * - MagicReflectionRateValue: 魔法反射率+1毎の評価値を指定します。
 */

(() => {
    'use strict';

    const pluginName = "Svr_CalcEquipItemPerRate";
    const parameters = PluginManager.parameters(pluginName);
    const optimizeEquipTypes = parameters['OptimizeEquipTypes'].split(',').map(Number);
    const clearEquipTypes = parameters['ClearEquipTypes'].split(',').map(Number);
    const clearEquipSE = String(parameters['ClearEquipSE'] || 'Equip2');
    const fixEquipSwitchId = Number(parameters['switchId'] || 0);
    const criticalRateValue = Number(parameters['CriticalRateValue'] || 1);
    const evasionRateValue = Number(parameters['EvasionRateValue'] || 2);
    const magicEvasionRateValue = Number(parameters['MagicEvasionRateValue'] || 1);
    const counterAttackRateValue = Number(parameters['CounterAttackRateValue'] || 3);
    const magicReflectionRateValue = Number(parameters['MagicReflectionRateValue'] || 2);

    // タイプと数値の対応を定義
    const OPTIMIZE_TYPE_VALUES = {
        'normal': 0,
        'attack': 1,
        'magic': 2,
        'defense': 3
    };

    const OPTIMIZE_TYPE_NAMES = {
        0: 'normal',
        1: 'attack',
        2: 'magic',
        3: 'defense'
    };

    // プラグインコマンドの登録
    PluginManager.registerCommand(pluginName, "setActorOptimizeType", args => {
        const actorId = Number(args.actorId);
        const optimizeType = String(args.optimizeType);
        $gameSystem.setActorOptimizeType(actorId, optimizeType);
    });

    // 新しいプラグインコマンド: タイプ取得
    PluginManager.registerCommand(pluginName, "getActorOptimizeType", args => {
        const actorId = Number(args.actorId);
        const variableId = Number(args.variableId);
        
        // タイプを取得して数値に変換
        const type = $gameSystem.getActorOptimizeType(actorId);
        const typeValue = OPTIMIZE_TYPE_VALUES[type];
        
        // 変数に格納
        $gameVariables.setValue(variableId, typeValue);
    });

    PluginManager.registerCommand(pluginName, "optimizePartyEquipments", args => {
        $gameParty.optimizeEquipmentsAll();
    });

    PluginManager.registerCommand(pluginName, "optimizeBattleMembersEquipments", args => {
        $gameParty.optimizeBattleMembersEquipments();
    });

    PluginManager.registerCommand(pluginName, "openOptimizeTypeSelector", args => {
        SceneManager.push(Scene_OptimizeTypeSelect);
    });

    //-----------------------------------------------------------------------------
    // Scene_OptimizeTypeSelect
    //
    // アクターと最強装備タイプを選択するシーン

    function Scene_OptimizeTypeSelect() {
        this.initialize.apply(this, arguments);
    }

    Scene_OptimizeTypeSelect.prototype = Object.create(Scene_MenuBase.prototype);
    Scene_OptimizeTypeSelect.prototype.constructor = Scene_OptimizeTypeSelect;

    Scene_OptimizeTypeSelect.prototype.initialize = function() {
        Scene_MenuBase.prototype.initialize.call(this);
    };

    Scene_OptimizeTypeSelect.prototype.create = function() {
        Scene_MenuBase.prototype.create.call(this);
        
        // レイヤーの作成
        this.createHelpWindow();
        this._portraitLayer = new Sprite(); // ポートレート用のレイヤーを作成
        this.addChild(this._portraitLayer);
        this.createPortraitSprite();
        this.createActorWindow();
        this.createTypeWindow();
        
        // ウィンドウをポートレートの前面に配置
        this.addChild(this._windowLayer);
    };

    Scene_OptimizeTypeSelect.prototype.createHelpWindow = function() {
        const rect = this.helpWindowRect();
        this._helpWindow = new Window_Help(rect);
        this._helpWindow.setText("最強装備タイプを変更するキャラクターを選択してください。");
        this.addWindow(this._helpWindow);
    };

    Scene_OptimizeTypeSelect.prototype.helpWindowRect = function() {
        const wx = 0;
        const wy = Graphics.boxHeight - this.calcWindowHeight(2, false);
        const ww = Graphics.boxWidth;
        const wh = this.calcWindowHeight(2, false);
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_OptimizeTypeSelect.prototype.actorWindowRect = function() {
        const ww = 600;
        const wh = this.calcWindowHeight(4.8, true);
        const wx = 50;
        const wy = this._helpWindow.height + 20;  // ヘルプウィンドウの下に配置
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_OptimizeTypeSelect.prototype.typeWindowRect = function() {
        const ww = 600;
        const wh = this.calcWindowHeight(4.4, true);
        const wx = 50;
        const wy = this._helpWindow.height + this.actorWindowRect().height + 28;  // アクターウィンドウの下に配置
        return new Rectangle(wx, wy, ww, wh);
    };

    // アクターウィンドウにカーソル移動時のハンドラを追加
    Scene_OptimizeTypeSelect.prototype.createActorWindow = function() {
        const rect = this.actorWindowRect();
        this._actorWindow = new Window_OptimizeTypeActorList(rect);
        this._actorWindow.setHandler('ok', this.onActorOk.bind(this));
        this._actorWindow.setHandler('cancel', this.popScene.bind(this));
        this._actorWindow.setHandler('select', this.onActorSelect.bind(this));
        this.addWindow(this._actorWindow);
        
        // 初期選択時の処理
        this._actorWindow.select(0);
        this.onActorSelect(); // 初期選択時のポートレート更新
    };

    // カーソル移動時のハンドラ
    Scene_OptimizeTypeSelect.prototype.onActorSelect = function() {
        if (!this._portraitSprite) return; // ポートレートスプライトが存在しない場合は処理をスキップ
        
        const actor = this._actorWindow.item();
        if (actor) {
            this.updatePortrait(actor);
        } else {
            this._portraitSprite.hide();
        }
    };

    Scene_OptimizeTypeSelect.prototype.createTypeWindow = function() {
        const rect = this.typeWindowRect();
        this._typeWindow = new Window_OptimizeTypeList(rect);
        this._typeWindow.setHandler('ok', this.onTypeOk.bind(this));
        this._typeWindow.setHandler('cancel', this.onTypeCancel.bind(this));
        this._typeWindow.deactivate();
        this._typeWindow.hide();
        this.addWindow(this._typeWindow);
    };

    Scene_OptimizeTypeSelect.prototype.createPortraitSprite = function() {
        this._portraitSprite = new Sprite();
        this._portraitSprite.x = Graphics.boxWidth - 670;
        this._portraitSprite.y = 20;
        this._portraitLayer.addChild(this._portraitSprite); // _portraitLayerに追加
        this._currentPortrait = null;
        this._portraitSprite.hide();
    };

    Scene_OptimizeTypeSelect.prototype.updatePortrait = function(actor) {
        if (actor) {
            const note = actor.actor().note;
            const match = /<Menu Portrait: (.+)>/.exec(note);
            if (match) {
                const fileName = match[1];
                const filePath = 'img/pictures/' + fileName;
                if (this._currentPortrait !== filePath) {
                    this._currentPortrait = filePath;
                    this._portraitSprite.bitmap = ImageManager.loadPicture(fileName);
                    this._portraitSprite.bitmap.addLoadListener(() => {
                        // 画像のロード完了後に位置とサイズを調整
                        this._portraitSprite.show();
                        
                        // 画像が大きすぎる場合はスケーリング
                        const maxWidth = 1280;  // 最大幅
                        const maxHeight = Graphics.boxHeight;  // 最大高さ
                        
                        if (this._portraitSprite.bitmap.width > maxWidth || 
                            this._portraitSprite.bitmap.height > maxHeight) {
                            const scaleX = maxWidth / this._portraitSprite.bitmap.width;
                            const scaleY = maxHeight / this._portraitSprite.bitmap.height;
                            const scale = Math.min(scaleX, scaleY);
                            this._portraitSprite.scale.x = scale;
                            this._portraitSprite.scale.y = scale;
                        } else {
                            this._portraitSprite.scale.x = 1;
                            this._portraitSprite.scale.y = 1;
                        }
                    });
                }
                this._portraitSprite.show();
            } else {
                this._portraitSprite.hide();
            }
        } else {
            this._portraitSprite.hide();
        }
    };

    const _Scene_OptimizeTypeSelect_onActorOk = Scene_OptimizeTypeSelect.prototype.onActorOk;
    Scene_OptimizeTypeSelect.prototype.onActorOk = function() {
        const actor = this._actorWindow.item();
        if (!actor) {
            this.popScene();  // やめるが選択された場合
        } else {
            // 現在の最強装備タイプを取得
            const currentType = $gameSystem.getActorOptimizeType(actor.actorId());
            
            this._typeWindow.show();
            this._typeWindow.activate();
            
            // タイプに応じてカーソル位置を設定
            switch (currentType) {
                case 'normal':
                    this._typeWindow.select(0);
                    break;
                case 'attack':
                    this._typeWindow.select(1);
                    break;
                case 'magic':
                    this._typeWindow.select(2);
                    break;
                case 'defense':
                    this._typeWindow.select(3);
                    break;
                default:
                    this._typeWindow.select(0);
            }
            
            this.updatePortrait(actor);
            this._helpWindow.setText(`${actor.name()}の最強装備タイプを選択してください。`);
        }
    };

    //-----------------------------------------------------------------------------
    // Window_OptimizeTypeList
    //-----------------------------------------------------------------------------

    // タイプ選択ウィンドウの行の高さも調整
    Window_OptimizeTypeList.prototype.lineHeight = function() {
        return 42;  // 行の高さを増加
    };

    Window_OptimizeTypeList.prototype.itemHeight = function() {
        return 56;  // アイテムの高さを増加
    };

    const _Scene_OptimizeTypeSelect_onTypeCancel = Scene_OptimizeTypeSelect.prototype.onTypeCancel;
    Scene_OptimizeTypeSelect.prototype.onTypeCancel = function() {
        _Scene_OptimizeTypeSelect_onTypeCancel.call(this);
        const actor = this._actorWindow.item();
        if (actor) {
            this.updatePortrait(actor);
        } else {
            this._portraitSprite.hide();
        }
        this._helpWindow.setText("最強装備タイプを変更するアクターを選択してください。");
    };

    // シーン更新処理の追加
    Scene_OptimizeTypeSelect.prototype.update = function() {
        Scene_MenuBase.prototype.update.call(this);
        if (this._actorWindow.active) {
            // アクターウィンドウがアクティブな時は毎フレームポートレートの表示を更新
            this.onActorSelect();
        }
    };

    // アクターウィンドウにカーソル移動時のハンドラを追加
    Scene_OptimizeTypeSelect.prototype.createActorWindow = function() {
        const rect = this.actorWindowRect();
        this._actorWindow = new Window_OptimizeTypeActorList(rect);
        this._actorWindow.setHandler('ok', this.onActorOk.bind(this));
        this._actorWindow.setHandler('cancel', this.popScene.bind(this));
        this._actorWindow.setHandler('select', this.onActorSelect.bind(this));  // 追加
        this.addWindow(this._actorWindow);

        // 初期選択時の処理を追加
        this._actorWindow.select(0);
        this.onActorSelect();
    };

    Scene_OptimizeTypeSelect.prototype.onTypeOk = function() {
        const actor = this._actorWindow.item();
        const type = this._typeWindow.type();
        $gameSystem.setActorOptimizeType(actor.actorId(), type);
        this._actorWindow.refresh();
        this.onTypeCancel();
    };

    Scene_OptimizeTypeSelect.prototype.onTypeCancel = function() {
        this._typeWindow.hide();
        this._typeWindow.deactivate();
        this._actorWindow.activate();
        this._portraitSprite.hide();
        this._helpWindow.setText("最強装備タイプを変更するキャラクターを選択してください。");
    };

    //-----------------------------------------------------------------------------
    // Window_OptimizeTypeActorList
    //
    // アクター一覧を表示するウィンドウ

    function Window_OptimizeTypeActorList() {
        this.initialize.apply(this, arguments);
    }

    Window_OptimizeTypeActorList.prototype = Object.create(Window_Selectable.prototype);
    Window_OptimizeTypeActorList.prototype.constructor = Window_OptimizeTypeActorList;

    Window_OptimizeTypeActorList.prototype.initialize = function(rect) {
        Window_Selectable.prototype.initialize.call(this, rect);
        this.makeItemList();
        this.refresh();
        this.select(0);
        this.activate();
    };

    Window_OptimizeTypeActorList.prototype.makeItemList = function() {
        this._data = ['cancel'].concat($gameParty.members());
    };

    Window_OptimizeTypeActorList.prototype.maxItems = function() {
        return this._data ? this._data.length : 0;
    };

    Window_OptimizeTypeActorList.prototype.item = function() {
        const index = this.index();
        return index > 0 ? this._data[index] : null;
    };

    Window_OptimizeTypeActorList.prototype.drawItem = function(index) {
        const rect = this.itemLineRect(index);
        if (index === 0) {
            // やめるの描画
            this.changePaintOpacity(true);
            this.drawTextEx("\\I[160]やめる", rect.x + 4, rect.y, rect.width);
        } else {
            const actor = this._data[index];
            if (actor) {
                const type = $gameSystem.getActorOptimizeType(actor.actorId());
                const typeText = this.getTypeText(type);
                this.changePaintOpacity(true);
                // アクター名を描画
                this.drawText(actor.name(), rect.x + 4, rect.y, rect.width / 2 - 4);
                // タイプを描画（アイコン付き）
                this.drawTextEx(`（${typeText}）`, rect.x + rect.width / 2, rect.y, rect.width / 2);
            }
        }
    };

    //-----------------------------------------------------------------------------
    // Window_OptimizeTypeList
    //-----------------------------------------------------------------------------

    Window_OptimizeTypeList.prototype.initialize = function(rect) {
        Window_Command.prototype.initialize.call(this, rect);
    };

    Window_OptimizeTypeActorList.prototype.getTypeText = function(type) {
        switch (type) {
            case 'normal':
                return '\\I[137]通常型';
            case 'attack':
                return '\\I[76]攻撃型';
            case 'magic':
                return '\\I[101]魔法攻撃型';
            case 'defense':
                return '\\I[81]防御型';
            default:
                return '\\I[137]通常型';
        }
    };

    // アイコンの分の余白を確保
    Window_OptimizeTypeActorList.prototype.itemHeight = function() {
        return 42;  // アイテムの高さを増加
    };

    // リストの行数を調整（必要に応じて）
    Window_OptimizeTypeActorList.prototype.numVisibleRows = function() {
        return 7;
    };

    // フォントサイズを調整（必要に応じて）
    Window_OptimizeTypeActorList.prototype.lineHeight = function() {
        return 42;  // 行の高さを増加
    };

    //-----------------------------------------------------------------------------
    // Window_OptimizeTypeList
    //
    // 最強装備タイプを選択するウィンドウ

    function Window_OptimizeTypeList() {
        this.initialize.apply(this, arguments);
    }

    Window_OptimizeTypeList.prototype = Object.create(Window_Command.prototype);
    Window_OptimizeTypeList.prototype.constructor = Window_OptimizeTypeList;

    Window_OptimizeTypeList.prototype.makeCommandList = function() {
        this.addCommand("\\I[137]通常型に変更する", 'normal');
        this.addCommand("\\I[76]攻撃型に変更する", 'attack');
        this.addCommand("\\I[101]魔法攻撃型に変更する", 'magic');
        this.addCommand("\\I[81]防御型に変更する", 'defense');
    };

    Window_OptimizeTypeList.prototype.drawItem = function(index) {
        const rect = this.itemLineRect(index);
        const align = this.itemTextAlign();
        this.resetTextColor();
        this.changePaintOpacity(this.isCommandEnabled(index));
        
        // 描画位置を縦方向に中央揃え
        const textHeight = this.textSizeEx(this.commandName(index)).height;
        const yOffset = Math.max(0, (rect.height - textHeight) / 2);
        
        this.drawTextEx(this.commandName(index), rect.x + 4, rect.y + yOffset, rect.width - 8);
    };

    // テキストサイズを計算するメソッド
    Window_OptimizeTypeList.prototype.textSizeEx = function(text) {
        this.resetFontSettings();
        const textState = this.createTextState(text, 0, 0, 0);
        textState.drawing = false;
        this.processAllText(textState);
        return { width: textState.outputWidth, height: textState.outputHeight };
    };

    // アイコンの分の余白を確保
    Window_OptimizeTypeList.prototype.itemHeight = function() {
        return 48;  // アイコンが表示できる高さに調整
    };

    // テキストの描画位置を調整
    Window_OptimizeTypeList.prototype.textPadding = function() {
        return 6;  // テキストの左余白を調整
    };

    Window_OptimizeTypeList.prototype.type = function() {
        return this._list[this.index()].symbol;
    };

    // 戦闘メンバーの装備最適化機能を追加
    Game_Party.prototype.optimizeBattleMembersEquipments = function() {
        this.battleMembers().forEach(function(actor) {
            const optimizeType = $gameSystem.getActorOptimizeType(actor.actorId());
            switch (optimizeType) {
                case 'attack':
                    actor.optimizeEquipmentsForAttack();
                    break;
                case 'magic':
                    actor.optimizeEquipmentsForMagic();
                    break;
                case 'defense':
                    actor.optimizeEquipmentsForDefense();
                    break;
                default:
                    actor.optimizeEquipments();
                    break;
            }
        });
    };

    // Game_Actor拡張: 職業変更時の処理を拡張
    const _Game_Actor_changeClass = Game_Actor.prototype.changeClass;
    Game_Actor.prototype.changeClass = function(classId, keepExp) {
        _Game_Actor_changeClass.call(this, classId, keepExp);
        this.initOptimizeTypeFromClass();
    };

    // Game_Actor拡張: セットアップ時の処理を拡張
    const _Game_Actor_setup = Game_Actor.prototype.setup;
    Game_Actor.prototype.setup = function(actorId) {
        _Game_Actor_setup.call(this, actorId);
        this.initOptimizeTypeFromClass();
    };

    // 職業のメモ欄から最強装備タイプを初期化
    Game_Actor.prototype.initOptimizeTypeFromClass = function() {
        // 現在の職業データを取得
        const currentClass = this.currentClass();
        if (!currentClass) return;

        // メモ欄から最強装備タイプを取得
        const matchResult = /<OptimizeType:(normal|attack|magic|defense)>/i.exec(currentClass.note);
        if (matchResult) {
            const type = matchResult[1].toLowerCase();
            // 現在設定されているタイプを取得
            if ($gameSystem) {
                const currentType = $gameSystem.getActorOptimizeType(this.actorId());
                
                // タイプが未設定の場合のみ、職業のタイプを設定
                if (currentType === 'normal' && !$gameSystem._actorOptimizeTypes[this.actorId()]) {
                    $gameSystem.setActorOptimizeType(this.actorId(), type);
                }
            }
        }
    };

    // Game_System拡張
    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function() {
        _Game_System_initialize.call(this);
        this._actorOptimizeTypes = {};
    };

    // Scene_Map拡張: マップシーン開始時の処理を拡張
    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.call(this);
        if ($gameParty && $gameActors && $gameSystem) {
            $gameParty.members().forEach(actor => {
                if (actor) {
                    actor.initOptimizeTypeFromClass();
                }
            });
        }
    };

    // Scene_Title拡張: 新規ゲーム開始時の処理を拡張
    const _Scene_Title_commandNewGame = Scene_Title.prototype.commandNewGame;
    Scene_Title.prototype.commandNewGame = function() {
        _Scene_Title_commandNewGame.call(this);
        if ($gameParty && $gameActors && $gameSystem) {
            $gameParty.members().forEach(actor => {
                if (actor) {
                    actor.initOptimizeTypeFromClass();
                }
            });
        }
    };

    // セーブデータをロードした時の処理を拡張
    const _Scene_Load_onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
        _Scene_Load_onLoadSuccess.call(this);
        if ($gameParty && $gameActors && $gameSystem) {
            $gameParty.members().forEach(actor => {
                if (actor) {
                    actor.initOptimizeTypeFromClass();
                }
            });
        }
    };

    Game_System.prototype.setActorOptimizeType = function(actorId, type) {
        if (!this._actorOptimizeTypes) {
            this._actorOptimizeTypes = {};
        }
        // 数値が渡された場合の対応を追加
        if (typeof type === 'number') {
            type = OPTIMIZE_TYPE_NAMES[type] || 'normal';
        }
        this._actorOptimizeTypes[actorId] = type;
    };

    Game_System.prototype.getActorOptimizeType = function(actorId) {
        if (!this._actorOptimizeTypes) {
            this._actorOptimizeTypes = {};
        }
        return this._actorOptimizeTypes[actorId] || 'normal';
    };
    
    // パーティ全体の最強装備処理を追加
    Game_Party.prototype.optimizeEquipmentsAll = function() {
        this.members().forEach(function(actor) {
            const optimizeType = $gameSystem.getActorOptimizeType(actor.actorId());
            switch (optimizeType) {
                case 'attack':
                    actor.optimizeEquipmentsForAttack();
                    break;
                case 'magic':
                    actor.optimizeEquipmentsForMagic();
                    break;
                case 'defense':
                    actor.optimizeEquipmentsForDefense();
                    break;
                default:
                    actor.optimizeEquipments();
                    break;
            }
        });
    };
    
    // Window_OptimizeChoiceの定義
    function Window_OptimizeChoice() {
        this.initialize(...arguments);
    }

    Window_OptimizeChoice.prototype = Object.create(Window_Command.prototype);
    Window_OptimizeChoice.prototype.constructor = Window_OptimizeChoice;

    Window_OptimizeChoice.prototype.initialize = function(rect) {
        Window_Command.prototype.initialize.call(this, rect);
        this.openness = 0;
        this.deactivate();
    };

    Window_OptimizeChoice.prototype.makeCommandList = function() {
        this.addCommand("\\I[137]最強装備に変更する", "optimize");
        this.addCommand("\\I[76]最強装備に変更する（攻撃型）", "optimizeAttack");
        this.addCommand("\\I[101]最強装備に変更する（魔法攻撃型）", "optimizeMagic");
        this.addCommand("\\I[81]最強装備に変更する（防御型）", "optimizeDefense");
        this.addCommand("\\I[160]やめる", "cancel");
    };

    Window_OptimizeChoice.prototype.processOk = function() {
        if (this.isCurrentItemEnabled()) {
            this.updateInputData();
            this.deactivate();
            this.callOkHandler();
        } else {
            this.playBuzzerSound();
        }
    };

    Window_OptimizeChoice.prototype.drawItem = function(index) {
        const rect = this.itemLineRect(index);
        const align = this.itemTextAlign();
        this.resetTextColor();
        this.changePaintOpacity(this.isCommandEnabled(index));
        this.drawTextEx(this.commandName(index), rect.x, rect.y, rect.width, align);
    };

    Window_OptimizeChoice.prototype.textWidthEx = function(text) {
        return this.drawTextEx(text, 0, this.contents.height);
    };

    // Scene_Equipの拡張
    const _Scene_Equip_create = Scene_Equip.prototype.create;
    Scene_Equip.prototype.create = function() {
        _Scene_Equip_create.call(this);
        this.createOptimizeChoiceWindow();
    };

    Scene_Equip.prototype.createOptimizeChoiceWindow = function() {
        const rect = this.optimizeChoiceWindowRect();
        this._optimizeChoiceWindow = new Window_OptimizeChoice(rect);
        this._optimizeChoiceWindow.setHandler("optimize", this.onOptimizeChoice.bind(this, 0));
        this._optimizeChoiceWindow.setHandler("optimizeAttack", this.onOptimizeChoice.bind(this, 1));
        this._optimizeChoiceWindow.setHandler("optimizeMagic", this.onOptimizeChoice.bind(this, 2));
        this._optimizeChoiceWindow.setHandler("optimizeDefense", this.onOptimizeChoice.bind(this, 3));
        this._optimizeChoiceWindow.setHandler("cancel", this.onOptimizeCancel.bind(this));
        this.addWindow(this._optimizeChoiceWindow);
    };

    Scene_Equip.prototype.optimizeChoiceWindowRect = function() {
        const ww = 500;
        const wh = this.calcWindowHeight(5, true);
        const wx = (Graphics.boxWidth - ww) / 2;
        const wy = (Graphics.boxHeight - wh) / 2;
        return new Rectangle(wx, wy, ww, wh);
    };

    // 最強装備のコマンドを拡張
    Scene_Equip.prototype.commandOptimize = function() {
        if ($gameSwitches.value(fixEquipSwitchId)) {
            // 装備固定モードがONの場合、何もせずにブザー音を鳴らす
            SoundManager.playBuzzer();
            this._commandWindow.activate();
            return;
        }
        
        // 装備固定モードがOFFの場合、選択肢ウィンドウを表示
        SoundManager.playOk();
        this._optimizeChoiceWindow.show();
        this._optimizeChoiceWindow.open();
        this._optimizeChoiceWindow.activate();
        this.deactivateOtherWindows();
    };

    Scene_Equip.prototype.deactivateOtherWindows = function() {
        this._commandWindow.deactivate();
        this._slotWindow.deactivate();
        this._itemWindow.deactivate();
        this._statusWindow.deactivate();
    };

    Scene_Equip.prototype.onOptimizeChoice = function(index) {
        switch(index) {
            case 0:
                this.actor().optimizeEquipments();
                break;
            case 1:
                this.actor().optimizeEquipmentsForAttack();
                break;
            case 2:
                this.actor().optimizeEquipmentsForMagic();
                break;
            case 3:
                this.actor().optimizeEquipmentsForDefense();
                break;
            case 4:
                SoundManager.playCancel();
                this.hideOptimizeChoiceWindow();
                return;
        }
        // 決定音を鳴らさずに装備音のみを再生
        SoundManager.playEquip();
        this._statusWindow.refresh();
        this._slotWindow.refresh();
        this._itemWindow.refresh();
        this.hideOptimizeChoiceWindow();
    };

    Scene_Equip.prototype.onOptimizeCancel = function() {
        SoundManager.playCancel();
        this.hideOptimizeChoiceWindow();
        // カーソルを一番上に戻す
        this._optimizeChoiceWindow.select(0);
    };

    Scene_Equip.prototype.hideOptimizeChoiceWindow = function() {
        this._optimizeChoiceWindow.close();
        this._optimizeChoiceWindow.hide();
        this._commandWindow.activate();
    };

    // 右クリックの処理を拡張
    const _Scene_Equip_onRightButtonDown = Scene_Equip.prototype.onRightButtonDown;
    Scene_Equip.prototype.onRightButtonDown = function() {
        if (this._optimizeChoiceWindow.active) {
            this.onOptimizeCancel();
        } else {
            _Scene_Equip_onRightButtonDown.call(this);
        }
    };

    const _Scene_Equip_update = Scene_Equip.prototype.update;
    Scene_Equip.prototype.update = function() {
        if (this._optimizeChoiceWindow && this._optimizeChoiceWindow.active) {
            this.updateOptimizeChoiceWindow();
        } else {
            _Scene_Equip_update.call(this);
        }
    };

    Scene_Equip.prototype.updateOptimizeChoiceWindow = function() {
        if (this._optimizeChoiceWindow.active) {
            this._optimizeChoiceWindow.update();
            if (Input.isTriggered('cancel') || TouchInput.isCancelled()) {
                this.onOptimizeCancel();
            }
        }
    };

    // タッチ操作の制御
    const _Scene_Equip_processTouch = Scene_Equip.prototype.processTouch;
    Scene_Equip.prototype.processTouch = function() {
        if (this._optimizeChoiceWindow && this._optimizeChoiceWindow.active) {
            if (TouchInput.isTriggered() && !this._optimizeChoiceWindow.isTouchedInsideFrame()) {
                this.onOptimizeCancel();
            }
        } else {
            _Scene_Equip_processTouch.call(this);
        }
    };

    // 攻撃力重視の最強装備
    Game_Actor.prototype.optimizeEquipmentsForAttack = function() {
        this.clearSpecifiedEquipments();
        this.calcOriginalStatus();
        this.optimizeSpecifiedEquipmentsForAttack();
    };

    // 魔法力重視の最強装備
    Game_Actor.prototype.optimizeEquipmentsForMagic = function() {
        this.clearSpecifiedEquipments();
        this.calcOriginalStatus();
        this.optimizeSpecifiedEquipmentsForMagic();
    };

    // 防御力重視の最強装備
    Game_Actor.prototype.optimizeEquipmentsForDefense = function() {
        this.clearSpecifiedEquipments();
        this.calcOriginalStatus();
        this.optimizeSpecifiedEquipmentsForDefense();
    };

    Game_Actor.prototype.optimizeSpecifiedEquipmentsForAttack = function() {
        const maxSlots = this.equipSlots().length;
        for (let i = 0; i < maxSlots; i++) {
//        for (let i = maxSlots; i >= 0; i--) {
            if (optimizeEquipTypes.includes(this.equipSlots()[i])) {
                this.optimizeEquipmentAtSlotForAttack(i);
            }
        }
    };

    Game_Actor.prototype.optimizeSpecifiedEquipmentsForMagic = function() {
        const maxSlots = this.equipSlots().length;
        for (let i = 0; i < maxSlots; i++) {
//        for (let i = maxSlots; i >= 0; i--) {
            if (optimizeEquipTypes.includes(this.equipSlots()[i])) {
                this.optimizeEquipmentAtSlotForMagic(i);
            }
        }
    };

    Game_Actor.prototype.optimizeSpecifiedEquipmentsForDefense = function() {
        const maxSlots = this.equipSlots().length;
        for (let i = 0; i < maxSlots; i++) {
            if (optimizeEquipTypes.includes(this.equipSlots()[i])) {
                this.optimizeEquipmentAtSlotForDefense(i);
            }
        }
    };
    
    Game_Actor.prototype.optimizeEquipmentAtSlotForAttack = function(slotId) {
//        console.log(`Optimizing equipment for attack at slot ${slotId}`);
        const etypeId = this.equipSlots()[slotId];
        const items = $gameParty.equipItems().filter(item => item.etypeId === etypeId);
        let bestItem = null;
        let bestPerformance = -1000;
        for (const item of items) {
            const performance = this.calcAttackOrientedEquipPerformance(item, slotId);
//            console.log(`  Item: ${item.name}, Performance: ${performance}`);
            if (performance > bestPerformance) {
                bestItem = item;
                bestPerformance = performance;
            }
        }
//        console.log(`Best item for slot ${slotId}: ${bestItem ? bestItem.name : 'None'}`);
        this.changeEquip(slotId, bestItem);
    };

    Game_Actor.prototype.optimizeEquipmentAtSlotForMagic = function(slotId) {
//        console.log(`Optimizing equipment for magic at slot ${slotId}`);
        const etypeId = this.equipSlots()[slotId];
        const items = $gameParty.equipItems().filter(item => item.etypeId === etypeId);
        let bestItem = null;
        let bestPerformance = -1000;
        for (const item of items) {
            const performance = this.calcMagicOrientedEquipPerformance(item, slotId);
//            console.log(`  Item: ${item.name}, Performance: ${performance}`);
            if (performance > bestPerformance) {
                bestItem = item;
                bestPerformance = performance;
            }
        }
//        console.log(`Best item for slot ${slotId}: ${bestItem ? bestItem.name : 'None'}`);
        this.changeEquip(slotId, bestItem);
    };

    Game_Actor.prototype.optimizeEquipmentAtSlotForDefense = function(slotId) {
        const etypeId = this.equipSlots()[slotId];
        const items = $gameParty.equipItems().filter(item => item.etypeId === etypeId);
        let bestItem = null;
        let bestPerformance = -1000;
        for (const item of items) {
            const performance = this.calcDefenseOrientedEquipPerformance(item, slotId);
            if (performance > bestPerformance) {
                bestItem = item;
                bestPerformance = performance;
            }
        }
        this.changeEquip(slotId, bestItem);
    };

    Game_Actor.prototype.calcEquipItemPerformanceNormal = function(item, slotId) {
        let result = this.calcEquipItemPerformance(item);

        // HPの評価を0.1倍にする
        result -= (item.params[0] * this.paramRate(0) * this.paramBuffRate(0)) * 0.9;

        // MPを無視
        result -= (item.params[1] * this.paramRate(1) * this.paramBuffRate(1));

        return Math.round(result * 10) / 10;
    };

    Game_Actor.prototype.calcEquipItemPerformanceForAttack = function(item, slotId) {
        let result = this.calcEquipItemPerformance(item);
//        console.log("基本:" + result);
        // 攻撃力の評価を2倍にする
//        result += item.params[2] * this.paramRate(2) * this.paramBuffRate(2);

        // 魔法力を無視
        result -= item.params[4] * this.paramRate(4) * this.paramBuffRate(4);
//        console.log("魔法力無視:" + result, this.paramRate(4), this.paramBuffRate(4));
/*
        // 魔法力の評価を0.2倍にする
        result -= (item.params[4] * this.paramRate(4) * this.paramBuffRate(4)) * 0.8;
*/

        // HPの評価を0.05倍にする
        result -= (item.params[0] * this.paramRate(0) * this.paramBuffRate(0)) * 0.95;

        // MPを無視
        result -= (item.params[1] * this.paramRate(1) * this.paramBuffRate(1));
//        console.log("MP無視:" + result);
        
        // 防御力の評価を0.5倍にする
        result -= (item.params[3] * this.paramRate(3) * this.paramBuffRate(3)) * 0.5;
//        console.log("防御 0.5:" + result);
        
        // 魔法防御の評価を0.5倍にする
        result -= (item.params[5] * this.paramRate(5) * this.paramBuffRate(5)) * 0.5;
//        console.log("魔法防御 0.5:" + result);
        
        return Math.round(result * 10) / 10;
    };

    Game_Actor.prototype.calcEquipItemPerformanceForMagic = function(item, slotId) {
        let result = this.calcEquipItemPerformance(item);
//        console.log(result);

        // 攻撃力を無視
        result -= item.params[2] * this.paramRate(2) * this.paramBuffRate(2);
/*
        // 攻撃力の評価を0.2倍にする
        result -= (item.params[2] * this.paramRate(2) * this.paramBuffRate(2)) * 0.8;
*/

        // 魔法力の評価を2倍にする
//        result += item.params[4] * this.paramRate(4) * this.paramBuffRate(4);

        // HPの評価を0.05倍にする
        result -= (item.params[0] * this.paramRate(0) * this.paramBuffRate(0)) * 0.95;

        // MPを無視
        result -= (item.params[1] * this.paramRate(1) * this.paramBuffRate(1));

        // 防御力の評価を0.5倍にする
        result -= (item.params[3] * this.paramRate(3) * this.paramBuffRate(3)) * 0.5;

        // 魔法防御の評価を0.5倍にする
        result -= (item.params[5] * this.paramRate(5) * this.paramBuffRate(5)) * 0.5;

        return Math.round(result * 10) / 10;
    };

    Game_Actor.prototype.calcEquipItemPerformanceForDefense = function(item, slotId) {
        let result = this.calcEquipItemPerformance(item);

        // HPの評価を0.1倍にする
        result -= (item.params[0] * this.paramRate(0) * this.paramBuffRate(0)) * 0.9;

        // MPを無視
        result -= (item.params[1] * this.paramRate(1) * this.paramBuffRate(1));

        // 攻撃力と魔法力の評価を0.5倍にする
        result -= (item.params[2] * this.paramRate(2) * this.paramBuffRate(2)) * 0.5;
        result -= (item.params[4] * this.paramRate(4) * this.paramBuffRate(4)) * 0.5;

        return Math.round(result * 10) / 10;
    };

    // 全て外す機能を拡張する
    const _Scene_Equip_commandClear = Scene_Equip.prototype.commandClear;
    Scene_Equip.prototype.commandClear = function() {
        if (this.actor().canClearAnyEquipment()) {
            this.playClearEquipSE();
            this.actor().clearSpecifiedEquipments();
            this._statusWindow.refresh();
            this._slotWindow.refresh();
            this._itemWindow.refresh();
        } else {
            SoundManager.playBuzzer(); // 全て外せない場合はブザーを鳴らす
        }
        this._commandWindow.activate();
    };

    // Define playClearEquipSE function
    Scene_Equip.prototype.playClearEquipSE = function() {
        const se = {
            name: clearEquipSE,
            volume: 90,
            pitch: 100,
            pan: 0
        };
        AudioManager.playSe(se);
    };
    
    Scene_Equip.prototype.playClearEquipSE = function() {
        AudioManager.playSe({name: clearEquipSE, pan: 0, pitch: 100, volume: 90});
    };

    // 指定された装備タイプだけを外す
    Game_Actor.prototype.clearSpecifiedEquipments = function() {
        for (let i = 0; i < this.equipSlots().length; i++) {
            if (clearEquipTypes.includes(this.equipSlots()[i])) {
                this.changeEquip(i, null);
            }
        }
    };

    // 最強装備の計算を拡張する
    const Game_Actor_prototype_optimizeEquipments = Game_Actor.prototype.optimizeEquipments;
    Game_Actor.prototype.optimizeEquipments = function() {
        if ($gameSwitches.value(fixEquipSwitchId)) {
            // 装備固定モードがONの場合、何もしない
            return;
        }
        // console.log("最強装備の計算開始");

        // 現在のHPを保存
        const currentHp = this._hp;

        this.clearSpecifiedEquipments();
        this.calcOriginalStatus();
        this.optimizeSpecifiedEquipments();

        // 保存したHPに戻す
        this._hp = currentHp;
        // HPの最大値を超えないようにする
        if (this._hp > this.mhp) {
            this._hp = this.mhp;
        }

        // console.log("最強装備の計算終了");
    };

    // 少なくとも1つの装備を外せるかどうかをチェック
    Game_Actor.prototype.canClearAnyEquipment = function() {
        return this.equipSlots().some((slotType, index) => 
            clearEquipTypes.includes(slotType) && !this.isEquipTypeFixed(slotType) && this.equips()[index]
        );
    };

    Game_Actor.prototype.optimizeSpecifiedEquipments = function() {
        const maxSlots = this.equipSlots().length;
        for (let i = 0; i < maxSlots; i++) {
            if (optimizeEquipTypes.includes(this.equipSlots()[i])) {
                this.optimizeEquipmentAtSlot(i);
            }
        }
    };

    Game_Actor.prototype.optimizeEquipmentAtSlot = function(slotId) {
        const etypeId = this.equipSlots()[slotId];
        const items = $gameParty.equipItems().filter(item => item.etypeId === etypeId);
        let bestItem = null;
        let bestPerformance = -1000;
//        console.log(`Slot ${slotId} (${$dataSystem.equipTypes[etypeId]}):`);
        for (const item of items) {
            const performance = this.calcEquipItemPerformanceEnhanced(item, slotId);
//            console.log(`  ${item.name}: ${performance}`);
            if (performance > bestPerformance) {
                bestItem = item;
                bestPerformance = performance;
            }
        }
//        console.log(`Best item for slot ${slotId}: ${bestItem ? bestItem.name : 'None'} (${bestPerformance})`);
        this.changeEquip(slotId, bestItem);
    };

    // アクターの元ステータス取得
    Game_Actor.prototype.calcOriginalStatus = function() {
        // console.log("calcOriginalStatus 開始");
        this._originalStatus = [];
        for (let i = 0; i < 8; i++) {
            this._originalStatus[i] = this.paramBase(i) + this.paramPlus(i);
            // console.log(`パラメータ ${i}: ${this._originalStatus[i]}`);
        }
        // console.log("calcOriginalStatus 終了");
    };

    Game_Actor.prototype.optimizeEquipmentsEnhanced = function() {
        // console.log("拡張最強装備の計算開始");
        if (!$gameParty || !$gameParty.equipItems) {
            // console.error("$gameParty or equipItems is undefined");
            return;
        }
        const slots = this.equipSlots();
        const equips = [];
        for (let i = 0; i < slots.length; i++) {
            const slotId = slots[i];
            // console.log(`スロットID: ${slotId} の処理`);
            const items = $gameParty.equipItems().filter(item => item.etypeId === slotId);
            // console.log(`対象アイテム数: ${items.length}`);
            const bestItem = this.getBestEquipItemEnhanced(items, slotId);
            // console.log(`選択されたベストアイテム: ${bestItem ? bestItem.name : 'なし'}`);
            equips.push(bestItem);
        }
        this.forceChangeEquip(equips);
        // console.log("拡張最強装備の計算終了");
    };

    Game_Actor.prototype.getBestEquipItemEnhanced = function(items, slotId) {
        // console.log(`getBestEquipItemEnhanced 開始 (スロットID: ${slotId})`);
        if (!items || !Array.isArray(items)) {
            // console.error("Items is undefined or not an array");
            return null;
        }
        let bestItem = null;
        let bestPerformance = -1000;
        for (const item of items) {
            const performance = this.calcEquipItemPerformanceEnhanced(item, slotId);
            // console.log(`アイテム: ${item.name}, 性能値: ${performance}`);
            if (performance > bestPerformance) {
                bestItem = item;
                bestPerformance = performance;
                // console.log(`新しいベストアイテム: ${item.name}`);
            }
        }
        // console.log(`getBestEquipItemEnhanced 終了, 選択: ${bestItem ? bestItem.name : 'なし'}`);
        return bestItem;
    };

    Game_Actor.prototype.calcEquipItemPerformanceEnhanced = function(item, slotId) {
        if (!item) {
            // console.error("Item is undefined in calcEquipItemPerformanceEnhanced");
            return -1000;
        }
//         console.log(`calcEquipItemPerformanceEnhanced 開始 (アイテム: ${item.name})`);
        
        // 基本性能の計算
        let result = this.calcEquipItemPerformanceNormal(item, slotId);
//         console.log(`基本性能: ${result}`);

/*
        // パーセンテージ増加の計算
        result += this.calcEquipItemPerRate(item);
         console.log(`割合計算後: ${result}`);
*/
        
        // 武器タイプ比率の適用
        if (DataManager.isWeapon(item)) {
            result *= this.applyWTypeRatio(item);
//             console.log(`武器タイプ比率適用後: ${result}`);
        }

        // 結果を1桁の小数点に四捨五入する
        result = Math.round(result * 10) / 10;

        // 拡張効果の計算
        if (item.exdataId !== undefined) {
            const enhancedPerformance = this.calcEnhancedItemPerformance(item, '');
            result += enhancedPerformance;
//             console.log(`拡張効果付帯アイテム性能: ${enhancedPerformance}, 合計: ${result}`);
        }

//         console.log(`calcEquipItemPerformanceEnhanced 終了, 最終結果: ${result}`);

        return result;
    };

    Game_Actor.prototype.calcEnhancedItemPerformance = function(item, mode) {
        let result = 0;
        
        if (item && item.traits) {
            for (const trait of item.traits) {
                switch (trait.code) {
                    case Game_BattlerBase.TRAIT_PARAM:
                        const paramId = trait.dataId;
                        const value = trait.value;
                        let increase = this.param(paramId) + this.paramRate(paramId) * this.paramBuffRate(paramId);
                        increase = (increase + item.params[paramId]) * (value - 1);
                        if (mode === 'attack' && paramId === 2) {  // 攻撃力
                            increase *= 1;
                        } else if (mode === 'attack' && paramId === 4) {  // 魔法力
                            increase *= 0;
                        } else if (mode === 'magic' && paramId === 4) {  // 魔法力
                            increase *= 1;
                        } else if (mode === 'magic' && paramId === 2) {  // 攻撃力
                            increase *= 0;
                        } else if (mode === 'defense' && paramId === 2) {  // 攻撃力
                            increase *= 0;
                        } else if (mode === 'defense' && paramId === 4) {  // 魔法力
                            increase *= 0;
                        } else if ((mode === 'attack' || mode === 'magic') && paramId === 3) {  // 防御力
                            increase *= 1;
                        } else if ((mode === 'attack' || mode === 'magic') && paramId === 5) {  // 魔法防御
                            increase *= 1;
                        } else if ((mode === 'attack' || mode === 'magic') && paramId === 0) {  // 最大HP
                            increase *= 1;
                        }
//                        console.log(increase, mode, paramId);
                        result += increase;
                        break;
                    case Game_BattlerBase.TRAIT_XPARAM:
                        switch (trait.dataId) {
                            case 1: // 回避率
                                let evasionValue = trait.value * 100 * evasionRateValue;
                                if (mode === 'defense') {
                                    evasionValue *= 1.5;  // 防御力重視時は回避率の評価を1.5倍に
                                }
                                result += evasionValue;
                                break;
                            case 2: // 会心率
                                let criticalValue = trait.value * 100 * criticalRateValue;
                                if (mode === 'attack') {
                                    criticalValue *= 1.5;  // 攻撃力重視時は会心率の評価を1.5倍に
                                } else if (mode === 'magic') {
                                    criticalValue *= 0.5;  // 魔法力重視時は会心率の評価を半分に
                                } else if (mode === 'defense') {
                                    criticalValue *= 0.5;  // 防御力重視時も会心率の評価を半分に
                                }
                                result += criticalValue;
                                break;
                            case 3: // 会心回避率
                                let criticalEvasionValue = trait.value * 100 * criticalRateValue;
                                if (mode === 'defense') {
                                    criticalEvasionValue *= 1.5;  // 防御力重視時は会心回避率の評価を1.5倍に
                                }
                                result += criticalEvasionValue;
                                break;
                            case 4: // 魔法回避率
                                let magicEvasionValue = trait.value * 100 * magicEvasionRateValue;
                                if (mode === 'defense') {
                                    magicEvasionValue *= 1.5;  // 防御力重視時は魔法回避率の評価を1.5倍に
                                }
                                result += magicEvasionValue;
                                // console.log(`魔法回避率: +${trait.value * 100}%, 評価値: ${trait.value * 100 * magicEvasionRateValue}`);
                                break;
                            case 5: // 魔法反射率
                                let magicReflectionValue = trait.value * 100 * magicReflectionRateValue;
                                if (mode === 'defense') {
                                    magicReflectionValue *= 1.5;  // 防御力重視時は魔法反射率の評価を1.5倍に
                                }
                                result += magicReflectionValue;
                                // console.log(`魔法反射率: +${trait.value * 100}%, 評価値: ${trait.value * 100 * magicReflectionRateValue}`);
                                break;
                            case 6: // 反撃率
                                let counterAttackValue = trait.value * 100 * counterAttackRateValue;
                                if (mode === 'attack') {
                                    counterAttackValue *= 1.5;  // 攻撃力重視時は反撃率の評価を1.5倍に
                                } else if (mode === 'defense') {
                                    counterAttackValue *= 1.5;  // 防御力重視時は反撃率の評価を1.5倍に
                                }
                                result += counterAttackValue;
                                // console.log(`反撃率: +${trait.value * 100}%, 評価値: ${trait.value * 100 * counterAttackRateValue}`);
                                break;
                            case 7: // HP 再生率
                                let hpRegenerationValue = trait.value * 100;
                                if (mode === 'defense') {
                                    hpRegenerationValue *= 1.5;  // 防御力重視時はHP再生率の評価を1.5倍に
                                }
                                result += hpRegenerationValue;
                                break;
                            case 8: // MP 再生率
                                let mpRegenerationValue = trait.value * 100;
                                result += mpRegenerationValue;
                                break;
                            case 9: // TP 再生率
                                let tpRegenerationValue = trait.value * 100;
                                result += tpRegenerationValue;
                                break;
                        }
                        break;
                    case Game_BattlerBase.TRAIT_SPARAM:
//                        switch (trait.dataId) {

//                        }
                        break;
                }
            }
        }
        
//         console.log(`calcEnhancedItemPerformance 終了, 合計: ${result}`);
        return result;
    };

    Game_Actor.prototype.getEnhancedItem = function(addItem) {
        let item = null;
        switch (addItem.type) {
            case 'item':
                item = $dataItems[addItem.id];
                break;
            case 'weapon':
                item = $dataWeapons[addItem.id];
                break;
            case 'armor':
                item = $dataArmors[addItem.id];
                break;
        }
        // console.log(`getEnhancedItem: ${addItem.type} ID ${addItem.id}, 結果: ${item ? item.name : 'なし'}`);
        return item;
    };

    // 割合で計算される装備アイテムの効果
    Game_Actor.prototype.calcEquipItemPerRate = function(item, mode) {
        let result = 0;
        if (item && item.traits) {
            for (const trait of item.traits) {
                if (trait.code === Game_BattlerBase.TRAIT_PARAM) {
                    const paramId = trait.dataId;
                    if (this._originalStatus && this._originalStatus[paramId] !== undefined) {
                        const originValue = this._originalStatus[paramId];
                        let increase = originValue * (trait.value - 1);
                        if (mode === 'attack' && paramId === 2) {  // 攻撃力
                            increase *= 1;
                        } else if (mode === 'attack' && paramId === 4) {  // 魔法力
                            increase *= 0;
                        } else if (mode === 'magic' && paramId === 4) {  // 魔法力
                            increase *= 1;
                        } else if (mode === 'magic' && paramId === 2) {  // 攻撃力
                            increase *= 0;
                        } else if (mode === 'defense' && paramId === 2) {  // 攻撃力
                            increase *= 0;
                        } else if (mode === 'defense' && paramId === 4) {  // 魔法力
                            increase *= 0;
                        } else if ((mode === 'attack' || mode === 'magic') && paramId === 3) {  // 防御力
                            increase *= 1;
                        } else if ((mode === 'attack' || mode === 'magic') && paramId === 5) {  // 魔法防御
                            increase *= 1;
                        } else if ((mode === 'attack' || mode === 'magic') && paramId === 0) {  // 最大HP
                            increase *= 1;
                        }
                        result += increase;
                    }
                }
            }
        }
        return result;
    };

    // 装備計算機能の拡張
/*
    Game_Actor.prototype.calcEquipItemPerformance = function(item) {
        let result = 0;
        if (item && this.attackElements && this.attackElements().includes(item.atypeId)) {
            result += (item.params[2] || 0) + (item.params[4] || 0);  // attack + strength
        }
        if (item && DataManager.isWeapon(item)) {
            result += item.params[2] || 0;  // attack
        }
        return result;
    };
*/

    Game_Actor.prototype.calcEquipItemPerRate = function(item) {
        let result = 0;
        if (item && item.traits) {
            for (const trait of item.traits) {
                if (trait.code === Game_BattlerBase.TRAIT_PARAM && trait.dataId === 2) {  // attack
                    result += trait.value * 100;
                }
            }
        }
        return result;
    };

    Game_Actor.prototype.applyWTypeRatio = function(item) {
        if (!item || !$dataSystem || !$dataSystem.weaponTypes) {
            return 1;
        }
        const wtype = $dataSystem.weaponTypes[item.wtypeId];
        if (wtype && this.traits) {
            let ratio = 1;
            for (const trait of this.traits()) {
                if (trait.code === Game_BattlerBase.TRAIT_WTYPE_ATTACK && trait.dataId === item.wtypeId) {
                    ratio *= trait.value;
                }
            }
            return ratio;
        }
        return 1;
    };

    // 攻撃力重視の装備計算
    Game_Actor.prototype.calcAttackOrientedEquipPerformance = function(item, slotId) {
        if (!item) {
//            console.log("calcAttackOrientedEquipPerformance: Item is null or undefined");
            return 0;
        }

//        console.log(`calcAttackOrientedEquipPerformance for ${item.name}:`);

        // 基本性能の計算
        let result = this.calcEquipItemPerformanceForAttack(item, slotId);
//        console.log(`  Basic performance: ${result}`);
        
/*
        // パーセンテージ増加の計算
        let perRate = this.calcEquipItemPerRate(item, 'attack');
        if (perRate != 0) {
            result = perRate;
        }
        console.log(`  After percentage increase: ${result} (added ${perRate})`);
*/

        // 武器タイプ比率の適用
        if (DataManager.isWeapon(item)) {
            let wTypeRatio = this.applyWTypeRatio(item);
            result *= wTypeRatio;
//            console.log(`  After weapon type ratio: ${result} (multiplied by ${wTypeRatio})`);
        }
        
        // 結果を1桁の小数点に四捨五入する
        let roundedResult = Math.round(result * 10) / 10;
//        console.log(`  Rounded result: ${roundedResult}`);
        
        // 拡張効果の計算
        if (item.exdataId !== undefined) {
            let enhancedPerformance = this.calcEnhancedItemPerformance(item, 'attack');
            roundedResult += enhancedPerformance;
//            console.log(`  After enhanced performance: ${roundedResult} (added ${enhancedPerformance})`);
        }
        
//        console.log(`Final result for ${item.name}: ${roundedResult}`);
        return roundedResult;
    };

    Game_Actor.prototype.calcMagicOrientedEquipPerformance = function(item, slotId) {
        if (!item) {
//            console.log("calcMagicOrientedEquipPerformance: Item is null or undefined");
            return 0;
        }

//        console.log(`calcMagicOrientedEquipPerformance for ${item.name}:`);

        // 基本性能の計算
        let result = this.calcEquipItemPerformanceForMagic(item, slotId);
//        console.log(`  Basic performance: ${result}`);
        
/*
        // パーセンテージ増加の計算
        let perRate = this.calcEquipItemPerRate(item, 'magic');
        if (perRate != 0) {
            result = perRate;
        }
        console.log(`  After percentage increase: ${result} (added ${perRate})`);
*/

        // 武器タイプ比率の適用
        if (DataManager.isWeapon(item)) {
            let wTypeRatio = this.applyWTypeRatio(item);
            result *= wTypeRatio;
//            console.log(`  After weapon type ratio: ${result} (multiplied by ${wTypeRatio})`);
        }
        
        // 結果を1桁の小数点に四捨五入する
        let roundedResult = Math.round(result * 10) / 10;
//        console.log(`  Rounded result: ${roundedResult}`);
        
        // 拡張効果の計算
        if (item.exdataId !== undefined) {
            let enhancedPerformance = this.calcEnhancedItemPerformance(item, 'magic');
            roundedResult += enhancedPerformance;
//            console.log(`  After enhanced performance: ${roundedResult} (added ${enhancedPerformance})`);
        }
        
//        console.log(`Final result for ${item.name}: ${roundedResult}`);
        return roundedResult;
    };

    Game_Actor.prototype.calcDefenseOrientedEquipPerformance = function(item, slotId) {
        if (!item) {
            return 0;
        }

        // 基本性能の計算
        let result = this.calcEquipItemPerformanceForDefense(item, slotId);
        
        // 武器タイプ比率の適用
        if (DataManager.isWeapon(item)) {
            let wTypeRatio = this.applyWTypeRatio(item);
            result *= wTypeRatio;
        }
        
        // 結果を1桁の小数点に四捨五入する
        let roundedResult = Math.round(result * 10) / 10;
        
        // 拡張効果の計算
        if (item.exdataId !== undefined) {
            let enhancedPerformance = this.calcEnhancedItemPerformance(item, 'defense');
            roundedResult += enhancedPerformance;
        }
        
//        console.log(`Final result for ${item.name}: ${roundedResult}`);
        return roundedResult;
    };

    // Window_EquipItemの拡張
    const _Window_EquipItem_makeItemList = Window_EquipItem.prototype.makeItemList;
    Window_EquipItem.prototype.makeItemList = function() {
        _Window_EquipItem_makeItemList.call(this);
        this.sortEquipItems();
        this.moveRemoveOptionToTop();
    };

    Window_EquipItem.prototype.moveRemoveOptionToTop = function() {
        const removeIndex = this._data.indexOf(null);
        if (removeIndex > 0) {
            const removeOption = this._data.splice(removeIndex, 1)[0];
            this._data.unshift(removeOption);
        }
    };

    const _Window_EquipItem_refresh = Window_EquipItem.prototype.refresh;
    Window_EquipItem.prototype.refresh = function() {
        if (this._needsSort) {
            this.sortEquipItems();
            this._needsSort = false;
        }
        _Window_EquipItem_refresh.call(this);
    };

    Window_EquipItem.prototype.sortEquipItems = function() {
        if (this._actor) {
            this._data = this._data.sort((a, b) => {
                if (a === null && b === null) return 0;
                if (a === null) return 1;
                if (b === null) return -1;
                return this.calcEquipItemPerformance(b) - this.calcEquipItemPerformance(a);
            });
        }
    };

    Window_EquipItem.prototype.calcEquipItemPerformance = function(item) {
        if (!this._actor || !item) return 0;
        return this._actor.calcEquipItemPerformanceEnhanced(item, this._slotId);
    };

    const _Window_EquipItem_setSlotId = Window_EquipItem.prototype.setSlotId;
    Window_EquipItem.prototype.setSlotId = function(slotId) {
        if (this._slotId !== slotId) {
            this._slotId = slotId;
// 20240825 軽量化
//            this._needsSort = true;
//            this.makeItemList();
//            this.refresh();
// 20240825 軽量化
            this.scrollTo(0, 0);
            this.select(0);  // カーソルを常に一番上（「外す」オプション）に設定
        }
    };

    // 初期化時にフラグを追加
    const _Window_EquipItem_initialize = Window_EquipItem.prototype.initialize;
    Window_EquipItem.prototype.initialize = function(rect) {
        _Window_EquipItem_initialize.call(this, rect);
        this._needsSort = false;
    };

    // 選択時の処理を拡張
    const _Window_EquipItem_select = Window_EquipItem.prototype.select;
    Window_EquipItem.prototype.select = function(index) {
        _Window_EquipItem_select.call(this, index);
        if (!this._isFirstSelection) {
            const item = this.item();
            if (item) {
                this.updateHelp();
            }
        }
        this._isFirstSelection = false;
    };

    // Game_Actorの拡張
    Game_Actor.prototype.calcEquipItemPerformanceEnhanced = function(item, slotId) {
        if (!item) return 0;

        let result = this.calcEquipItemPerformanceNormal(item, slotId);

        if (DataManager.isWeapon(item)) {
            result *= this.applyWTypeRatio(item);
        }

        result = Math.round(result * 10) / 10;

        if (item.exdataId !== undefined) {
            const enhancedPerformance = this.calcEnhancedItemPerformance(item, '');
            result += enhancedPerformance;
        }

        return result;
    };


})();
