/*:
 * @target MZ
 * @plugindesc キャンセルゲージプラグイン v1.1.0
 * @author Svr
 * @url 
 * @help
 * キャンセルゲージプラグイン v1.1.0
 *
 * 説明:
 * キャンセルボタンまたは右クリックを長押しすると、画面上にゲージが表示され、
 * 指定した時間押し続けると、現在のイベント処理を中断し、
 * 指定したコモンイベントを強制実行します。
 * ゲージが表示される時と最大になった時に、それぞれ設定した効果音が鳴ります。
 *
 * 使用方法:
 * プラグインコマンド「ゲージの有効化」でゲージ機能を有効にし、
 * キャンセルボタンまたは右クリックを長押しすると、ゲージが表示されます。
 * ゲージが満タンになると、指定したコモンイベントが実行されます。
 *
 * ------------------------------------------------------------
 * プラグインコマンド
 * ------------------------------------------------------------
 * ▼ ゲージの有効化
 * キャンセルゲージ機能を有効にします。
 * 中断時に実行するコモンイベントIDを指定できます。
 * 現在のイベントを強制終了するかどうかを選択できます。
 *
 * ▼ ゲージの無効化
 * キャンセルゲージ機能を無効にします。
 *
 * @param design
 * @text ■ 見た目設定
 *
 * @param gaugeWidth
 * @text ゲージの幅
 * @desc ゲージの幅を指定します。
 * @type number
 * @parent design
 * @min 10
 * @max 500
 * @default 200
 *
 * @param gaugeHeight
 * @text ゲージの高さ
 * @desc ゲージの高さを指定します。
 * @type number
 * @parent design
 * @min 5
 * @max 100
 * @default 20
 *
 * @param gaugeColor1
 * @text ゲージの色1
 * @desc ゲージのグラデーション開始色を指定します。
 * @type string
 * @parent design
 * @default rgba(255, 64, 64, 0.8)
 *
 * @param gaugeColor2
 * @text ゲージの色2
 * @desc ゲージのグラデーション終了色を指定します。
 * @type string
 * @parent design
 * @default rgba(255, 200, 64, 0.8)
 *
 * @param gaugePosition
 * @text ゲージの位置
 * @desc ゲージの表示位置を指定します。
 * @type select
 * @option 右上
 * @value rightTop
 * @option 右下
 * @value rightBottom
 * @option 左上
 * @value leftTop
 * @option 左下
 * @value leftBottom
 * @parent design
 * @default rightTop
 *
 * @param gaugeText
 * @text ゲージ説明テキスト
 * @desc ゲージの左側に表示するテキストを指定します。
 * @type string
 * @default キャンセル
 *
 * @param textColor
 * @text テキストの色
 * @desc テキストの色をCSS形式で指定します。例: #ffffff, rgba(255,255,255,1)
 * @type string
 * @default rgba(255, 255, 255, 1)
 *
 * @param textPadding
 * @text テキストとゲージの間隔
 * @desc テキストとゲージの間の余白をピクセル単位で指定します。
 * @type number
 * @min 0
 * @max 50
 * @default 10
 *
 * @param textOffsetY
 * @text テキスト垂直位置調整
 * @desc テキストの垂直方向の位置を調整します。負の値で上に、正の値で下に移動します。
 * @type number
 * @min -50
 * @max 50
 * @default 0
 *
 * @param sound
 * @text ■ 効果音設定
 *
 * @param startSound
 * @text ■ 開始時効果音
 * @parent sound
 *
 * @param startSoundEnabled
 * @text 開始時効果音の有効化
 * @desc ゲージが表示されたときに効果音を鳴らすかどうか設定します。
 * @type boolean
 * @parent startSound
 * @default true
 *
 * @param startSoundName
 * @text 開始時効果音ファイル名
 * @desc ゲージが表示されたときに鳴らす効果音のファイル名を指定します。
 * @type file
 * @dir audio/se
 * @parent startSound
 * @default Cursor3
 *
 * @param startSoundVolume
 * @text 開始時効果音音量
 * @desc 開始時効果音の音量を指定します。
 * @type number
 * @min 0
 * @max 100
 * @parent startSound
 * @default 60
 *
 * @param startSoundPitch
 * @text 開始時効果音ピッチ
 * @desc 開始時効果音のピッチを指定します。
 * @type number
 * @min 50
 * @max 150
 * @parent startSound
 * @default 100
 *
 * @param startSoundPan
 * @text 開始時効果音位相
 * @desc 開始時効果音の位相を指定します。
 * @type number
 * @min -100
 * @max 100
 * @parent startSound
 * @default 0
 *
 * @param completeSound
 * @text ■ 完了時効果音
 * @parent sound
 *
 * @param soundEnabled
 * @text 完了時効果音の有効化
 * @desc ゲージが最大になったときに効果音を鳴らすかどうか設定します。
 * @type boolean
 * @parent completeSound
 * @default true
 *
 * @param soundName
 * @text 完了時効果音ファイル名
 * @desc ゲージが最大になったときに鳴らす効果音のファイル名を指定します。
 * @type file
 * @dir audio/se
 * @parent completeSound
 * @default Decision1
 *
 * @param soundVolume
 * @text 完了時効果音音量
 * @desc 完了時効果音の音量を指定します。
 * @type number
 * @min 0
 * @max 100
 * @parent completeSound
 * @default 90
 *
 * @param soundPitch
 * @text 完了時効果音ピッチ
 * @desc 完了時効果音のピッチを指定します。
 * @type number
 * @min 50
 * @max 150
 * @parent completeSound
 * @default 100
 *
 * @param soundPan
 * @text 完了時効果音位相
 * @desc 完了時効果音の位相を指定します。
 * @type number
 * @min -100
 * @max 100
 * @parent completeSound
 * @default 0
 *
 * @param requiredTime
 * @text 必要時間(秒)
 * @desc キャンセルボタンを何秒間長押しすると処理が実行されるかを指定します。
 * @type number
 * @decimals 1
 * @min 0.1
 * @max 10.0
 * @default 1.0
 *
 * @param debug
 * @text デバッグモード
 * @desc デバッグログを出力するかどうか設定します。
 * @type boolean
 * @default false
 *
 * @command enableGauge
 * @text ゲージの有効化
 * @desc キャンセルゲージを有効にします。
 *
 * @arg commonEventId
 * @text コモンイベントID
 * @desc 中断時に実行するコモンイベントのIDを指定します。0を指定すると実行しません。
 * @type common_event
 * @default 0
 *
 * @arg forceTerminate
 * @text 強制終了
 * @desc 現在のイベントを強制終了するかどうか指定します。
 * @type boolean
 * @default true
 *
 * @command disableGauge
 * @text ゲージの無効化
 * @desc キャンセルゲージを無効にします。
 */

(function() {
    'use strict';
    
    //=============================================================================
    // プラグイン名と読み込み
    //=============================================================================
    const pluginName = document.currentScript.src.split("/").pop().replace(/\.js$/, "");
    const parameters = PluginManager.parameters(pluginName);
    
    //=============================================================================
    // パラメータの取得
    //=============================================================================
    const Param = {
        gaugeWidth: Number(parameters['gaugeWidth'] || 200),
        gaugeHeight: Number(parameters['gaugeHeight'] || 20),
        gaugeColor1: String(parameters['gaugeColor1'] || 'rgba(255, 64, 64, 0.8)'),
        gaugeColor2: String(parameters['gaugeColor2'] || 'rgba(255, 200, 64, 0.8)'),
        gaugePosition: String(parameters['gaugePosition'] || 'rightTop'),
        requiredTime: Number(parameters['requiredTime'] || 1.0),
        debug: parameters['debug'] === 'true', // 文字列の'true'をbooleanのtrueに変換
        gaugeText: String(parameters['gaugeText'] || 'キャンセル'),
        textColor: String(parameters['textColor'] || 'rgba(255, 255, 255, 1)'),
        textFont: String(parameters['textFont'] || '16px GameFont'),
        textPadding: Number(parameters['textPadding'] || 10),
        textOffsetY: Number(parameters['textOffsetY'] || 0),
        // 開始時効果音
        startSoundEnabled: parameters['startSoundEnabled'] !== 'false',
        startSoundName: String(parameters['startSoundName'] || 'Cursor3'),
        startSoundVolume: Number(parameters['startSoundVolume'] || 60),
        startSoundPitch: Number(parameters['startSoundPitch'] || 100),
        startSoundPan: Number(parameters['startSoundPan'] || 0),
        // 完了時効果音（既存のパラメータ）
        soundEnabled: parameters['soundEnabled'] !== 'false', // デフォルトはtrue
        soundName: String(parameters['soundName'] || 'Decision1'),
        soundVolume: Number(parameters['soundVolume'] || 90),
        soundPitch: Number(parameters['soundPitch'] || 100),
        soundPan: Number(parameters['soundPan'] || 0)
    };
    
    //=============================================================================
    // デバッグ関数
    //=============================================================================
    function debugLog(...args) {
        if (Param.debug) {
            console.log('[CancelGauge]', ...args);
        }
    }
    
    //=============================================================================
    // ゲージ管理クラス
    //=============================================================================
    class CancelGaugeManager {
        constructor() {
            this._isEnabled = false;
            this._commonEventId = 0;
            this._forceTerminate = true;
        }
        
        get isEnabled() {
            return this._isEnabled;
        }
        
        get commonEventId() {
            return this._commonEventId;
        }
        
        get forceTerminate() {
            return this._forceTerminate;
        }
        
        enable(commonEventId, forceTerminate) {
            this._isEnabled = true;
            this._commonEventId = commonEventId || 0;
            this._forceTerminate = forceTerminate !== false;
        }
        
        disable() {
            this._isEnabled = false;
            this._commonEventId = 0;
            this._forceTerminate = true;
        }
    }
    
    //=============================================================================
    // キャンセルゲージスプライト
    //=============================================================================
    class Sprite_CancelGauge extends Sprite {
        constructor() {
            super();
            this.bitmap = new Bitmap(Graphics.width, Graphics.height);
            this._createGauge();
            this._rate = 0;
            this._cancelPressed = false;
            this._cancelPressTime = 0;
            this._fadeTimer = 0;
            this._updatePosition();
            this.visible = false;
            this._soundPlayed = false; // 効果音を再生したかどうかのフラグ
            this._startSoundPlayed = false; // 開始音を再生したかどうかのフラグ
            
            // この行を追加: Z座標を非常に高い値に設定して最前面に表示
            this.z = 9999;
        }
        
        /**
         * テキストの作成
         * @private
         */
        _createText() {
            if (!Param.gaugeText) return;
            
            // テキスト用のスプライトを作成
            this._textSprite = new Sprite(new Bitmap(512, 64));
            this.addChild(this._textSprite);
            
            // 初回描画
            this._updateTextBitmap();
        }

        /**
         * テキストビットマップの更新
         * @private
         */
        _updateTextBitmap() {
            if (!this._textSprite || !Param.gaugeText) return;
            
            const bitmap = this._textSprite.bitmap;
            bitmap.clear();
            
            // フォントをゲームのメインフォントに設定
            bitmap.fontFace = $gameSystem ? $gameSystem.mainFontFace() : 'GameFont';
            bitmap.fontSize = 16; // 適切なサイズに調整
            bitmap.textColor = Param.textColor;
            bitmap.outlineWidth = 4;
            bitmap.outlineColor = 'rgba(0, 0, 0, 0.8)';
            
            // テキストを描画
            bitmap.drawText(Param.gaugeText, 0, 0, 512, 32, 'left');
            
            // テキスト幅を測定して保存
            this._textWidth = bitmap.measureTextWidth(Param.gaugeText) + Param.textPadding;
            
            // テキスト描画後に位置を更新
            this._updateGaugePosition();
        }

        /**
         * ゲージの位置を更新
         * @private
         */
        _updateGaugePosition() {
            if (!this._background || !this._foreground) return;
            
            // テキストがある場合は背景とゲージの位置を調整
            this._background.x = this._textWidth;
            this._foreground.x = this._textWidth;
            
            // テキストの垂直位置をゲージの中央に合わせる
            if (this._textSprite) {
                this._textSprite.y = (Param.gaugeHeight - 24) / 2; // 24は大体の文字高さ
            }
        }
        
        /**
         * 背景の作成
         * @private
         */
        _createBackground() {
            const width = Param.gaugeWidth;
            const height = Param.gaugeHeight;
            
            this._background = new Sprite(new Bitmap(width, height));
            this._background.bitmap.fillAll('rgba(0, 0, 0, 0.7)');
            this.addChild(this._background);
            
            // テキストがある場合は背景の位置を調整
            if (this._textWidth > 0) {
                this._background.x = this._textWidth;
            }
        }
        
        /**
         * ゲージの作成
         * @private
         */
        _createForeground() {
            const width = Param.gaugeWidth;
            const height = Param.gaugeHeight;
            
            this._foreground = new Sprite(new Bitmap(width, height));
            this._foreground.bitmap.gradientFillRect(
                0, 
                0, 
                width, 
                height, 
                Param.gaugeColor1,
                Param.gaugeColor2, 
                true
            );
            this.addChild(this._foreground);
            this._foreground.scale.x = 0;
            
            // テキストがある場合はゲージの位置を調整
            if (this._textWidth > 0) {
                this._foreground.x = this._textWidth;
            }
        }
        
        /**
         * ゲージの作成
         * @private
         */
        _createGauge() {
            debugLog("ゲージ作成開始");
            
            // ゲージ背景
            this._background = new Sprite();
            this._background.bitmap = new Bitmap(Param.gaugeWidth, Param.gaugeHeight);
            this._background.bitmap.fillAll('rgba(0, 0, 0, 0.7)');
            this.addChild(this._background);
            
            // ゲージ前景
            this._foreground = new Sprite();
            this._foreground.bitmap = new Bitmap(Param.gaugeWidth, Param.gaugeHeight);
            this._foreground.bitmap.gradientFillRect(
                0, 
                0, 
                Param.gaugeWidth, 
                Param.gaugeHeight, 
                Param.gaugeColor1,
                Param.gaugeColor2, 
                true
            );
            this.addChild(this._foreground);
            this._foreground.scale.x = 0;
            
            // テキスト表示用スプライト - 常に作成する
            // デフォルトテキストを設定
            const displayText = Param.gaugeText || "キャンセル";
            debugLog("表示テキスト:", displayText);
            
            this._text = new Sprite();
            this._text.bitmap = new Bitmap(256, 32);
            this._text.bitmap.fontFace = $gameSystem ? $gameSystem.mainFontFace() : 'GameFont';
            this._text.bitmap.fontSize = 16;
            this._text.bitmap.textColor = Param.textColor || 'rgba(255, 255, 255, 1)';
            this._text.bitmap.outlineWidth = 3;
            this._text.bitmap.outlineColor = 'rgba(0, 0, 0, 0.8)';
            this._text.bitmap.drawText(displayText, 0, 0, 256, 32, 'left');
            this._text.y = (Param.gaugeHeight - 16) / 2 + Param.textOffsetY;
            
            // テキスト幅を取得
            this._textWidth = this._text.bitmap.measureTextWidth(displayText) + (Param.textPadding || 10);
            debugLog("テキスト幅:", this._textWidth);
            
            // ゲージの位置を調整
            this._background.x = this._textWidth;
            this._foreground.x = this._textWidth;
            
            this.addChild(this._text);
            
            debugLog("ゲージ作成完了");
        }
        
        /**
         * 位置の更新
         * @private
         */
        _updatePosition() {
            const width = Param.gaugeWidth;
            const height = Param.gaugeHeight;
            const padding = 10;
            const totalWidth = width + this._textWidth;
            
            switch (Param.gaugePosition) {
                case 'rightTop':
                    this.x = Graphics.width - totalWidth - padding;
                    this.y = padding;
                    break;
                case 'rightBottom':
                    this.x = Graphics.width - totalWidth - padding;
                    this.y = Graphics.height - height - padding;
                    break;
                case 'leftTop':
                    this.x = padding;
                    this.y = padding;
                    break;
                case 'leftBottom':
                    this.x = padding;
                    this.y = Graphics.height - height - padding;
                    break;
            }
        }
        
        /**
         * ゲージの更新
         */
        update() {
            super.update();
            
            if (!$gameSystem.isCancelGaugeEnabled()) {
                this.visible = false;
                this._cancelPressed = false;
                this._cancelPressTime = 0;
                this._rate = 0;
                this._foreground.scale.x = 0;
                this._soundPlayed = false; // 効果音フラグをリセット
                this._startSoundPlayed = false; // 開始音フラグをリセット
                return;
            }
            
            // メニューが開いている場合はゲージを表示しない
            if (SceneManager.isSceneChanging() || this.isMenuScene()) {
                this.visible = false;
                return;
            }
            
            // フェードアウト処理
            if (this._fadeTimer > 0) {
                this._fadeTimer--;
                if (this._fadeTimer === 0) {
                    this.visible = false;
                }
            }
            
            // ゲージとテキストの更新
            this._foreground.scale.x = this._rate;
            
            // キャンセルボタンの状態更新
            this.updateCancelButton();
            
            // テキストが可視状態になったとき、もう一度テキストを更新
            if (this.visible && !this._wasVisible) {
                this._updateTextBitmap();
            }
            
            this._wasVisible = this.visible;
        }
        
        /**
         * 開始効果音の再生
         */
        playStartSound() {
            if (Param.startSoundEnabled && Param.startSoundName) {
                debugLog(`開始効果音再生: ${Param.startSoundName}`);
                AudioManager.playSe({
                    name: Param.startSoundName,
                    volume: Param.startSoundVolume,
                    pitch: Param.startSoundPitch,
                    pan: Param.startSoundPan
                });
            }
        }
        
        /**
         * キャンセルボタンの更新
         */
        updateCancelButton() {
            // キャンセルボタンが押されているか（右クリックは除外）
            // TouchInput.isCancelled()は右クリックも検知するので、Input.isPressed('cancel')のみを使用
            const cancelPressed = Input.isPressed('cancel');
            
            // キャンセルボタンが押されたとき
            if (!this._cancelPressed && cancelPressed) {
                debugLog("キャンセル押下検知"); // デバッグ用
                this._cancelPressed = true;
                this._cancelPressTime = 0;
                // 強制的に表示
                this.visible = true;
                this.opacity = 255;
                this._soundPlayed = false; // 効果音フラグをリセット
                
                // 開始効果音を再生
                if (!this._startSoundPlayed) {
                    this.playStartSound();
                    this._startSoundPlayed = true;
                }
            }
            
            // キャンセルボタンが離されたとき
            if (this._cancelPressed && !cancelPressed) {
                this._cancelPressed = false;
                this.prepareHide();
                this._soundPlayed = false; // 効果音フラグをリセット
                this._startSoundPlayed = false; // 開始音フラグをリセット
            }
            
            // キャンセルボタンが押されている間
            if (this._cancelPressed) {
                this._cancelPressTime++;
                const requiredFrames = Math.floor(Param.requiredTime * 60);
                const rate = Math.min(this._cancelPressTime / requiredFrames, 1);
                this._rate = rate;
                this._foreground.scale.x = rate;
                
                // 強制的に表示
                this.visible = true;
                this.opacity = 255;
                
                // 10%ごとにデバッグログを出力
                if (Math.floor(rate * 10) > Math.floor((rate - (1 / requiredFrames)) * 10)) {
                    debugLog(`ゲージ率: ${Math.floor(rate * 100)}%`); 
                }
                
                // ゲージが最大に近づいたら効果音を再生（99%以上で）
                if (rate >= 0.99 && !this._soundPlayed && Param.soundEnabled) {
                    this.playCancelSound();
                    this._soundPlayed = true; // 効果音を再生したフラグを立てる
                }
                
                // 必要時間押されたら実行
                if (this._cancelPressTime >= requiredFrames) {
                    debugLog("ゲージ最大 - コモンイベント実行");
                    this.executeCommonEvent();
                    this._cancelPressed = false;
                    this.reset();
                }
            }
        }
        
        /**
         * キャンセル効果音の再生
         */
        playCancelSound() {
            if (Param.soundEnabled && Param.soundName) {
                debugLog(`効果音再生: ${Param.soundName}`);
                AudioManager.playSe({
                    name: Param.soundName,
                    volume: Param.soundVolume,
                    pitch: Param.soundPitch,
                    pan: Param.soundPan
                });
            }
        }

        /**
         * 現在のシーンがメニュー系のシーンかどうかを判定
         * @returns {boolean}
         */
        isMenuScene() {
            const scene = SceneManager._scene;
            return (
                scene instanceof Scene_Menu ||
                scene instanceof Scene_Item ||
                scene instanceof Scene_Skill ||
                scene instanceof Scene_Equip ||
                scene instanceof Scene_Status ||
                scene instanceof Scene_Options ||
                scene instanceof Scene_Save ||
                scene instanceof Scene_Load ||
                scene instanceof Scene_GameEnd ||
                scene instanceof Scene_Shop
            );
        }
        
        /**
         * 右クリック状態の取得（常にfalseを返す）
         * @returns {boolean}
         */
        isRightPressed() {
            return false; // 右クリックには対応しない
        }
        
        /**
         * 現在アクティブなインタープリタを取得
         * @returns {Game_Interpreter}
         */
        getCurrentActiveInterpreter() {
            let interpreter = $gameMap._interpreter;
            if (!interpreter) return null;
            
            // 最も深い階層の実行中のインタープリタを探す
            while (interpreter._childInterpreter && interpreter._childInterpreter.isRunning()) {
                interpreter = interpreter._childInterpreter;
            }
            
            return interpreter;
        }
        
        /**
         * インタープリター終了検出用のメソッド
         */
        _setupInterpreterCompletion(interpreter, callback) {
            // 元のupdateメソッドを保存
            const originalUpdate = interpreter.update;
            
            // updateメソッドをオーバーライド
            interpreter.update = function() {
                const wasRunning = this.isRunning();
                // 元のupdateを呼び出し
                originalUpdate.call(this);
                // 実行状態が変わったかチェック
                if (wasRunning && !this.isRunning()) {
                    // 実行が終了したらコールバックを呼び出し
                    callback();
                    // 元のメソッドに戻す
                    this.update = originalUpdate;
                }
            };
        }

        /**
         * コモンイベントの実行
         */
        executeCommonEvent() {
            const commonEventId = $gameSystem.cancelGaugeCommonEventId();
            const forceTerminate = $gameSystem.isCancelGaugeForceTerminate();
            debugLog(`コモンイベント実行: ID=${commonEventId}, 強制終了=${forceTerminate}`);
            
            // ゲージを無効化
            debugLog("ゲージを無効化");
            $gameSystem.disableCancelGauge();
            
            // 強制終了前のインタープリターを保存
            const interpreterToTerminate = forceTerminate ? this.getCurrentActiveInterpreter() : null;
            
            // コモンイベントの実行（指定されている場合）
            if (commonEventId > 0) {
                const commonEvent = $dataCommonEvents[commonEventId];
                if (commonEvent) {
                    debugLog(`コモンイベント ${commonEventId} を即時実行`);
                    
                    // 現在のインタープリターを保存
                    const originalInterpreter = $gameMap._interpreter;
                    
                    // 新しいインタープリターを作成
                    const newInterpreter = new Game_Interpreter();
                    
                    // コモンイベントをセットアップ
                    newInterpreter.setup(commonEvent.list);
                    
                    // コモンイベント終了後の処理
                    const endCallback = () => {
                        debugLog("コモンイベント終了");
                        
                        // 元のインタープリターに戻す
                        $gameMap._interpreter = originalInterpreter;
                        
                        // コモンイベント終了後に強制終了を実行
                        if (forceTerminate && interpreterToTerminate && interpreterToTerminate.isRunning()) {
                            debugLog("現在実行中の処理を強制終了します（呼び出し元は継続）");
                            // 現在の処理位置を最後まで進めることで、現在の処理のみを終了
                            // 子インタープリタがある場合はそれも終了
                            if (interpreterToTerminate._childInterpreter) {
                                interpreterToTerminate._childInterpreter.terminate();
                                interpreterToTerminate._childInterpreter = null;
                            }
                            // インデックスを最後まで進める（現在の処理のみ終了）
                            interpreterToTerminate._index = interpreterToTerminate._list.length;
                        }
                        
                        // ゲージを無効化する処理
                        debugLog("ゲージを無効化");
                        $gameSystem.disableCancelGauge();
                    };
                    
                    // 終了検出用の処理を追加
                    this._setupInterpreterCompletion(newInterpreter, endCallback);
                    
                    // インタープリターを入れ替え
                    $gameMap._interpreter = newInterpreter;
                    
                    // 即座に1フレーム実行して開始
                    newInterpreter.update();
                } else {
                    debugLog(`コモンイベント ${commonEventId} が見つかりません`);
                    
                    // コモンイベントが見つからない場合でも強制終了を実行
                    if (forceTerminate && interpreterToTerminate && interpreterToTerminate.isRunning()) {
                        debugLog("現在実行中の処理を強制終了します");
                        if (interpreterToTerminate._childInterpreter) {
                            interpreterToTerminate._childInterpreter.terminate();
                            interpreterToTerminate._childInterpreter = null;
                        }
                        interpreterToTerminate._index = interpreterToTerminate._list.length;
                    }
                    
                    // ゲージを無効化
                    debugLog("ゲージを無効化");
                    $gameSystem.disableCancelGauge();
                }
            } else {
                // コモンイベントIDが0の場合は即座に強制終了
                if (forceTerminate && interpreterToTerminate && interpreterToTerminate.isRunning()) {
                    debugLog("現在実行中の処理を強制終了します");
                    if (interpreterToTerminate._childInterpreter) {
                        interpreterToTerminate._childInterpreter.terminate();
                        interpreterToTerminate._childInterpreter = null;
                    }
                    interpreterToTerminate._index = interpreterToTerminate._list.length;
                }
                
                // ゲージを無効化
                debugLog("ゲージを無効化");
                $gameSystem.disableCancelGauge();
            }

            // メッセージシステムを完全にリセットして次へ
            const scene = SceneManager._scene;
            const messageWindow = scene._messageWindow;
            const interpreter = $gameMap._interpreter;
            
            // メッセージウィンドウの完全リセット
            if (messageWindow) {
                messageWindow.terminateMessage();
                messageWindow._textState = null;
                messageWindow._waitCount = 0;
                messageWindow._pauseSkip = false;
                messageWindow._showFast = false;
                messageWindow._lineShowFast = false;
                messageWindow.pause = false;
                
                // ウィンドウを即座に閉じる
                messageWindow.close();
                messageWindow.openness = 0;
            }
            
            // $gameMessageをクリア
            $gameMessage.clear();
            
            // インタープリターを進める
            if (interpreter && interpreter.isRunning()) {
                // 待機状態をクリア
                interpreter._waitMode = '';
                interpreter._waitCount = 0;
                
                // メッセージコマンドをスキップ
                let skipped = false;
                const command = interpreter.currentCommand();
                if (command && command.code === 101) {
                    // 文章の表示コマンドとその続き(401)をスキップ
                    interpreter._index++;
                    while (interpreter._index < interpreter._list.length) {
                        const nextCmd = interpreter._list[interpreter._index];
                        if (nextCmd && nextCmd.code === 401) {
                            interpreter._index++;
                        } else {
                            break;
                        }
                    }
                    skipped = true;
                }
                
                // インタープリターを数フレーム実行
                for (let i = 0; i < 3; i++) {
                    interpreter.update();
                    if (messageWindow) messageWindow.update();
                }
                
//                    console.log(skipped ? "メッセージをスキップして次へ進みました" : "インタープリターを更新しました");
            }
        }
        
        /**
         * ゲージの値を設定
         * @param rate
         */
        setRate(rate) {
            this._rate = rate;
            // 強制的に表示するように変更
            if (rate > 0 && $gameSystem.isCancelGaugeEnabled()) {
                this.visible = true;
            }
        }
        
        /**
         * ゲージを非表示にする準備をする
         */
        prepareHide() {
            this._fadeTimer = 30; // 約0.5秒
        }
        
        /**
         * ゲージを初期化する
         */
        reset() {
            this._rate = 0;
            this._foreground.scale.x = 0;
            this.visible = false;
            this._fadeTimer = 0;
            this._cancelPressed = false;
            this._cancelPressTime = 0;
            this._soundPlayed = false; // 効果音フラグをリセット
            this._startSoundPlayed = false; // 開始音フラグをリセット
        }
    }
    
    //=============================================================================
    // Game_System拡張
    //=============================================================================
    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function() {
        _Game_System_initialize.call(this);
        this._cancelGaugeEnabled = false;
        this._cancelGaugeCommonEventId = 0;
        this._cancelGaugeForceTerminate = true; // 強制終了フラグを追加
    };
    
    Game_System.prototype.isCancelGaugeEnabled = function() {
        return this._cancelGaugeEnabled;
    };
    
    Game_System.prototype.cancelGaugeCommonEventId = function() {
        return this._cancelGaugeCommonEventId;
    };

    // 強制終了フラグのゲッター
    Game_System.prototype.isCancelGaugeForceTerminate = function() {
        return this._cancelGaugeForceTerminate !== false; // undefined対策
    };
    
    Game_System.prototype.enableCancelGauge = function(commonEventId, forceTerminate) {
        this._cancelGaugeEnabled = true;
        this._cancelGaugeCommonEventId = commonEventId || 0;
        this._cancelGaugeForceTerminate = forceTerminate !== false; // デフォルトはtrue
    };
    
    Game_System.prototype.disableCancelGauge = function() {
        this._cancelGaugeEnabled = false;
        this._cancelGaugeCommonEventId = 0;
        this._cancelGaugeForceTerminate = true; // デフォルトに戻す
    };
    
    //=============================================================================
    // Scene_Base拡張
    //=============================================================================
    const _Scene_Base_create = Scene_Base.prototype.create;
    Scene_Base.prototype.create = function() {
        _Scene_Base_create.call(this);
        this.createCancelGauge();
    };
    
    Scene_Base.prototype.createCancelGauge = function() {
        debugLog(`${this.constructor.name} - キャンセルゲージを作成`);
        this._cancelGauge = new Sprite_CancelGauge();
        
        // 最前面に配置するための新しい方法
        if (this._windowLayer && this._windowLayer.parent) {
            this._windowLayer.parent.addChild(this._cancelGauge);
        } else {
            this.addChild(this._cancelGauge);
        }
    };
    
    //=============================================================================
    // Scene_Map拡張 - 特に地図シーンでの表示を確実にする
    //=============================================================================
    const _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
    Scene_Map.prototype.createDisplayObjects = function() {
        _Scene_Map_createDisplayObjects.call(this);
        
        // この部分は削除してcreateAllWindowsだけに統一
        // キャンセルゲージは createAllWindows で作成される
    };
    
    const _SceneManager_updateScene = SceneManager.updateScene;
    SceneManager.updateScene = function() {
        _SceneManager_updateScene.call(this);
        
        // 現在のシーンとキャンセルゲージが存在する場合、最前面に持ってくる
        const scene = this._scene;
        if (scene && scene._cancelGauge) {
            if (scene.children.includes(scene._cancelGauge)) {
                // 一度削除して再追加することで最前面に
                scene.removeChild(scene._cancelGauge);
                scene.addChild(scene._cancelGauge);
            }
        }
    };
    //=============================================================================
    // Scene_Map拡張 - 特に地図シーンでの表示を確実にする
    //=============================================================================
    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_createAllWindows.call(this);
        
        debugLog("Scene_Map - キャンセルゲージを作成");
        // ゲージがすでに作成されていれば削除
        if (this._cancelGauge) {
            this.removeChild(this._cancelGauge);
        }
        
        // 最前面に配置するための新しい方法
        this._cancelGauge = new Sprite_CancelGauge();
        
        // 新しい表示コンテナにゲージを追加
        // 重要: SceneManager.currentSceneではなく、thisを使用
        if (this._windowLayer && this._windowLayer.parent) {
            this._windowLayer.parent.addChild(this._cancelGauge);
        } else {
            this.addChild(this._cancelGauge);
        }
    };
    
    // Game_Interpreterクラスに追加するメソッド
    Game_Interpreter.prototype.setCommentCallback = function(callback) {
        this._commentCallback = callback;
    };

    // Game_Interpreterのupdateメソッドをオーバーライド
    const _Game_Interpreter_update = Game_Interpreter.prototype.update;
    Game_Interpreter.prototype.update = function() {
        const isRunning = this.isRunning();
        _Game_Interpreter_update.call(this);
        
        // 実行が終了したときにコールバックを呼び出す
        if (isRunning && !this.isRunning() && this._commentCallback) {
            this._commentCallback();
            this._commentCallback = null;
        }
    };
    
    //=============================================================================
    // プラグインコマンド
    //=============================================================================
    PluginManager.registerCommand(pluginName, "enableGauge", args => {
        const commonEventId = Number(args.commonEventId) || 0;
        const forceTerminate = args.forceTerminate !== "false"; // 文字列の"false"の場合だけfalseに
        $gameSystem.enableCancelGauge(commonEventId, forceTerminate);
    });
    
    PluginManager.registerCommand(pluginName, "disableGauge", args => {
        $gameSystem.disableCancelGauge();
    });
    
})();