/*:ja
 * @plugindesc PIXIJSテクスチャエラー修正プラグイン
 * @author YourName
 *
 * @help
 * このプラグインはRPGツクールMZでの高速背景切り替え時に発生する
 * 以下のエラーを修正します：
 * - Cannot read property 'resolution' of null
 * - Cannot read property 'isPowerOfTwo' of null
 *
 * 使用方法：
 * プラグインをプロジェクトに追加し、他のプラグインより上位に配置してください。
 */

(function() {
    'use strict';
    
    //====================================================================
    // TextureMatrix関連の修正
    //====================================================================
    
    // オリジナルのTextureMatrix.updateメソッドをバックアップ
    const _TextureMatrix_update = PIXI.TextureMatrix.prototype.update;
    
    // 安全チェックを含むようにupdateメソッドをオーバーライド
    PIXI.TextureMatrix.prototype.update = function(forceUpdate) {
        // テクスチャまたはbaseTextureが存在しない場合は処理をスキップ
        if (!this.texture || !this.texture.baseTexture) {
            return false;
        }
        // 元のメソッドを呼び出す
        return _TextureMatrix_update.call(this, forceUpdate);
    };
    
    //====================================================================
    // TilingSprite関連の修正
    //====================================================================
    
    // オリジナルのTilingSprite._renderメソッドをバックアップ
    const _TilingSprite_render = PIXI.extras.TilingSprite.prototype._render;
    
    // 安全チェックを含むように_renderメソッドをオーバーライド
    PIXI.extras.TilingSprite.prototype._render = function(renderer) {
        // テクスチャがnullまたは無効な場合は処理をスキップ
        if (!this.texture || !this.texture.valid) {
            return;
        }
        // 元のメソッドを呼び出す
        _TilingSprite_render.call(this, renderer);
    };
    
    //====================================================================
    // TilingSpriteRenderer関連の修正（isPowerOfTwoエラー対策）
    //====================================================================
    
    // オリジナルのTilingSpriteRenderer.renderメソッドをバックアップ
    const _TilingSpriteRenderer_render = PIXI.extras.TilingSpriteRenderer.prototype.render;
    
    // 安全チェックを含むようにrenderメソッドをオーバーライド
    PIXI.extras.TilingSpriteRenderer.prototype.render = function(ts) {
        // テクスチャがnullまたは無効な場合は処理をスキップ
        if (!ts.texture || !ts.texture.valid || !ts.texture.baseTexture) {
            return;
        }
        // 元のメソッドを呼び出す
        _TilingSpriteRenderer_render.call(this, ts);
    };
    
    //====================================================================
    // Sprite関連の追加安全対策
    //====================================================================
    
    // オリジナルのSprite._renderWebGLメソッドをバックアップ
    const _Sprite_renderWebGL = Sprite.prototype._renderWebGL;
    
    // 安全チェックを含むように_renderWebGLメソッドをオーバーライド
    Sprite.prototype._renderWebGL = function(renderer) {
        // テクスチャがnullまたは無効な場合は処理をスキップ
        if (this.texture && (!this.texture.valid || !this.texture.baseTexture)) {
            return;
        }
        // 元のメソッドを呼び出す
        _Sprite_renderWebGL.call(this, renderer);
    };
    
    //====================================================================
    // Scene_Base拡張 - 背景切り替え時の安全策
    //====================================================================
    
    // オリジナルのclearBackgroundメソッドをバックアップ
    const _Scene_Base_clearBackground = Scene_Base.prototype.clearBackground;
    
    // 背景クリア時に追加の安全チェックを行う
    Scene_Base.prototype.clearBackground = function() {
        // 元のメソッドを呼び出す
        _Scene_Base_clearBackground.call(this);
        
        // 背景をクリアした後、確実にテクスチャを解放するための遅延処理
        if (this._backgroundSprite && this._backgroundSprite.texture) {
            // テクスチャを無効化して明示的にGCを促す
            this._backgroundSprite.texture.valid = false;
            this._backgroundSprite.texture = null;
        }
    };
})();