/*:
 * @target MZ
 * @plugindesc コモンイベント1001番から1100番までの変数を処理し、イベント補正、依頼補正、依頼達成回数補正を適用して計算します。Svr_プレフィックス付き。
 * @author スタジオVR (Modified)
 *
 * @command countStarsAndCalculate
 * @text 星数を変換＆計算
 * @desc 変数3201番から3300番の数値を「★」に変換し、各種補正値を適用して計算します。
 *
 * @help
 * ----------------------------------------------------------------------------
 * プラグインコマンド:
 *   Svr_CountStarsInCommonEvents countStarsAndCalculate
 * 
 * 説明:
 *   このプラグインは、以下の処理を行います：
 * 
 *   1. コモンイベント1001番から1100番までに対応する変数3201番から3300番の
 *      数値を「★」に変換し、それぞれ対応する変数3101番から3200番に格納します。
 * 
 *   2. 変数3201番から3300番の数値に対して、カーブ計算式を適用し、
 *      イベント補正、依頼補正、依頼達成回数補正を適用して計算し、
 *      結果を変数3301番から3400番に格納します。
 * 
 *   3. 各コモンイベントの注釈に記述された <Var2490: 数式> を解析し、
 *      結果を対応する変数3601番から3700番に格納します。
 * 
 *   4. アイテムID1001番から1100番までのアイテム名を取得し、対応する
 *      変数3501番から3600番に格納します。
 * 
 *   5. 理論上の最大値を計算し、変数4601番から4700番に格納します。
 * 
 * 注意:
 *   - 変数3101番から3200番は「★」の数を格納するために使用します。
 *   - 変数3301番から3400番は計算結果を格納するために使用します。
 *   - 変数3401番から3500番は依頼達成回数を格納するために使用します。
 *   - 変数3501番から3600番はアイテム名を格納するために使用します。
 *   - 変数3601番から3700番は<Var2490: 数式>の値を格納するために使用します。
 *   - 変数4601番から4700番は理論上の最大値を格納するために使用します。
 *   - 変数2488は報酬の基準値乗算に使用します。
 *   - 変数4079は依頼G報酬の倍率を設定します。
 *   - 変数4080は依頼G報酬のカーブ係数を設定します。
 *   - 変数4078は依頼G報酬のクリア回数補正値を設定します。
 *   - スイッチ1003番が有効（ON）のときのみ、コンソールログが表示されます。
 * 
 * ----------------------------------------------------------------------------
 */

(() => {
    const pluginName = "Svr_CountStarsInCommonEvents";

    PluginManager.registerCommand(pluginName, "countStarsAndCalculate", args => {
        const startCommonEventId = 1001;
        const endCommonEventId = 1100;
        const inputVariableStartId = 3201;
        const inputVariableEndId = 3300;
        const starVariableStartId = 3101;
        const starVariableEndId = 3200;
        const calculationVariableStartId = 3301;
        const calculationVariableEndId = 3400;
        const questClearVariableStartId = 3401;
        const questClearVariableEndId = 3500;
        const itemIdStart = 1001;
        const itemIdEnd = 1100;
        const itemNameVariableStartId = 3501;
        const itemNameVariableEndId = 3600;
        const var2490VariableStartId = 3601;
        const var2490VariableEndId = 3700;
        const loggingSwitchId = 1003;
        const totalEvents = endCommonEventId - startCommonEventId + 1;

        // 範囲の確認
        if ((inputVariableEndId - inputVariableStartId + 1) < totalEvents) {
            if ($gameSwitches.value(loggingSwitchId)) {
                console.warn(`入力変数の範囲（${inputVariableStartId}～${inputVariableEndId}）が必要な数（${totalEvents}）に満たない可能性があります。`);
            }
        }

        if ((starVariableEndId - starVariableStartId + 1) < totalEvents) {
            if ($gameSwitches.value(loggingSwitchId)) {
                console.warn(`星変換変数の範囲（${starVariableStartId}～${starVariableEndId}）が必要な数（${totalEvents}）に満たない可能性があります。`);
            }
        }

        if ((calculationVariableEndId - calculationVariableStartId + 1) < totalEvents) {
            if ($gameSwitches.value(loggingSwitchId)) {
                console.warn(`計算結果変数の範囲（${calculationVariableStartId}～${calculationVariableEndId}）が必要な数（${totalEvents}）に満たない可能性があります。`);
            }
        }

        if ((questClearVariableEndId - questClearVariableStartId + 1) < totalEvents) {
            if ($gameSwitches.value(loggingSwitchId)) {
                console.warn(`依頼達成回数変数の範囲（${questClearVariableStartId}～${questClearVariableEndId}）が必要な数（${totalEvents}）に満たない可能性があります。`);
            }
        }

        if ((itemNameVariableEndId - itemNameVariableStartId + 1) < (itemIdEnd - itemIdStart + 1)) {
            if ($gameSwitches.value(loggingSwitchId)) {
                console.warn(`アイテム名変数の範囲（${itemNameVariableStartId}～${itemNameVariableEndId}）が必要な数（${itemIdEnd - itemIdStart + 1}）に満たない可能性があります。`);
            }
        }

        if ((var2490VariableEndId - var2490VariableStartId + 1) < totalEvents) {
            if ($gameSwitches.value(loggingSwitchId)) {
                console.warn(`Var2490変数の範囲（${var2490VariableStartId}～${var2490VariableEndId}）が必要な数（${totalEvents}）に満たない可能性があります。`);
            }
        }

        // 1. 変数3201番から3300番の数値を「★」に変換し、変数3101番から3200番に格納
        for (let i = 0; i < totalEvents; i++) {
            const commonEventId = startCommonEventId + i;
            const inputVariableId = inputVariableStartId + i;
            const starVariableId = starVariableStartId + i;

            const inputValue = $gameVariables.value(inputVariableId);
            const stars = '★'.repeat(inputValue);
            $gameVariables.setValue(starVariableId, stars);

            if ($gameSwitches.value(loggingSwitchId)) {
                console.log(`コモンイベントID ${commonEventId}:`);
                console.log(`  変数ID ${inputVariableId} の値 (${inputValue}) を「★」に変換して、変数ID ${starVariableId} に格納しました。`);
            }
        }

        // 2. 各コモンイベントの注釈を読み込み、基本計算と他の補正を先に実行
        let tempCalculations = []; // 計算結果を一時的に保持する配列
        for (let i = 0; i < totalEvents; i++) {
            const commonEventId = startCommonEventId + i;
            const inputVariableId = inputVariableStartId + i;
            const calculationVariableId = calculationVariableStartId + i;
            const questClearVariableId = questClearVariableStartId + i;

            const commonEvent = $dataCommonEvents[commonEventId];
            if (!commonEvent) {
                if ($gameSwitches.value(loggingSwitchId)) {
                    console.warn(`コモンイベントID ${commonEventId} が存在しません。変数ID ${calculationVariableId} には0を格納します。`);
                }
                tempCalculations[i] = 0;
                continue;
            }

            let requestCorrection = 0;
            let questClearCorrection = null;

            // イベント補正以外の補正値を注釈から読み取る
            for (const command of commonEvent.list) {
                if (command.code === 108 || command.code === 408) {
                    const comment = command.parameters[0];
                    const requestMatch = comment.match(/<RequestCorrection:\s*(.+?)\s*>/i);
                    const questClearMatch = comment.match(/<QuestClearCorrection:\s*(.+?)\s*>/i);
                    if (requestMatch) {
                        requestCorrection = evaluateCorrection(requestMatch[1]);
                    }
                    if (questClearMatch) {
                        questClearCorrection = evaluateCorrection(questClearMatch[1]);
                    }
                }
            }

            // 基本値と補正値の取得
            const baseValue = $gameVariables.value(inputVariableId);
            const baseMultiplier = $gameVariables.value(2488);
            const clearCount = $gameVariables.value(questClearVariableId);
            const multiplier = $gameVariables.value(4079);
            const curveFactor = $gameVariables.value(4080);
            const defaultClearBonus = $gameVariables.value(4078);
            const actualClearBonus = questClearCorrection !== null ? questClearCorrection : defaultClearBonus;

            let calculation;
            if (baseValue === 1) {
                // ★1の場合は基礎値×基準倍率のみ
                calculation = baseValue * baseMultiplier;
            } else {
                // ★2以降の場合
                // 1. まず基礎計算
                calculation = baseValue * baseMultiplier;
                
                // 2. カーブ計算を適用（★の数-1で計算）
                const starLevel = baseValue - 1;
                const curveMultiplier = multiplier * (starLevel + (starLevel * starLevel * curveFactor));
                calculation = Math.floor(calculation * (1 + curveMultiplier));
            }

            // 依頼補正の適用
            if (requestCorrection) {
                calculation = Math.floor(calculation * (1 + requestCorrection));
            }

            // クリア回数補正の適用
            if (clearCount > 0 && actualClearBonus) {
                calculation = Math.floor(calculation * (1 + (clearCount * actualClearBonus)));
            }

            // イベント補正を適用する前の計算結果を一時的に保存
            tempCalculations[i] = calculation;

            if ($gameSwitches.value(loggingSwitchId)) {
                console.log(`コモンイベントID ${commonEventId}:`);
                console.log(`  基本値: ${baseValue} × ${baseMultiplier}`);
                if (baseValue > 1) {
                    console.log(`  カーブ計算: 倍率=${multiplier}, カーブ係数=${curveFactor}, ★レベル=${baseValue}`);
                }
                console.log(`  依頼補正: ${requestCorrection}`);
                console.log(`  クリア回数: ${clearCount} (補正値: ${actualClearBonus})`);
                console.log(`  中間結果: ${tempCalculations[i]}`);
            }
        }

        // すべての計算が完了した後に EventCorrection を適用
        for (let i = 0; i < totalEvents; i++) {
            const commonEventId = startCommonEventId + i;
            const calculationVariableId = calculationVariableStartId + i;

            const commonEvent = $dataCommonEvents[commonEventId];
            if (!commonEvent) {
                continue;
            }

            let eventCorrection = 0;

            // イベント補正値を注釈から読み取る
            for (const command of commonEvent.list) {
                if (command.code === 108 || command.code === 408) {
                    const comment = command.parameters[0];
                    const eventMatch = comment.match(/<EventCorrection:\s*(.+?)\s*>/i);

                    if (eventMatch) {
                        // イベント補正は変数4076が0でない場合のみ適用
                        if ($gameVariables.value(4076) !== 0) {
                            eventCorrection = evaluateCorrection(eventMatch[1]);
                        }
                    }
                }
            }

            // イベント補正を適用して最終計算
            const finalCalculation = Math.floor(tempCalculations[i] * (1 + eventCorrection));

            // 計算結果を変数に格納
            $gameVariables.setValue(calculationVariableId, finalCalculation);

            if ($gameSwitches.value(loggingSwitchId)) {
                console.log(`コモンイベントID ${commonEventId}:`);
                console.log(`  イベント補正: ${eventCorrection}`);
                console.log(`  最終結果: ${finalCalculation} を変数ID ${calculationVariableId} に格納しました。`);
            }
        }

        // 3. 各コモンイベントの注釈 <Var2490: 数式> を解析し、変数3601番から3700番に格納
        for (let i = 0; i < totalEvents; i++) {
            const commonEventId = startCommonEventId + i;
            const var2490VariableId = var2490VariableStartId + i;

            const commonEvent = $dataCommonEvents[commonEventId];
            if (!commonEvent) {
                if ($gameSwitches.value(loggingSwitchId)) {
                    console.warn(`コモンイベントID ${commonEventId} が存在しません。変数ID ${var2490VariableId} には空白を格納します。`);
                }
                $gameVariables.setValue(var2490VariableId, '');
                continue;
            }

            let var2490Value = null;
            const regex = /<Var2490:\s*([^>]+)\s*>/i;

            for (const command of commonEvent.list) {
                if (command.code === 108 || command.code === 408) { // コメントコマンド
                    const comment = command.parameters[0];
                    const match = comment.match(regex);
                    if (match) {
                        let expression = match[1].trim();
                        // 式を評価
                        var2490Value = evaluateExpression(expression);
                        if ($gameSwitches.value(loggingSwitchId)) {
                            console.log(`コモンイベントID ${commonEventId}: <Var2490: ${expression}> を検出。評価結果: ${var2490Value}`);
                        }
                        break; // 一つの注釈のみを対象とする場合
                    }
                }
            }

            if (var2490Value !== null) {
                if (var2490Value === 0) {
                    $gameVariables.setValue(var2490VariableId, '');
                    if ($gameSwitches.value(loggingSwitchId)) {
                        console.log(`  変数ID ${var2490VariableId} に空白を格納しました (値が0)。`);
                    }
                } else {
                    $gameVariables.setValue(var2490VariableId, var2490Value);
                    if ($gameSwitches.value(loggingSwitchId)) {
                        console.log(`  変数ID ${var2490VariableId} に ${var2490Value} を格納しました。`);
                    }
                }
            } else {
                // 注釈に <Var2490: ...> が見つからなかった場合、空白を格納
                $gameVariables.setValue(var2490VariableId, '');
                if ($gameSwitches.value(loggingSwitchId)) {
                    console.log(`コモンイベントID ${commonEventId}: <Var2490: ...> の注釈が見つかりませんでした。変数ID ${var2490VariableId} に空白を格納しました。`);
                }
            }
        }

        // 4. アイテムID1001番から1100番のアイテム名を取得し、変数3501番から3600番に格納
        for (let i = 0; i < (itemIdEnd - itemIdStart + 1); i++) {
            const itemId = itemIdStart + i;
            const itemNameVariableId = itemNameVariableStartId + i;

            let itemName = '';
            if ($dataItems[itemId]) {
                itemName = $dataItems[itemId].name || ' ';
            } else {
                if ($gameSwitches.value(loggingSwitchId)) {
                    console.warn(`アイテムID ${itemId} が見つかりません。変数ID ${itemNameVariableId} には空白を格納します。`);
                }
                itemName = ' ';
            }

            // アイテム名を変数に格納
            $gameVariables.setValue(itemNameVariableId, itemName);

            if ($gameSwitches.value(loggingSwitchId)) {
                console.log(`  アイテムID ${itemId} の名前 (${itemName}) を変数ID ${itemNameVariableId} に格納しました。`);
            }

            // "×0" の値を空白に置き換え
            if (itemName === "×0") {
                $gameVariables.setValue(itemNameVariableId, ' ');
                if ($gameSwitches.value(loggingSwitchId)) {
                    console.log(`  変数ID ${itemNameVariableId} の値が "×0" であったため、空白に置き換えました。`);
                }
            }
        }

// 5. 最大レベル値の計算と格納を追加
for (let i = 0; i < totalEvents; i++) {
    const commonEventId = startCommonEventId + i;
    const questClearVariableId = questClearVariableStartId + i;
    const maxLevelVariableId = 4601 + i; // 4601～4700
    const inputVariableId = inputVariableStartId + i; // 3201～3300から対応する変数

    // 必要な値を取得
    const clearCount = $gameVariables.value(questClearVariableId);
    const baseValue = $gameVariables.value(inputVariableId); // 3201～3300の値を取得
    const var2441 = $gameVariables.value(2441);

    // シミュレータの基本計算ロジックを使用
    const adjustedVar1178 = baseValue + clearCount;
    const growthFactor = Math.log(adjustedVar1178 + 1);
    
    // 最小値と最大値の計算
    const baseCalc = var2441 * adjustedVar1178 * growthFactor;
    const minValue = Math.floor(baseCalc);
    const maxValue = Math.ceil(baseCalc + (var2441 - 1));

    // レベル範囲を格納
    const levelRange = `${minValue}~${maxValue}`;
    $gameVariables.setValue(maxLevelVariableId, levelRange);

    if ($gameSwitches.value(loggingSwitchId)) {
        console.log(`コモンイベントID ${commonEventId} の最大レベル値計算:`);
        console.log(`  クリア回数: ${clearCount}`);
        console.log(`  基本値(変数${inputVariableId}): ${baseValue}`);
        console.log(`  変数2441の値: ${var2441}`);
        console.log(`  調整後の値: ${adjustedVar1178}`);
        console.log(`  基本計算結果: ${baseCalc}`);
        console.log(`  最小値: ${minValue}`);
        console.log(`  最大値: ${maxValue}`);
        console.log(`  レベル範囲: ${levelRange}`);
        console.log(`  変数ID ${maxLevelVariableId} に格納しました。`);
    }
}

        if ($gameSwitches.value(loggingSwitchId)) {
            console.log(`全ての処理が完了しました。`);
        }
    });

    // 式を評価する関数
    function evaluateExpression(expression) {
        // \cc[condition]{trueResult}{falseResult} の形式を処理
        const ccRegex = /\\cc\[([^\]]+)\]\{([^}]*)\}\{([^}]*)\}/gi;
        expression = expression.replace(ccRegex, (match, condition, trueResult, falseResult) => {
            return evaluateCondition(condition) ? evaluateExpression(trueResult) : evaluateExpression(falseResult);
        });
    
        // \vc[...]の形式を処理
        if (expression.startsWith('\\vc[') && expression.endsWith(']')) {
            expression = expression.slice(4, -1); // \vc[ と ] を除去
        }
    
        // 変数参照を実際の値に置き換える
        expression = expression.replace(/\\v\[(\d+)\]/gi, (match, p1) => {
            return $gameVariables.value(parseInt(p1));
        });
    
        expression = expression.replace(/v(\d+)/gi, (match, p1) => {
            return $gameVariables.value(parseInt(p1));
        });
    
        // 式を評価
        try {
            const result = eval(expression);
            return isNaN(result) ? expression : result;
        } catch (error) {
            console.error('式の評価中にエラーが発生しました:', expression, error);
            return expression;
        }
    }

    // 条件を評価する関数
    function evaluateCondition(condition) {
        // スイッチの条件を評価
        if (condition.startsWith('s')) {
            const switchId = parseInt(condition.slice(1));
            return $gameSwitches.value(switchId);
        }
        // 変数の条件を評価
        else if (condition.startsWith('v')) {
            const parts = condition.match(/v(\d+)(==|!=|>|<|>=|<=)(\d+)/);
            if (parts) {
                const [, variableId, operator, value] = parts;
                const variableValue = $gameVariables.value(parseInt(variableId));
                const compareValue = parseInt(value);
                switch (operator) {
                    case '==': return variableValue == compareValue;
                    case '!=': return variableValue != compareValue;
                    case '>': return variableValue > compareValue;
                    case '<': return variableValue < compareValue;
                    case '>=': return variableValue >= compareValue;
                    case '<=': return variableValue <= compareValue;
                }
            }
        }
        // デフォルトでfalseを返す
        return false;
    }

    // 補正値を評価する関数
    function evaluateCorrection(expression) {
        // 変数参照（\v[n]）を実際の値に置き換える
        expression = expression.replace(/\\v\[(\d+)\]/gi, (match, p1) => {
            return $gameVariables.value(parseInt(p1));
        });

        // 単純な数式を評価
        try {
            return eval(expression);
        } catch (error) {
            console.error('補正値の評価中にエラーが発生しました:', expression, error);
            return 0; // エラーの場合は0を返す
        }
    }
})();