/*:
 * @plugindesc 特定のステート時にダメージを吸収するプラグイン
 * @author スタジオVR
 *
 * @param Absorption State ID
 * @desc ダメージ吸収効果を持つステートのID
 * @type state
 * @default 51
 *
 * @param Absorption Percentage
 * @desc 吸収するダメージの割合（%）
 * @type number
 * @min 1
 * @max 100
 * @default 15
 *
 * @param Use Variable for Percentage
 * @desc 吸収率に変数を使用するかどうか
 * @type boolean
 * @default false
 *
 * @param Absorption Percentage Variable ID
 * @desc 吸収率を指定する変数のID（Use Variable for Percentage が true の場合に使用）
 * @type variable
 * @default 1
 *
 * @param Show Absorption Message
 * @desc ダメージ吸収時にメッセージを表示するか
 * @type boolean
 * @default true
 *
 * @param Absorption Message
 * @desc ダメージ吸収時に表示するメッセージ（%1:吸収者の名前, %2:吸収量）
 * @default %1は%2のHPを吸収した！
 *
 * @help
 * このプラグインは、特定のステートが付与されているキャラクターが
 * 攻撃を行った際に、与えたダメージの指定された割合分のHPを回復する
 * ダメージ吸収機能を実装します。
 * 
 * 吸収率は固定値または変数で指定可能です：
 * - 固定値の場合：Absorption Percentage に数値を入力し、Use Variable for Percentage を false に設定
 * - 変数の場合：Use Variable for Percentage を true に設定し、Absorption Percentage Variable ID に変数IDを指定
 */

(() => {
    const pluginName = "Svr_DamageAbsorption";
    const parameters = PluginManager.parameters(pluginName);

    const absorptionStateId = Number(parameters['Absorption State ID'] || 51);
    const absorptionPercentage = Number(parameters['Absorption Percentage'] || 15);
    const useVariable = parameters['Use Variable for Percentage'] === 'true';
    const absorptionPercentageVariableId = Number(parameters['Absorption Percentage Variable ID'] || 1);
    const showAbsorptionMessage = parameters['Show Absorption Message'] !== 'false';
    const absorptionMessage = parameters['Absorption Message'] || '%1は%2のHPを吸収した！';

    const getAbsorptionPercentage = () => {
        if (useVariable) {
            return $gameVariables.value(absorptionPercentageVariableId) / 100;
        } else {
            return absorptionPercentage / 100;
        }
    };

    const processAbsorption = (subject, target, value) => {
        const absorptionRate = getAbsorptionPercentage();
        const absorbedAmount = Math.floor(value * absorptionRate);
        subject.gainHp(absorbedAmount);

        if (showAbsorptionMessage) {
            const message = absorptionMessage.replace('%1', subject.name()).replace('%2', absorbedAmount);
            BattleManager.queueAbsorptionMessage(message);
        }
    };

    const _Game_Action_executeHpDamage = Game_Action.prototype.executeHpDamage;
    Game_Action.prototype.executeHpDamage = function(target, value) {
        _Game_Action_executeHpDamage.call(this, target, value);
        
        if (value > 0 && this.subject().isStateAffected(absorptionStateId)) {
            processAbsorption(this.subject(), target, value);
        }
    };

    BattleManager.queueAbsorptionMessage = function(message) {
        if (!this._absorptionMessages) {
            this._absorptionMessages = [];
        }
        this._absorptionMessages.push(message);
    };

    const _BattleManager_endAction = BattleManager.endAction;
    BattleManager.endAction = function() {
        _BattleManager_endAction.call(this);
        this.displayAbsorptionMessages();
    };

    BattleManager.displayAbsorptionMessages = function() {
        if (this._absorptionMessages && this._absorptionMessages.length > 0) {
            const window = this._logWindow;
            this._absorptionMessages.forEach(message => window.push('addText', message));
            this._absorptionMessages = [];
        }
    };
})();