/*:
 * @plugindesc 特定のステート時にダメージを反射するプラグイン（反射率調整可能版・変数対応）
 * @author スタジオVR
 *
 * @param Reflection State ID
 * @desc ダメージ反射効果を持つステートのID
 * @type state
 * @default 596
 *
 * @param Reflection Percentage
 * @desc 反射するダメージの割合（%）または変数ID
 * @type string
 * @default 15
 *
 * @param Use Variable
 * @desc 反射率に変数を使用するかどうか
 * @type boolean
 * @default false
 *
 * @param Show Reflection Message
 * @desc ダメージ反射時にメッセージを表示するか
 * @type boolean
 * @default true
 *
 * @param Reflection Message
 * @desc ダメージ反射時に表示するメッセージ（%1:反射者の名前, %2:反射ダメージ）
 * @default %1は%2のダメージを跳ね返した！
 *
 * @help
 * このプラグインは、特定のステートが付与されているキャラクターが
 * 攻撃を受けた際に、そのダメージの指定された割合を攻撃者に跳ね返すパッシブスキルを
 * 実装します。反射割合を1%から100%の間で自由に設定できます。
 * 
 * 反射率は固定値または変数で指定可能です：
 * - 固定値の場合：Reflection Percentage に数値を入力し、Use Variable を false に設定
 * - 変数の場合：Reflection Percentage に変数IDを入力し、Use Variable を true に設定
 */

(() => {
    const pluginName = "Svr_DamageReflection";
    const parameters = PluginManager.parameters(pluginName);

    const reflectionStateId = Number(parameters['Reflection State ID'] || 596);
    const reflectionPercentageParam = parameters['Reflection Percentage'] || '15';
    const useVariable = parameters['Use Variable'] === 'true';
    const showReflectionMessage = parameters['Show Reflection Message'] !== 'false';
    const reflectionMessage = parameters['Reflection Message'] || '%1は%2のダメージを跳ね返した！';

    const messageCache = new Map();

    const getReflectionPercentage = () => {
        if (useVariable) {
            const variableId = Number(reflectionPercentageParam);
            return $gameVariables.value(variableId) / 100;
        } else {
            return Number(reflectionPercentageParam) / 100;
        }
    };

    const processReflection = (target, value, subject) => {
        const reflectionPercentage = getReflectionPercentage();
        const reflectionDamage = Math.floor(value * reflectionPercentage);
        subject.gainHp(-reflectionDamage);

        if (showReflectionMessage) {
            let message = messageCache.get(target.name());
            if (!message) {
                message = reflectionMessage.replace('%1', target.name());
                messageCache.set(target.name(), message);
            }
            BattleManager.queueReflectionMessage(message.replace('%2', reflectionDamage));
        }
    };

    const _Game_Action_executeHpDamage = Game_Action.prototype.executeHpDamage;
    Game_Action.prototype.executeHpDamage = function(target, value) {
        _Game_Action_executeHpDamage.call(this, target, value);
        
        if (value > 0 && target.isStateAffected(reflectionStateId)) {
            processReflection(target, value, this.subject());
        }
    };

    BattleManager.queueReflectionMessage = function(message) {
        if (!this._reflectionMessages) {
            this._reflectionMessages = [];
        }
        this._reflectionMessages.push(message);
    };

    const _BattleManager_endAction = BattleManager.endAction;
    BattleManager.endAction = function() {
        _BattleManager_endAction.call(this);
        this.displayReflectionMessages();
    };

    BattleManager.displayReflectionMessages = function() {
        if (this._reflectionMessages && this._reflectionMessages.length > 0) {
            const window = this._logWindow;
            this._reflectionMessages.forEach(message => window.push('addText', message));
            this._reflectionMessages = [];
        }
    };
})();