/*:
 * @target MZ
 * @plugindesc ステートの能力値効果を変数と四則演算で指定できるプラグイン（変数更新対応版）
 * @author スタジオVR
 *
 * @help
 * このプラグインは、ステートの能力値効果を変数と四則演算で指定できるようにします。
 * 基本パラメータに加え、命中率、回避率、会心率、会心回避率、魔法回避率、魔法反射率、
 * 反撃率、HP再生率、MP再生率、TP再生率にも対応しています。
 * 
 * 使用方法:
 * 1. プラグインをプロジェクトに追加します。
 * 2. ステートのメモ欄に以下のような記述を追加します：
 *    <DynamicStateEffect:パラメータID,計算式,モード>
 *    例: 
 *    <DynamicStateEffect:0,\v[4007],percent>
 *    <DynamicStateEffect:2,\v[3]+100,value>
 *    <DynamicStateEffect:hit,\v[5]*1.5,percent>
 * 
 * パラメータID:
 * 0: MHP,  1: MMP,  2: ATK,  3: DEF,  4: MAT,  5: MDF,  6: AGI,  7: LUK
 * hit: 命中率,     eva: 回避率,     cri: 会心率,     cev: 会心回避率
 * mev: 魔法回避率, mrf: 魔法反射率, cnt: 反撃率,
 * hrg: HP再生率,   mrg: MP再生率,   trg: TP再生率
 * 
 * 計算式:
 * - \v[n]形式で変数を指定します（nは変数ID）
 * - 四則演算（+, -, *, /）が使用可能です
 * - 括弧()も使用可能です
 * 
 * モード:
 * value: 数値で適用
 * percent: ％で適用
 * 注意：拡張パラメータの場合、percentは0-100の範囲で指定してください。
 * 
 * デバッグ:
 * スイッチID1003をONにすると、コンソールにデバッグログが表示されます。
 */

(function() {
    const _Game_BattlerBase_param = Game_BattlerBase.prototype.param;
    Game_BattlerBase.prototype.param = function(paramId) {
        if (this._calculatingDynamicEffect) return _Game_BattlerBase_param.call(this, paramId);
        this._calculatingDynamicEffect = true;
        const value = _Game_BattlerBase_param.call(this, paramId) + this.dynamicStateEffects(paramId);
        this._calculatingDynamicEffect = false;
        return value;
    };

    const _Game_BattlerBase_xparam = Game_BattlerBase.prototype.xparam;
    Game_BattlerBase.prototype.xparam = function(xparamId) {
        if (this._calculatingDynamicEffect) return _Game_BattlerBase_xparam.call(this, xparamId);
        this._calculatingDynamicEffect = true;
        const value = _Game_BattlerBase_xparam.call(this, xparamId) + this.dynamicStateEffects(this.xparamIdToString(xparamId));
        this._calculatingDynamicEffect = false;
        return value;
    };

    Game_BattlerBase.prototype.xparamIdToString = function(xparamId) {
        const xparamNames = ['hit', 'eva', 'cri', 'cev', 'mev', 'mrf', 'cnt', 'hrg', 'mrg', 'trg'];
        return xparamNames[xparamId];
    };

    Game_BattlerBase.prototype.dynamicStateEffects = function(paramId) {
        let value = 0;
        this.states().forEach(state => {
            const effects = this.getDynamicStateEffects(state);
            effects.forEach(effect => {
                if (effect.paramId == paramId) {
                    const effectValue = this.calculateDynamicEffect(effect.formula);
                    if (effect.mode === 'percent') {
                        if (typeof paramId === 'number') {
                            // 基本パラメータの場合
                            const baseParam = this.paramBase(paramId);
                            value += Math.floor(baseParam * effectValue / 100);
                        } else {
                            // 拡張パラメータの場合
                            value += effectValue / 100;
                        }
                    } else {
                        // 数値モードの場合、そのまま加算
                        value += effectValue;
                    }
                    this.debugLog(`Dynamic effect applied: ${state.name}, ${paramId}, ${effectValue}, ${effect.mode}`);
                }
            });
        });
        return value;
    };

    Game_BattlerBase.prototype.getDynamicStateEffects = function(state) {
        const effects = [];
        const notes = state.note.split(/[\r\n]+/);
        notes.forEach(note => {
            if (note.match(/<DynamicStateEffect:(.+)>/i)) {
                const parts = RegExp.$1.split(',');
                if (parts.length === 3) {
                    effects.push({
                        paramId: parts[0].trim(),
                        formula: parts[1].trim(),
                        mode: parts[2].trim().toLowerCase()
                    });
                }
            }
        });
        return effects;
    };

    Game_BattlerBase.prototype.calculateDynamicEffect = function(formula) {
        if (!formula) {
            this.debugLog('Formula is missing');
            return 0;
        }

        // 変数の置換
        let calculableFormula = formula.replace(/\\v\[(\d+)\]/g, (match, p1) => {
            return $gameVariables.value(parseInt(p1));
        });

        // 安全な計算のため、計算式を制限
        if (!/^[\d\s+\-*/().]+$/.test(calculableFormula)) {
            this.debugLog(`Invalid formula: ${formula}`);
            return 0;
        }

        try {
            const result = eval(calculableFormula);
            this.debugLog(`Formula evaluated: ${formula} = ${result}`);
            return result;
        } catch (e) {
            this.debugLog(`Error evaluating formula: ${formula}, ${e}`);
            return 0;
        }
    };

    let lastLogTime = 0;
    const LOG_INTERVAL = 1000; // ログの出力間隔（ミリ秒）

    Game_BattlerBase.prototype.debugLog = function(message) {
        if ($gameSwitches.value(1003)) {
            const currentTime = Date.now();
            if (currentTime - lastLogTime > LOG_INTERVAL) {
                console.log(`[DynamicStateEffect] ${message}`);
                lastLogTime = currentTime;
            }
        }
    };

    // 変数の値が変更されたときに呼び出されるフック
    const _Game_Variables_setValue = Game_Variables.prototype.setValue;
    Game_Variables.prototype.setValue = function(variableId, value) {
        _Game_Variables_setValue.call(this, variableId, value);
        $gameParty.members().forEach(member => {
            member.clearDynamicStateEffectsCache();
        });
    };

    Game_BattlerBase.prototype.clearDynamicStateEffectsCache = function() {
        this._dynamicStateEffectsCache = {};
    };
})();