/*:
 * @target MZ
 * @pluginname Svr_EnemyActionScale
 * @author スタジオVR
 * @base VisuMZ_1_BattleCore
 * @help
 * フロントビュー戦闘で敵が行動する際に
 * 拡大・縮小エフェクトを追加するプラグインです。
 * 
 * 作者：スタジオVR
 * バージョン：1.0.0
 * 
 * @param ScaleSettings
 * @text 拡大設定
 * @type struct<ScaleConfig>
 * 
 * @param TimingSettings
 * @text タイミング設定
 * @type struct<TimingConfig>
 */

/*~struct~ScaleConfig:
 * @param MaxScale
 * @text 最大拡大率
 * @type number
 * @decimals 2
 * @min 1.00
 * @max 2.00
 * @default 1.20
 * @desc 敵画像の最大拡大率（1.00が等倍）
 *
 * @param ScaleUpSpeed
 * @text 拡大速度
 * @type number
 * @decimals 2
 * @min 0.01
 * @max 1.00
 * @default 0.10
 * @desc 拡大する速度
 * 
 * @param ScaleDownSpeed
 * @text 縮小速度
 * @type number
 * @decimals 2
 * @min 0.01
 * @max 1.00
 * @default 0.08
 * @desc 縮小する速度
 */

/*~struct~TimingConfig:
 * @param HoldDuration
 * @text 拡大維持時間
 * @type number
 * @min 1
 * @max 1000
 * @default 250
 * @desc 拡大状態を維持する時間（ミリ秒）
 *
 * @param EnableEasing
 * @text イージング有効
 * @type boolean
 * @default true
 * @desc イージング（滑らかな動き）を有効にするか
 * 
 * @param ActionStart
 * @text 行動開始時
 * @type boolean
 * @default true
 * @desc 敵の行動開始時にエフェクトを実行するか
 */

(() => {
    const pluginName = "Svr_EnemyActionScale";
    
    // パラメーター取得
    const parameters = PluginManager.parameters(pluginName);
    const scaleConfig = JSON.parse(parameters.ScaleSettings || '{}');
    const timingConfig = JSON.parse(parameters.TimingSettings || '{}');
    
    // デフォルト値の設定
    const DEFAULT_CONFIG = {
        maxScale: Number(scaleConfig.MaxScale || 1.20),
        scaleUpSpeed: Number(scaleConfig.ScaleUpSpeed || 0.10),
        scaleDownSpeed: Number(scaleConfig.ScaleDownSpeed || 0.08),
        holdDuration: Number(timingConfig.HoldDuration || 250),
        enableEasing: timingConfig.EnableEasing === 'true',
        actionStart: timingConfig.ActionStart === 'true'
    };

    // 元のSprite_Enemyのupdateを保持
    const _Sprite_Enemy_initialize = Sprite_Enemy.prototype.initialize;
    Sprite_Enemy.prototype.initialize = function(battler) {
        _Sprite_Enemy_initialize.call(this, battler);
        this._scaleTarget = 1.0;
        this._isScaling = false;
        this._scaleProgress = 0;
        this._isScalingUp = true;
    };
    
    const _Sprite_Enemy_update = Sprite_Enemy.prototype.update;
    Sprite_Enemy.prototype.update = function() {
        _Sprite_Enemy_update.call(this);
        if (this._isScaling) {
            this.updateScale();
        }
    };
    
    // スケール更新処理
    Sprite_Enemy.prototype.updateScale = function() {
        if (DEFAULT_CONFIG.enableEasing) {
            this.updateScaleWithEasing();
        } else {
            this.updateScaleLinear();
        }
    };
    
    // イージング付きスケール更新
    Sprite_Enemy.prototype.updateScaleWithEasing = function() {
        const speed = this._isScalingUp ? 
            DEFAULT_CONFIG.scaleUpSpeed : 
            DEFAULT_CONFIG.scaleDownSpeed;
            
        this._scaleProgress += speed;
        
        if (this._scaleProgress >= 1) {
            this.scale.x = this._scaleTarget;
            this.scale.y = this._scaleTarget;
            this._isScaling = false;
            return;
        }
        
        const easeValue = this.easeInOutQuad(this._scaleProgress);
        const currentScale = this._startScale + (this._scaleTarget - this._startScale) * easeValue;
        this.scale.x = currentScale;
        this.scale.y = currentScale;
    };
    
    // 線形スケール更新
    Sprite_Enemy.prototype.updateScaleLinear = function() {
        if (this.scale.x !== this._scaleTarget) {
            const speed = this._isScalingUp ? 
                DEFAULT_CONFIG.scaleUpSpeed : 
                DEFAULT_CONFIG.scaleDownSpeed;
            
            const direction = this._scaleTarget > this.scale.x ? 1 : -1;
            this.scale.x += speed * direction;
            this.scale.y = this.scale.x;
            
            if ((direction === 1 && this.scale.x >= this._scaleTarget) ||
                (direction === -1 && this.scale.x <= this._scaleTarget)) {
                this.scale.x = this._scaleTarget;
                this.scale.y = this._scaleTarget;
                this._isScaling = false;
            }
        }
    };
    
    // イージング計算関数
    Sprite_Enemy.prototype.easeInOutQuad = function(t) {
        return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
    };
    
    // 拡大開始
    Sprite_Enemy.prototype.startScaleUp = function() {
        this._startScale = this.scale.x;
        this._scaleTarget = DEFAULT_CONFIG.maxScale;
        this._scaleProgress = 0;
        this._isScaling = true;
        this._isScalingUp = true;
    };
    
    // 通常サイズに戻す
    Sprite_Enemy.prototype.startScaleDown = function() {
        this._startScale = this.scale.x;
        this._scaleTarget = 1.0;
        this._scaleProgress = 0;
        this._isScaling = true;
        this._isScalingUp = false;
    };
    
    // 行動開始時の処理を拡張
    const _Game_Enemy_performActionStart = Game_Enemy.prototype.performActionStart;
    Game_Enemy.prototype.performActionStart = function(action) {
        _Game_Enemy_performActionStart.call(this, action);
        if (DEFAULT_CONFIG.actionStart && this.battler()) {
            this.battler().startScaleUp();
            setTimeout(() => {
                this.battler().startScaleDown();
            }, DEFAULT_CONFIG.holdDuration);
        }
    };
    
    // バトラー参照用のメソッド追加
    Game_Enemy.prototype.battler = function() {
        return SceneManager._scene._spriteset.findTargetSprite(this);
    };
})();