/*:
 * @target MZ
 * @plugindesc v3.1 装備品にランクを表示し、ランク別売却機能を追加するプラグイン
 * @author YourName
 * @url http://www.yourwebsite.com
 *
 * @param rankSettings
 * @text ランク設定
 * @type struct<RankSetting>[]
 * @desc 装備タイプごとのランクとパラメータ範囲の設定
 * @default ["{\"equipType\":\"1\",\"rank\":\"S\",\"minValue\":\"100\",\"maxValue\":\"999\"}","{\"equipType\":\"1\",\"rank\":\"A\",\"minValue\":\"80\",\"maxValue\":\"99\"}","{\"equipType\":\"1\",\"rank\":\"B\",\"minValue\":\"60\",\"maxValue\":\"79\"}","{\"equipType\":\"1\",\"rank\":\"C\",\"minValue\":\"40\",\"maxValue\":\"59\"}","{\"equipType\":\"1\",\"rank\":\"D\",\"minValue\":\"20\",\"maxValue\":\"39\"}","{\"equipType\":\"1\",\"rank\":\"E\",\"minValue\":\"0\",\"maxValue\":\"19\"}"]
 *
 * @param sellSound
 * @text 売却時効果音
 * @type struct<Sound>
 * @desc 装備売却時に再生する効果音の設定
 * @default {"name":"Shop1","volume":"90","pitch":"100","pan":"0"}
 *
 * @param failSound
 * @text 売却失敗時効果音
 * @type struct<Sound>
 * @desc 売却可能な装備がなかった時に再生する効果音の設定
 * @default {"name":"Buzzer1","volume":"90","pitch":"100","pan":"0"}
 *
 * @command sellEquipmentByRank
 * @text 指定ランクの装備を売却
 * @desc 指定したランクの余剰装備をすべて売却します（装備中のものは除く）。
 *
 * @arg rank
 * @type string
 * @text 売却するランク
 * @desc 売却する装備のランクを指定します。
 *
 * @command sellEquipmentByRankOrLower
 * @text 指定ランク以下の装備を売却
 * @desc 指定したランク以下の余剰装備をすべて売却します（装備中のものは除く）。
 *
 * @arg rank
 * @type number
 * @text 売却する最大ランク
 * @desc 売却する装備の最大ランクを指定します。このランク以下の装備が売却対象になります。
 *
 * @help Svr_EquipmentRankDisplay.js
 *
 * このプラグインは、装備品とアイテムの合計パラメータに基づいてランクを表示します。
 * プラグインパラメータで装備タイプごとにランクと合計パラメータの範囲を設定できます。
 * アイテム名の左側に [ランク] の形式でランクが表示されます。
 * 
 * 装備のメモ欄に<Sell Price: x>と記述することで、その装備の売却価格をxに設定できます。
 * 
 * プラグインコマンド:
 * 1. 指定したランクの余剰装備をすべて売却することができます（装備中のものは除く）。
 * 2. 指定したランク以下の余剰装備をすべて売却することができます（装備中のものは除く）。
 * 
 * 装備タイプID:
 * 1: 武器, 2: 盾, 3: 頭, 4: 身体, 5: 装飾品
 */
 
 /*~struct~Sound:
 * @param name
 * @text SE名
 * @type file
 * @dir audio/se
 * @desc 再生する効果音ファイル
 *
 * @param volume
 * @text 音量
 * @type number
 * @min 0
 * @max 100
 * @default 90
 * @desc 効果音の音量（0～100）
 *
 * @param pitch
 * @text ピッチ
 * @type number
 * @min 50
 * @max 150
 * @default 100
 * @desc 効果音のピッチ（50～150）
 *
 * @param pan
 * @text 位相
 * @type number
 * @min -100
 * @max 100
 * @default 0
 * @desc 効果音の位相（-100～100）
 */

(() => {
    const pluginName = "Svr_EquipmentRankDisplay";

    const parameters = PluginManager.parameters(pluginName);
    const rankSettings = JSON.parse(parameters['rankSettings'] || '[]').map(setting => {
        const parsed = JSON.parse(setting);
        return {
            equipType: Number(parsed.equipType),
            rank: parsed.rank,
            minValue: Number(parsed.minValue),
            maxValue: Number(parsed.maxValue)
        };
    });
    const sellSound = JSON.parse(parameters['sellSound'] || '{}');
    const failSound = JSON.parse(parameters['failSound'] || '{}');

    function getRank(totalParams, equipType) {
        for (const setting of rankSettings) {
            if (setting.equipType === equipType &&
                totalParams >= setting.minValue &&
                totalParams <= setting.maxValue) {
                return setting.rank;
            }
        }
        return '';
    }

    function getSellPrice(item) {
        const sellPriceMatch = item.note.match(/<Sell Price:\s*(\d+)>/i);
        return sellPriceMatch ? parseInt(sellPriceMatch[1]) : 0;
    }

    function calcItemRank(item) {
        if (!item) return '';
        const equipType = item.etypeId || 0;
        
        // Only calculate rank for equipment types 1-5
        if (equipType < 1 || equipType > 5) return '';

        const totalParams = item.params ? item.params.reduce((sum, param) => sum + param, 0) : 0;
        let rank = getRank(totalParams, equipType);
        
        if (!rank) {
            const sellPrice = getSellPrice(item);
            const calculatedRank = Math.floor(sellPrice / 10);
            rank = calculatedRank < 10 ? `0${calculatedRank}` : `${calculatedRank}`;
        }
        
        return rank;
    }

    function drawItemWithRank(item, x, y, width) {
        if (item) {
            const rank = calcItemRank(item);
            const rankWidth = rank ? this.textWidth('[' + rank + ']') : 0;
            const nameWidth = width - rankWidth;

            if (rank) {
                this.drawText('[' + rank + ']', x, y, rankWidth, 'left');
            }
            this.drawItemName(item, x + rankWidth, y, nameWidth);

            // Draw item count
            const itemCount = $gameParty.numItems(item);
            this.drawText(`x${itemCount}`, x, y, width, 'right');
        }
    }

    // Modify Window_EquipItem
    Window_EquipItem.prototype.drawItem = function(index) {
        const rect = this.itemLineRect(index);
        const item = this.itemAt(index);
        if (item) {
            this.changePaintOpacity(this.isEnabled(item));
            drawItemWithRank.call(this, item, rect.x, rect.y, rect.width);
            this.changePaintOpacity(true);
        } else {
            const removeText = TextManager.removeEquip || '外す';
            this.changePaintOpacity(true);
            const iconWidth = ImageManager.iconWidth;
            const textWidth = this.textWidth(removeText);
            this.drawIcon(160, rect.x, rect.y + 2);
            this.drawText(removeText, rect.x + iconWidth + 4, rect.y, textWidth);
        }
    };

    // Modify Window_ItemList
    const _Window_ItemList_drawItem = Window_ItemList.prototype.drawItem;
    Window_ItemList.prototype.drawItem = function(index) {
        const rect = this.itemLineRect(index);
        const item = this.itemAt(index);
        if (item) {
            this.changePaintOpacity(this.isEnabled(item));
            drawItemWithRank.call(this, item, rect.x, rect.y, rect.width);
            this.changePaintOpacity(true);
        }
    };

    // Fix "Remove" option in equip screen
    const _Window_EquipItem_includes = Window_EquipItem.prototype.includes;
    Window_EquipItem.prototype.includes = function(item) {
        // Always include null item (which represents "Remove" option)
        if (item === null) {
            return true;
        }
        return _Window_EquipItem_includes.call(this, item);
    };

    // Get all equipped items
    function getAllEquippedItems() {
        const equippedItems = new Set();
        $gameParty.members().forEach(actor => {
            actor.equips().forEach(item => {
                if (item) equippedItems.add(item.id);
            });
        });
        return equippedItems;
    }

    // Sell items function
    function sellItems(condition) {
        let totalGold = 0;
        let soldItems = 0;
        const equippedItems = getAllEquippedItems();

        // Function to sell a single item
        function sellItem(item) {
            if (!equippedItems.has(item.id)) {
                const price = getSellPrice(item);
                const count = $gameParty.numItems(item);
                $gameParty.loseItem(item, count);
                $gameParty.gainGold(price * count);
                totalGold += price * count;
                soldItems += count;
            }
        }

        // Get all items (weapons and armors)
        const allEquipment = $gameParty.allItems().filter(item => 
            (DataManager.isWeapon(item) || DataManager.isArmor(item)) && calcItemRank(item) !== ''
        );

        // Sell equipment
        allEquipment.forEach(item => {
            if (condition(item)) {
                sellItem(item);
            }
        });

        return { totalGold, soldItems };
    }

    // Function to resolve variable references
    function resolveVariableReference(value) {
        if (typeof value === 'string' && value.match(/\\V\[(\d+)\]/)) {
            const variableId = parseInt(RegExp.$1);
            return $gameVariables.value(variableId);
        }
        return value;
    }

    // Add plugin commands
    PluginManager.registerCommand(pluginName, "sellEquipmentByRank", args => {
        const rankToSell = resolveVariableReference(args.rank);
        const { totalGold, soldItems } = sellItems(item => calcItemRank(item) === rankToSell);

        // Show result message
        if (soldItems > 0) {
            playSellSound();
            $gameMessage.add(`ランク${rankToSell}の装備 x ${soldItems}を売却しました。`);
            $gameMessage.add(`${totalGold}ゴールドを獲得しました。`);
        } else {
            playFailSound();
            $gameMessage.add(`売却可能なランク${rankToSell}の装備はありませんでした。`);
        }
    });

    function playSellSound() {
        if (sellSound.name) {
            AudioManager.playSe({
                name: sellSound.name,
                volume: Number(sellSound.volume),
                pitch: Number(sellSound.pitch),
                pan: Number(sellSound.pan)
            });
        }
    }

    function playFailSound() {
        if (failSound.name) {
            AudioManager.playSe({
                name: failSound.name,
                volume: Number(failSound.volume),
                pitch: Number(failSound.pitch),
                pan: Number(failSound.pan)
            });
        }
    }
    
    PluginManager.registerCommand(pluginName, "sellEquipmentByRankOrLower", args => {
        const maxRankToSell = Number(resolveVariableReference(args.rank));
        const { totalGold, soldItems } = sellItems(item => {
            const rank = calcItemRank(item);
            if (rank === '') return false; // Skip items without rank
            const numericRank = rank.startsWith('0') ? Number(rank[1]) : Number(rank);
            return !isNaN(numericRank) && numericRank <= maxRankToSell;
        });

        // Show result message
        if (soldItems > 0) {
            playSellSound();
            $gameMessage.add(`ランク${maxRankToSell}以下の装備 x ${soldItems}を売却しました。`);
            $gameMessage.add(`${totalGold}ゴールドを獲得しました。`);
        } else {
            playFailSound();
            $gameMessage.add(`売却可能なランク${maxRankToSell}以下の装備はありませんでした。`);
        }
    });
})();