/*:
 * @plugindesc マップイベントの注釈に指定されたタグがあれば、マウスオーバー時にポップアップで表示するプラグイン
 * @author スタジオVR
 * @help
 * このプラグインでは、マップ上のイベントにマウスカーソルを合わせると、
 * イベントの注釈に書かれた特定のタグ(<PopupMessage>...</PopupMessage>)
 * の内容をポップアップで表示します。
 * 改行する際は<br>で改行を行います。
 *
 * @param textMargin
 * @text テキストマージン
 * @type number
 * @default 10
 * @desc ポップアップウィンドウ内のテキストマージンを指定します。
 *
 * @param soundEffect
 * @text サウンドエフェクト
 * @desc マウスオーバー時に再生するSEファイル名を指定します。
 * @default Cursor1
 * @type file
 * @dir audio/se/
 *
 * @param popupPosition
 * @text ポップアップ位置
 * @type select
 * @option 上部
 * @value upper
 * @option 下部
 * @value bottom
 * @option 右側
 * @value right
 * @option 左側
 * @value left
 * @default right
 */

(function() {
    const parameters = PluginManager.parameters('svr_EventPopupNotifier');
    const textMargin = Number(parameters['textMargin'] || 10);
    let popupPosition = parameters['popupPosition'] || 'right';

    // プラグインコマンドの登録
    PluginManager.registerCommand('svr_EventPopupNotifier', 'setPosition', args => {
        popupPosition = args.popupPosition;
    });

    // ポップアップウィンドウの定義
    class PopupWindow extends Window_Base {
        constructor(message, x, y, width, height) {
            super(new Rectangle(x, y, width, height));
            this.drawTextEx(message, textMargin, textMargin);
            this.openness = 255;
        }
    }

    let popups = {}; // 各イベントIDに対するポップアップウィンドウを格納するオブジェクト

    // テキストのサイズを計算する関数
const calculateWindowSize = (text) => {
    const dummyWindow = new Window_Base(new Rectangle());
    let maxWidth = 0;
    const lines = text.split('\n');
    let totalHeight = 0;
    const fontHeight = dummyWindow.lineHeight();
    const padding = dummyWindow.padding;
    const lineSpacing = 8; // 行間のスペース

    lines.forEach((line, index) => {
        const lineWidth = dummyWindow.textWidth(line) + textMargin * 2;
        maxWidth = Math.max(maxWidth, lineWidth);
        totalHeight += fontHeight;
        if (index < lines.length - 1) {
            totalHeight += lineSpacing;
        }
    });

    totalHeight += padding * 2;
    totalHeight += textMargin * 2;

    const width = maxWidth + padding * 2;
    const height = totalHeight;

    dummyWindow.destroy();
    return { width, height };
};

// ポップアップ位置を計算
const calculatePopupPosition = (event, position, popupWidth, popupHeight) => {
    const x = event.screenX();
    const y = event.screenY();
    const eventWidth = $gameMap.tileWidth();
    const eventGraphicHeight = $gameMap.tileHeight();
    let newX = x;
    let newY = y;

    // 基本位置を設定
    switch (position) {
        case 'upper':
            newX = x - popupWidth / 2;
            newY = y - eventGraphicHeight - popupHeight;
            break;
        case 'bottom':
            newX = x - popupWidth / 2;
            newY = y + eventGraphicHeight / 2;
            break;
        case 'right':
            newX = x + eventWidth / 2;
            newY = y - popupHeight / 2;
            break;
        case 'left':
            newX = x - eventWidth - popupWidth / 2;
            newY = y - popupHeight / 2;
            break;
    }

    // 画面外にはみ出ないように位置を調整
    const maxRight = Graphics.boxWidth - popupWidth;
    const maxBottom = Graphics.boxHeight - popupHeight;
    
    if (newX < 0) {
        newX = 0; // 左端に合わせる
    } else if (newX > maxRight) {
        newX = maxRight; // 右端に合わせる
    }

    if (newY < 0) {
        newY = 0; // 上端に合わせる
    } else if (newY > maxBottom) {
        newY = maxBottom; // 下端に合わせる
    }

    return { x: newX, y: newY };
};


// ポップアップを表示
const showPopup = (eventId, message) => {
    const event = $gameMap.event(eventId);
    if (!event || !message) return;

    // ポップアップウィンドウのサイズを計算
    const { width: popupWidth, height: popupHeight } = calculateWindowSize(message);
    // ポップアップウィンドウの位置を計算
    const { x, y } = calculatePopupPosition(event, popupPosition, popupWidth, popupHeight);

    // 既存のポップアップを削除
    if (popups[eventId]) {
        SceneManager._scene.removeChild(popups[eventId]);
        delete popups[eventId];
    }

    // 新しいポップアップを作成
    const popup = new PopupWindow(message, x, y, popupWidth, popupHeight);
    SceneManager._scene.addChild(popup);
    popups[eventId] = popup;
};





// マウスがイベントの上にあるか検出
const isMouseOverEvent = (event) => {
    const eventX = event.screenX();
    const eventY = event.screenY();
    const mouseX = TouchInput.x;
    const mouseY = TouchInput.y;
    const eventWidth = 48; // イベントのグラフィックの幅
    const eventHeight = 48; // イベントのグラフィックの高さ

    const adjustedEventY = eventY - eventHeight / 2;
    return mouseX >= eventX - eventWidth / 2 && mouseX <= eventX + eventWidth / 2 &&
        mouseY >= adjustedEventY - eventHeight / 2 && mouseY <= adjustedEventY + eventHeight / 2;
};

// 注釈からポップアップメッセージを抽出する
const extractPopupMessage = (eventId) => {
    const event = $gameMap.event(eventId);
    if (!event) return null;

    const activePage = event.findProperPageIndex();
    if (activePage < 0) return null;

    const list = event.event().pages[activePage].list;
    let fullText = "";
    let isRecording = false;

    // エスケープコードを実際の値に変換する関数
    const convertEscapeCharacters = (text) => {
        text = text.replace(/\\/g, '\x1b');
        text = text.replace(/\x1b\x1b/g, '\\');
        text = text.replace(/\x1bV\[(\d+)\]/gi, (_, n) => {
            return $gameVariables.value(parseInt(n)).toString();
        });
        // RPGツクールMVまたはMZにおける他のエスケープコードを追加する場合はここに記述します
        return text;
    };

    for (const command of list) {
        if (command.code === 108 || command.code === 408) { // 注釈コマンド
            if (command.parameters[0].includes("<PopupMessage>")) {
                isRecording = true;
                fullText = ""; // 新しい注釈なので初期化
            }
            if (isRecording) {
                fullText += command.parameters[0];
            }
            if (command.parameters[0].includes("</PopupMessage>")) {
                isRecording = false;
                const match = fullText.match(/<PopupMessage>([\s\S]*?)<\/PopupMessage>/);
                if (match) {
                    let message = match[1].trim();
                    message = message.replace(/<br>/gi, '\n'); // <br> タグを改行に置き換える
                    message = convertEscapeCharacters(message); // 変数などのエスケープコードを実際の値に変換
                    return message;
                }
            }
        }
    }
    return null; // マッチするテキストがなかった場合
};



const playSoundEffect = () => {
    const se = { name: parameters['soundEffect'], volume: 90, pitch: 100, pan: 0 };
    AudioManager.playSe(se);
};

let playedSoundEffects = {}; // 各イベントIDに対するSE再生フラグ

// メイン処理
const main = () => {
    $gameMap.events().forEach(event => {
        const eventId = event.eventId();
        if (isMouseOverEvent(event)) {
            const message = extractPopupMessage(eventId);
            if (message) {
                if (!playedSoundEffects[eventId]) {
                    playSoundEffect();
                    playedSoundEffects[eventId] = true;
                }
                if (!popups[eventId]) {
                    showPopup(eventId, message);
                }
            }
        } else {
            if (popups[eventId]) {
                SceneManager._scene.removeChild(popups[eventId]);
                delete popups[eventId];
            }
            delete playedSoundEffects[eventId];
        }
    });
};

    // ゲームループにメイン処理を組み込む
    const originalUpdate = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        originalUpdate.call(this);
        main();
    };
})();