/*:
 * @plugindesc 変数またはスイッチの状態に応じてアイコンを表示するプラグイン（再帰的変換処理版）
 * @author StudioVR (修正: Assistant)
 * @target MZ
 * @help
 * このプラグインは、テキスト中に特殊な制御文字を使用することで
 * 変数またはスイッチの状態に応じて異なるアイコンを表示します。
 * 
 * 使用方法:
 * 1. 変数の値に応じてアイコンを表示:
 *    \i_v[xx,yy,zz]
 *    xx: 変数ID または エスケープコード（例: \V[1]）
 *    yy: 変数が0の場合に表示するアイコンID
 *    zz: 変数が0以外の場合に表示するアイコンID
 * 
 * 2. スイッチの状態に応じてアイコンを表示:
 *    \i_s[xx,yy,zz]
 *    xx: スイッチID または エスケープコード（例: \V[1]）
 *    yy: スイッチがOFFの場合に表示するアイコンID
 *    zz: スイッチがONの場合に表示するアイコンID
 * 
 * 3. 条件式に応じてアイコンを表示:
 *    \i_if[xx,yy,zz]
 *    xx: 条件式（例: v1=0, v2!=1）
 *    yy: 条件式がtrueの場合に表示するアイコンID
 *    zz: 条件式がfalseの場合に表示するアイコンID
 */

(function() {
    const pluginName = "Svr_IconOutput";
    const DEBUG_MODE = true;

    function debug(message) {
        if (DEBUG_MODE) {
//            console.log(`[${pluginName}] ${message}`);
        }
    }

    const _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
    const _Window_Message_convertEscapeCharacters = Window_Message.prototype.convertEscapeCharacters;

    function convertVariableContent(text, depth = 0) {
        if (depth > 10) {
            debug(`Maximum recursion depth reached: ${text}`);
            return text;
        }

        debug(`Converting variable content (depth ${depth}): ${text}`);
        const result = text.replace(/\\/g, "\x1b")
                           .replace(/\x1bV\[(\d+)\]/gi, (_, p1) => {
                               const value = $gameVariables.value(parseInt(p1));
                               debug(`Variable ${p1} value: ${value}`);
                               return typeof value === 'string' ? convertVariableContent(value, depth + 1) : value;
                           })
                           .replace(/\x1b/g, "\\");
        debug(`Converted variable content (depth ${depth}): ${result}`);
        return result;
    }

    function evaluateCondition(condition) {
        debug(`Evaluating condition: ${condition}`);
        // 条件式を解析
        const match = condition.match(/^v(\d+)(=|!=)(\d+)$/);
        if (!match) {
            debug(`Invalid condition format: ${condition}`);
            return false;
        }

        const [_, varId, operator, value] = match;
        const variableValue = $gameVariables.value(Number(varId));
        const compareValue = Number(value);

        let result;
        switch (operator) {
            case '=':
                result = variableValue === compareValue;
                break;
            case '!=':
                result = variableValue !== compareValue;
                break;
            default:
                result = false;
        }

        debug(`Condition evaluation: var${varId}(${variableValue}) ${operator} ${value} = ${result}`);
        return result;
    }
    
    function convertIconEscapeCharacters(text) {
        debug(`Converting icon escape characters: ${text}`);
        
        text = text.replace(/\\i_v\[\s*(.*?)\s*,\s*(\d+)\s*,\s*(\d+)\s*\]/gi, (match, valueStr, iconId1, iconId2) => {
            debug(`Processing i_v: ${match}`);
            let value;
            let varId;

            const varMatch = valueStr.match(/^\\V\[(\d+)\]$/i);
            if (varMatch) {
                varId = $gameVariables.value(Number(varMatch[1]));
                value = $gameVariables.value(varId);
            } else {
                varId = Number(valueStr);
                value = $gameVariables.value(varId);
            }

            value = Number(value);
            const iconId = (value === 0) ? Number(iconId1) : Number(iconId2);
            debug(`i_v result: varId=${varId}, value=${value}, iconId=${iconId}`);

            return `\\I[${iconId}]`;
        });

        text = text.replace(/\\i_s\[\s*(.*?)\s*,\s*(\d+)\s*,\s*(\d+)\s*\]/gi, (match, valueStr, iconId1, iconId2) => {
            debug(`Processing i_s: ${match}`);
            let value;
            let switchId;

            const switchMatch = valueStr.match(/^\\V\[(\d+)\]$/i);
            if (switchMatch) {
                switchId = $gameVariables.value(Number(switchMatch[1]));
                value = $gameSwitches.value(switchId);
            } else {
                switchId = Number(valueStr);
                value = $gameSwitches.value(switchId);
            }

            const iconId = value === true ? Number(iconId2) : Number(iconId1);
            debug(`i_s result: switchId=${switchId}, value=${value}, iconId=${iconId}`);

            return `\\I[${iconId}]`;
        });

        text = text.replace(/\\i_if\[\s*(.*?)\s*,\s*(\d+)\s*,\s*(\d+)\s*\]/gi, (match, condition, iconId1, iconId2) => {
            debug(`Processing i_if: ${match}`);
            const result = evaluateCondition(condition.trim());
            const iconId = result ? Number(iconId1) : Number(iconId2);
            debug(`i_if result: condition=${condition}, result=${result}, iconId=${iconId}`);
            return `\\I[${iconId}]`;
        });

        debug(`Converted icon escape characters: ${text}`);
        return text;
    }

    function processAllEscapeCharacters(text) {
        let processedText = text;
        let prevText;
        do {
            prevText = processedText;
            processedText = convertVariableContent(processedText);
            processedText = convertIconEscapeCharacters(processedText);
        } while (processedText !== prevText);
        return processedText;
    }

    Window_Base.prototype.convertEscapeCharacters = function(text) {
        debug(`Window_Base converting: ${text}`);
        text = processAllEscapeCharacters(text);
        const result = _Window_Base_convertEscapeCharacters.call(this, text);
        debug(`Window_Base converted: ${result}`);
        return result;
    };

    Window_Message.prototype.convertEscapeCharacters = function(text) {
        debug(`Window_Message converting: ${text}`);
        text = processAllEscapeCharacters(text);
        const result = _Window_Message_convertEscapeCharacters.call(this, text);
        debug(`Window_Message converted: ${result}`);
        return result;
    };
})();