/*:
 * @plugindesc 400番台のコモンイベントを記録・再実行できるプラグイン
 * @author スタジオVR
 * @target MZ
 * @base PluginCommonBase
 * @orderAfter PluginCommonBase
 * 
 * @param showDebugInfo
 * @text デバッグ情報表示
 * @desc テストプレイ時にデバッグ情報を表示するかどうか
 * @type boolean
 * @default true
 * 
 * @command recordLastCommonEvent
 * @text 直近のコモンイベントを記録
 * @desc 400番台のコモンイベントのみ記録します。記録時に変数329の値も保存します。
 * 
 * @command executeLastCommonEvent
 * @text 記録したコモンイベントを実行
 * @desc 記録した400番台のコモンイベントを実行します。実行前に保存した変数329の値を復元します。
 *
 * @help Svr_LastCommonEvent.js
 * 
 * 400番台(400-499)のコモンイベントのみを記録・再実行できるプラグインです。
 * 400番台以外のコモンイベントの場合、記録値は0にリセットされます。
 * 
 * 400番台のコモンイベントを記録する際は変数329の値も保存され、
 * コモンイベント実行時に自動的に復元されます。
 * 
 * 【プラグインコマンド】
 * ・直近のコモンイベントを記録
 *   実行時点で最後に実行されていたコモンイベントを確認し、
 *   400番台の場合のみ記録します。
 *   400番台以外の場合は記録値が0になります。
 *   400番台の場合、変数329の値も保存されます。
 * 
 * ・記録したコモンイベントを実行
 *   記録済みのコモンイベントを実行します。
 *   実行できるのは400番台のコモンイベントのみです。
 *   実行前に、記録時に保存した変数329の値が復元されます。
 * 
 * 【スクリプト】
 * $gameSystem.getLastCommonEventId()
 *   記録されたコモンイベントの番号を取得します。
 *   400番台でない場合は0が返ります。
 */

(() => {
    'use strict';
    
    const pluginName = 'Svr_LastCommonEvent';
    const parameters = PluginManager.parameters(pluginName);
    const showDebugInfo = parameters['showDebugInfo'] === 'true';
    const VARIABLE_ID = 329;

    const is400Series = (id) => id >= 400 && id < 500;

    // プラグインコマンド：直近のコモンイベントを記録
    PluginManager.registerCommand(pluginName, "recordLastCommonEvent", args => {
        const findLastCommonEvent = (interpreter) => {
            if (!interpreter) return null;
            
            if (interpreter._childInterpreter) {
                const childResult = findLastCommonEvent(interpreter._childInterpreter);
                if (childResult !== null) return childResult;
            }
            
            if (interpreter._commonEventId) {
                return interpreter._commonEventId;
            }
            
            if (interpreter._list) {
                for (let i = interpreter._index; i >= 0; i--) {
                    const command = interpreter._list[i];
                    if (command && command.code === 117) {
                        return command.parameters[0];
                    }
                }
            }
            
            return null;
        };
        
        const currentCommonEventId = findLastCommonEvent($gameMap._interpreter) || 0;

        if (is400Series(currentCommonEventId)) {
            $gameSystem._lastCommonEventId = currentCommonEventId;
            $gameSystem._savedVariable329 = $gameVariables.value(VARIABLE_ID);
            if ($gameTemp.isPlaytest()) {
                const commonEvent = $dataCommonEvents[currentCommonEventId];
                const name = commonEvent ? commonEvent.name : '未設定';
                console.log(`直近のコモンイベントを記録 - ID: ${currentCommonEventId}, 名前: ${name}`);
                console.log(`変数329の値を保存: ${$gameSystem._savedVariable329}`);
            }
        } else {
            $gameSystem._lastCommonEventId = 0;
            $gameSystem._savedVariable329 = 0;
            if ($gameTemp.isPlaytest()) {
                console.log(`400番代以外のコモンイベントのため記録を0にリセット - ID: ${currentCommonEventId}`);
            }
        }
    });

    // プラグインコマンド：記録したコモンイベントを実行
    PluginManager.registerCommand(pluginName, "executeLastCommonEvent", args => {
        const commonEventId = $gameSystem.getLastCommonEventId();
        if (is400Series(commonEventId)) {
            // 変数329の値を復元
            $gameVariables.setValue(VARIABLE_ID, $gameSystem._savedVariable329);
            
            if ($gameTemp.isPlaytest()) {
                const commonEvent = $dataCommonEvents[commonEventId];
                const name = commonEvent ? commonEvent.name : '未設定';
                console.log(`記録したコモンイベントを実行 - ID: ${commonEventId}, 名前: ${name}`);
                console.log(`変数329の値を復元: ${$gameSystem._savedVariable329}`);
            }
            $gameTemp.reserveCommonEvent(commonEventId);
        } else if ($gameTemp.isPlaytest()) {
            console.log(`実行をスキップ（400番代以外のコモンイベント） - ID: ${commonEventId}`);
        }
    });

    // $gameSystemの初期化
    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function() {
        _Game_System_initialize.call(this);
        this._lastCommonEventId = 0;
        this._savedVariable329 = 0;
    };

    // 記録したコモンイベント番号を取得
    Game_System.prototype.getLastCommonEventId = function() {
        return this._lastCommonEventId || 0;
    };

    // デバッグウィンドウ
    function Window_LastCommonEvent() {
        this.initialize(...arguments);
    }

    Window_LastCommonEvent.prototype = Object.create(Window_Base.prototype);
    Window_LastCommonEvent.prototype.constructor = Window_LastCommonEvent;

    Window_LastCommonEvent.prototype.initialize = function() {
        const width = 300;
        const height = 80;  // 高さを増やして変数の値も表示
        const x = Graphics.width - width;
        const y = 0;
        Window_Base.prototype.initialize.call(this, new Rectangle(x, y, width, height));
        this.opacity = 200;
        this.refresh();
    };

    Window_LastCommonEvent.prototype.refresh = function() {
        this.contents.clear();
        const id = $gameSystem.getLastCommonEventId();
        const commonEvent = $dataCommonEvents[id];
        const name = commonEvent ? commonEvent.name : '';
        this.contents.fontSize = 16;
        this.drawText(`最後のコモンイベント: ${id}`, 0, 0, this.contents.width, 'left');
        this.drawText(`名前: ${name}`, 0, 24, this.contents.width, 'left');
        this.drawText(`変数329: ${$gameSystem._savedVariable329}`, 0, 48, this.contents.width, 'left');
    };

    // デバッグウィンドウの作成
    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_createAllWindows.call(this);
        if ($gameTemp.isPlaytest() && showDebugInfo) {
            this._lastCommonEventWindow = new Window_LastCommonEvent();
            this.addWindow(this._lastCommonEventWindow);
        }
    };

    // デバッグウィンドウの更新
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if (this._lastCommonEventWindow) {
            this._lastCommonEventWindow.refresh();
        }
    };

})();