/*:
 * @plugindesc データロード時に変数を設定するプラグイン
 * @author Claude
 * @target MV MZ
 * 
 * @param VariableSettings
 * @text 変数設定
 * @type struct<VariableSetting>[]
 * @desc データロード時に設定する変数のリスト
 * 
 * @param SwitchSettings
 * @text スイッチ設定
 * @type struct<SwitchSetting>[]
 * @desc データロード時に設定するスイッチのリスト
 * 
 * @help
 * データロード時に指定した変数に値を設定するプラグインです。
 * プラグインパラメータで変数IDと設定する値を指定できます。
 * 
 * 使用方法:
 * 1. プラグインパラメータの「変数設定」で変数IDと値を設定します
 * 2. プラグインパラメータの「スイッチ設定」でスイッチIDと状態を設定します
 * 3. データロード時に自動的に指定した値が設定されます
 */

/*~struct~VariableSetting:
 * @param VariableId
 * @text 変数ID
 * @type variable
 * @desc 設定する変数のID
 * 
 * @param Value
 * @text 設定値
 * @type number
 * @desc 変数に設定する値
 * @default 0
 */

/*~struct~SwitchSetting:
 * @param SwitchId
 * @text スイッチID
 * @type switch
 * @desc 設定するスイッチのID
 * 
 * @param Value
 * @text ON/OFF
 * @type boolean
 * @desc スイッチの状態
 * @default false
 */
 
(() => {
    'use strict';
    
    const pluginName = 'Svr_LoadVariable';
    
    // プラグインパラメータの取得
    const parameters = PluginManager.parameters(pluginName);
    const variableSettings = JSON.parse(parameters['VariableSettings'] || '[]')
        .map(setting => JSON.parse(setting));
    const switchSettings = JSON.parse(parameters['SwitchSettings'] || '[]')
        .map(setting => JSON.parse(setting));
        
    // DataManager.loadGameに処理を追加
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.call(this, contents);
        setVariablesAndSwitches();
    };
    
    // 変数とスイッチを設定する関数
    function setVariablesAndSwitches() {
        variableSettings.forEach(setting => {
            const varId = Number(setting.VariableId);
            const value = Number(setting.Value);
            if (varId > 0) {
                $gameVariables.setValue(varId, value);
            }
        });
        
        switchSettings.forEach(setting => {
            const switchId = Number(setting.SwitchId);
            const value = setting.Value === 'true';
            if (switchId > 0) {
                $gameSwitches.setValue(switchId, value);
            }
        });
    }
})();