/*:
 * @target MZ
 * @plugindesc マップ上で特定のステートアイコンをリアルタイムに表示し、情報をポップアップ表示するプラグイン（改良版）
 * @author スタジオVR
 *
 * @param Debug Mode
 * @desc デバッグモードを有効にする（コンソールにログを出力します）
 * @type boolean
 * @default true
 *
 * @command hideStateIcons
 * @text ステートアイコンを非表示
 * @desc マップ上のステートアイコンを非表示にします。
 *
 * @command showStateIcons
 * @text ステートアイコンを表示
 * @desc マップ上のステートアイコンを表示します。
 *
 * @command updateStateIcons
 * @text ステートアイコンを更新
 * @desc マップ上のステートアイコンを手動で更新します。
 *
 * @help
 * Svr_MapStateIconDisplay
 * 
 * このプラグインは、マップ上でPT1番目のメンバーに付与されている
 * ステートID560~620のアイコンを画面の指定位置に表示し、
 * マウスオーバー時にステート情報をポップアップ表示します。
 * バトル中には表示されません。
 * 
 * ステートは付与された順番で表示されます。
 * 
 * ステートのメモ欄に以下のタグを記述することで、ポップアップ内容をカスタマイズできます：
 * <StatePopup>
 * 名前: ステート名
 * 効果: ステートの効果説明
 * </StatePopup>
 * 
 * または
 * 
 * <StatePopup>
 * 効果: ステートの効果説明
 * </StatePopup>
 * 
 * 変数を使用する場合は \V[n] を使用してください。例：
 * 効果: 攻撃力が\V[10]%上昇
 * 
 * プラグインコマンド:
 * - ステートアイコンを非表示: マップ上のステートアイコンを非表示にします。
 * - ステートアイコンを表示: マップ上のステートアイコンを再表示します。
 * - ステートアイコンを更新: マップ上のステートアイコンを手動で更新します。
 */

const pluginName = "Svr_MapStateIconDisplay";

(() => {
    const parameters = PluginManager.parameters(pluginName);
    const debugMode = parameters['Debug Mode'] === 'true';

    function log(message) {
        if (debugMode) {
            console.log(`[${pluginName}] ${message}`);
        }
    }

    // 直接設定
    const displayX = 1020;
    const displayY = 450;
    const iconSize = 32;
    const iconsPerRow = 8;
    const popupWidth = 265;
    const popupLineHeight = 36;
    const popupNameFontSize = 18;
    const popupEffectFontSize = 15;
    const popupFixedX = displayX -5;
    const popupFixedY = displayY + 100;

    // プラグインコマンドの登録
    PluginManager.registerCommand(pluginName, "hideStateIcons", () => {
        $gameSystem.setStateIconsVisible(false);
        log("Hide state icons command executed");
    });

    PluginManager.registerCommand(pluginName, "showStateIcons", () => {
        $gameSystem.setStateIconsVisible(true);
        log("Show state icons command executed");
    });

    PluginManager.registerCommand(pluginName, "updateStateIcons", () => {
        $gameSystem.requestStateIconUpdate();
        log("Update state icons command executed");
    });

    // Game_Systemの拡張
    Game_System.prototype.initStateIconsVisible = function() {
        this._stateIconsVisible = true;
        this._stateIconUpdateRequested = false;
    };

    Game_System.prototype.isStateIconsVisible = function() {
        if (this._stateIconsVisible === undefined) {
            this.initStateIconsVisible();
        }
        return this._stateIconsVisible;
    };

    Game_System.prototype.setStateIconsVisible = function(visible) {
        if (this._stateIconsVisible !== visible) {
            this._stateIconsVisible = visible;
            this._stateIconUpdateRequested = true;
            log(`State icons visibility set to: ${visible}`);
        }
    };

    Game_System.prototype.requestStateIconUpdate = function() {
        this._stateIconUpdateRequested = true;
    };

    Game_System.prototype.isStateIconUpdateRequested = function() {
        return this._stateIconUpdateRequested;
    };

    Game_System.prototype.clearStateIconUpdateRequest = function() {
        this._stateIconUpdateRequested = false;
    };

    // Game_Battlerの拡張
    const _Game_Battler_initialize = Game_Battler.prototype.initialize;
    Game_Battler.prototype.initialize = function() {
        _Game_Battler_initialize.call(this);
        this._stateAdditionOrder = [];
    };

    const _Game_Battler_addState = Game_Battler.prototype.addState;
    Game_Battler.prototype.addState = function(stateId) {
        const hadState = this._states.includes(stateId);
        _Game_Battler_addState.call(this, stateId);
        if (!hadState && this._states.includes(stateId)) {
            this._stateAdditionOrder.push(stateId);
            // リアルタイム更新をコメントアウト
            // this.requestStateIconUpdate();
        }
    };

    const _Game_Battler_removeState = Game_Battler.prototype.removeState;
    Game_Battler.prototype.removeState = function(stateId) {
        const hadState = this._states.includes(stateId);
        _Game_Battler_removeState.call(this, stateId);
        if (hadState && !this._states.includes(stateId)) {
            this._stateAdditionOrder = this._stateAdditionOrder.filter(id => id !== stateId);
            // リアルタイム更新をコメントアウト
            // this.requestStateIconUpdate();
        }
    };

    Game_Battler.prototype.sortedStates = function() {
        return this._stateAdditionOrder.filter(id => this._states.includes(id));
    };

    // リアルタイム更新をコメントアウト
    /*
    Game_Battler.prototype.requestStateIconUpdate = function() {
        if ($gameSystem) {
            $gameSystem.requestStateIconUpdate();
        }
    };
    */

    // ステートアイコン表示用のスプライト
    class StateIconSprite extends Sprite {
        constructor() {
            super();
            this.initialize(...arguments);
        }

        initialize() {
            super.initialize();
            this.createBackground();
            this.createIcons();
            this._lastIconCount = 0;
            this._hoveredIndex = -1;
        }

        createBackground() {
            this.background = new PIXI.Graphics();
            this.addChild(this.background);
        }

        createIcons() {
            this.iconSprites = [];
            for (let i = 0; i < 51; i++) {
                const sprite = new Sprite();
                sprite.anchor.set(0.5);
                sprite.scale.set(iconSize / ImageManager.iconWidth);
                this.addChild(sprite);
                this.iconSprites.push(sprite);
            }
        }

        update() {
            super.update();
            if ($gameSystem.isStateIconUpdateRequested()) {
                this.updateIcons();
                $gameSystem.clearStateIconUpdateRequest();
            }
            this.updateHover();
            this.visible = $gameSystem.isStateIconsVisible();
        }

        updateIcons() {
            const actor = $gameParty.members()[0];
            if (!actor) return;

            const states = actor.sortedStates().filter(stateId => {
                const state = $dataStates[stateId];
                return state && state.iconIndex > 0 && stateId >= 560 && stateId <= 620;
            });
            const iconCount = states.length;

            this.updateBackground(iconCount);
            this.updateIconSprites(states);

            this._lastIconCount = iconCount;
        }

        updateBackground(iconCount) {
            this.background.clear();
            if (iconCount > 0) {
                const width = Math.min(iconCount, iconsPerRow) * iconSize;
                const height = Math.ceil(iconCount / iconsPerRow) * iconSize;
                this.background.beginFill(0x000000, 0.5);
                this.background.drawRect(0, 0, width, height);
                this.background.endFill();
            }
        }

        updateIconSprites(states) {
            states.forEach((stateId, index) => {
                const state = $dataStates[stateId];
                const sprite = this.iconSprites[index];
                sprite.bitmap = ImageManager.loadSystem('IconSet');
                sprite.setFrame(state.iconIndex % 16 * ImageManager.iconWidth, Math.floor(state.iconIndex / 16) * ImageManager.iconHeight, ImageManager.iconWidth, ImageManager.iconHeight);
                sprite.x = (index % iconsPerRow) * iconSize + iconSize / 2;
                sprite.y = Math.floor(index / iconsPerRow) * iconSize + iconSize / 2;
                sprite.show();
                sprite.state = state;
            });

            for (let i = states.length; i < this.iconSprites.length; i++) {
                this.iconSprites[i].hide();
            }
        }

        updateHover() {
            const mouseX = TouchInput.x - this.x;
            const mouseY = TouchInput.y - this.y;
            let hoveredIndex = -1;

            this.iconSprites.forEach((sprite, index) => {
                if (sprite.visible) {
                    const distance = Math.sqrt(Math.pow(mouseX - sprite.x, 2) + Math.pow(mouseY - sprite.y, 2));
                    if (distance <= iconSize / 2) {
                        hoveredIndex = index;
                    }
                    // ホバー状態に応じてフィルターを適用
                    if (index === hoveredIndex) {
                        sprite.filters = [new PIXI.filters.GlowFilter({ distance: 15, outerStrength: 2, color: 0xFFFFFF })];
                    } else {
                        sprite.filters = null;
                    }
                }
            });

            this._hoveredIndex = hoveredIndex;
        }

        getHoveredState() {
            return this._hoveredIndex !== -1 ? this.iconSprites[this._hoveredIndex].state : null;
        }
    }

    // ポップアップ表示用のウィンドウ
    class StatePopupWindow extends Window_Base {
        constructor(rect) {
            super(rect);
            this.initialize(...arguments);
        }
    
        initialize(rect) {
            super.initialize(rect);
            this.opacity = 0;
            this.contentsOpacity = 0;
            this._state = null;
            this.padding = 12; // パディングを調整
        }
    
        setStateInfo(state) {
            this._state = state;
            this.refresh();
        }
    
        refresh() {
            if (!this._state) return;
            this.contents.clear();
            this.resetFontSettings();
            
            const stateInfo = this.parseStatePopup(this._state.note);
            const name = stateInfo.名前 || this._state.name;
            const effect = stateInfo.効果 || '効果の説明がありません。';
            
            const nameHeight = this.drawStateName(name);
            const effectHeight = this.drawStateEffect(effect, nameHeight);
            
            // ウィンドウの高さを動的に調整
            this.height = nameHeight + effectHeight + this.padding * 2;
        }
    
        drawStateName(name) {
            this.contents.fontSize = popupNameFontSize;
            const y = 0;
            const width = this.contents.width;
            this.drawText(name, 0, y, width, 'Left');
            return this.lineHeight();
        }
    
        drawStateEffect(effect, y) {
            this.contents.fontSize = popupEffectFontSize;
            const width = this.contents.width;
            const textState = this.createTextState(this.convertEscapeCharacters(effect), 0, y, width);
            this.processAllText(textState);
            return textState.y + textState.height - y;
        }
    
        parseStatePopup(note) {
            const regex = /<StatePopup>([\s\S]*?)<\/StatePopup>/i;
            const match = note.match(regex);
            if (match) {
                const content = match[1];
                const lines = content.split('\n');
                const result = {};
                lines.forEach(line => {
                    const [key, ...valueParts] = line.split(':').map(s => s.trim());
                    const value = valueParts.join(':').trim();
                    if (key && value) {
                        result[key] = value;
                    }
                });
                return result;
            }
            return {};
        }
    
        convertEscapeCharacters(text) {
            text = text.replace(/\\V\[(\d+)\]/gi, (_, p1) => $gameVariables.value(parseInt(p1)));
            text = text.replace(/<br>/gi, '\n');  // <br>タグを改行文字に変換
            return text;
        }
    
        update() {
            super.update();
            if (this._state) {
                this.contentsOpacity = Math.min(this.contentsOpacity + 16, 255);
            } else {
                this.contentsOpacity = Math.max(this.contentsOpacity - 16, 0);
            }
        }
    
        // 自動改行を実装
        processAllText(textState) {
            while (textState.index < textState.text.length) {
                this.processCharacter(textState);
                this.processWordWrap(textState);
            }
            this.flushTextState(textState);
        }
    
        processWordWrap(textState) {
            if (textState.buffer) {
                const nextWidth = this.textWidth(textState.buffer + textState.text[textState.index]);
                if (textState.x + nextWidth > textState.width) {
                    this.flushTextState(textState);
                    this.processNewLine(textState);
                }
            }
        }

        show() {
            this.visible = true;
        }

        hide() {
            this.visible = false;
        }
    }

    // Scene_Mapの拡張
    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_createAllWindows.call(this);
        this.createStateIconSprite();
        this.createStatePopupWindow();
    };

    Scene_Map.prototype.createStateIconSprite = function() {
        this._stateIconSprite = new StateIconSprite();
        this._stateIconSprite.x = displayX;
        this._stateIconSprite.y = displayY;
        this.addChild(this._stateIconSprite);
    };

    Scene_Map.prototype.createStatePopupWindow = function() {
        const rect = new Rectangle(popupFixedX, popupFixedY, popupWidth, this.calcWindowHeight(3, false));
        this._statePopupWindow = new StatePopupWindow(rect);
        this.addWindow(this._statePopupWindow);
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        this.updateStatePopup();
        this.updateStateIconVisibility();
    };

    Scene_Map.prototype.updateStatePopup = function() {
        if (this._stateIconSprite && this._statePopupWindow) {
            const hoveredState = this._stateIconSprite.getHoveredState();
            if (hoveredState && $gameSystem.isStateIconsVisible()) {
                this._statePopupWindow.setStateInfo(hoveredState);
                this._statePopupWindow.show();
            } else {
                this._statePopupWindow.hide();
            }
        }
    };
    
    Scene_Map.prototype.updateStateIconVisibility = function() {
        if (this._stateIconSprite && this._statePopupWindow) {
            const isChoiceWindowActive = $gameMessage.isChoice();
            const shouldBeVisible = !isChoiceWindowActive && $gameSystem.isStateIconsVisible();
            
            if (this._stateIconSprite.visible !== shouldBeVisible) {
                this._stateIconSprite.visible = shouldBeVisible;
                if (!shouldBeVisible) {
                    this._statePopupWindow.hide();
                }
                log(`State icon visibility updated: ${shouldBeVisible}`);
            }
        }
    };
    
    // バトル中に表示しないようにする
    const _Scene_Battle_start = Scene_Battle.prototype.start;
    Scene_Battle.prototype.start = function() {
        _Scene_Battle_start.call(this);
        if (SceneManager._previousScene instanceof Scene_Map) {
            SceneManager._previousScene._stateIconSprite.visible = false;
            SceneManager._previousScene._statePopupWindow.hide();
            log("State icons hidden for battle");
        }
    };

    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.call(this);
        this.refreshStateIconDisplay();
    };

    Scene_Map.prototype.refreshStateIconDisplay = function() {
        if (this._stateIconSprite) {
            const isVisible = $gameSystem.isStateIconsVisible();
            this._stateIconSprite.visible = isVisible;
            if (isVisible) {
                $gameSystem.requestStateIconUpdate();
            }
            this._stateIconSprite.update(); // 即座に更新を強制
            log(`State icons visibility on map refresh: ${isVisible}`);
        }
    };

    // メニューから戻った時の処理を追加
    const _Scene_Map_terminate = Scene_Map.prototype.terminate;
    Scene_Map.prototype.terminate = function() {
        _Scene_Map_terminate.call(this);
        $gameSystem.requestStateIconUpdate();
    };

    // Scene_MenuBase拡張（メニュー系シーンの基底クラス）
    const _Scene_MenuBase_terminate = Scene_MenuBase.prototype.terminate;
    Scene_MenuBase.prototype.terminate = function() {
        _Scene_MenuBase_terminate.call(this);
        $gameSystem.requestStateIconUpdate();
    };
})();