/*:
 * @plugindesc [Ver1.0.1] メッセージカウントダウン表示
 * @author スタジオVR
 * @target MZ
 *
 * @param CountdownImages
 * @text カウントダウン画像設定
 * @type struct<CountdownImage>[]
 * @desc カウントダウンに使用する画像設定
 * 
 * @param DefaultX
 * @text デフォルトX座標
 * @type number
 * @min -9999
 * @default 408
 * 
 * @param DefaultY
 * @text デフォルトY座標
 * @type number
 * @min -9999
 * @default 312
 * 
 * @param DefaultOpacity
 * @text デフォルト透明度
 * @type number
 * @min 0
 * @max 255
 * @default 255
 *
 * @param FadeInDuration
 * @text フェードイン時間
 * @type number
 * @min 0
 * @default 30
 * @desc フェードインにかかるフレーム数
 *
 * @param FadeOutDuration
 * @text フェードアウト時間
 * @type number
 * @min 0
 * @default 30
 * @desc フェードアウトにかかるフレーム数
 *
 * @param EnableSwitch
 * @text 有効化スイッチ
 * @type switch
 * @default 1012
 * @desc このスイッチがONの時のみカウントダウンを表示します
 *
 * @command startCountdown
 * @text カウントダウン開始
 * @desc カウントダウン表示を開始します
 *
 * @arg x
 * @type number
 * @text X座標
 * @default 408
 * 
 * @arg y
 * @type number
 * @text Y座標
 * @default 312
 * 
 * @arg opacity
 * @type number
 * @text 透明度
 * @min 0
 * @max 255
 * @default 255
 *
 * @arg fadeIn
 * @type boolean
 * @text フェードイン
 * @default true
 *
 * @arg fadeOut
 * @type boolean
 * @text フェードアウト
 * @default true
 *
 * @arg images
 * @type string[]
 * @text 画像ファイル
 * @desc カウントダウンに使用する画像ファイル（img/pictures/内）。未指定時はプラグインパラメータの設定を使用
 * @default []
 *
 * @command stopCountdown
 * @text カウントダウン終了
 * @desc カウントダウン表示を終了します
 *
 * @command pauseCountdown
 * @text カウントダウン一時停止
 * @desc カウントダウン表示を一時停止します
 *
 * @command resumeCountdown
 * @text カウントダウン再開
 * @desc カウントダウン表示を再開します
 *
 * @command hideCountdown
 * @text カウントダウン非表示
 * @desc カウントダウンを一時的に非表示にします
 *
 * @command showCountdown
 * @text カウントダウン再表示
 * @desc 非表示にしたカウントダウンを再表示します
 */
 
/*~struct~CountdownImage:
 * @param ImageFile
 * @type file
 * @dir img/pictures
 * @text 画像ファイル
 * @desc カウントダウンに使用する画像ファイル
 * 
 * @param Order
 * @type number
 * @text 表示順序
 * @desc 画像の表示順序（小さい数字から順に表示）
 * @min 1
 * @default 1
 */
 
(() => {
    'use strict';
    const pluginName = "Svr_MessageCountdown";
    
    class CountdownState {
        constructor() {
            this.initialize();
        }
        
        initialize() {
            this.active = false;
            this.paused = false;
            this.hidden = false;
            this.currentIndex = 0;
            this.images = [];
            this.x = 0;
            this.y = 0;
            this.opacity = 255;
            this.sprite = null;
            this.fadeIn = true;
            this.fadeOut = true;
            this.fadeTimer = 0;
            this.isFading = false;
            this.isChanging = false;
            this.savedOpacity = 255;
            this.currentImageFile = '';
            this.needsRestore = false;
            this.messageCount = 0;
            this.lastSwitchState = false;
            this.wasActiveBeforeSwitch = false;
        }

        toSaveObject() {
            return {
                active: this.active,
                paused: this.paused,
                hidden: this.hidden,
                currentIndex: this.currentIndex,
                images: this.images,
                x: this.x,
                y: this.y,
                opacity: this.opacity,
                fadeIn: this.fadeIn,
                fadeOut: this.fadeOut,
                savedOpacity: this.savedOpacity,
                messageCount: this.messageCount,
                wasActiveBeforeSwitch: this.wasActiveBeforeSwitch
            };
        }

        fromSaveObject(saveObject) {
            if (!saveObject) return;
            Object.assign(this, saveObject);
        }
    }

    const countdown = new CountdownState();

    const parameters = PluginManager.parameters(pluginName);
    const countdownImages = JSON.parse(parameters.CountdownImages || '[]').map(image => JSON.parse(image));
    const fadeInDuration = Number(parameters.FadeInDuration || 30);
    const fadeOutDuration = Number(parameters.FadeOutDuration || 30);
    const enableSwitch = Number(parameters.EnableSwitch || 1012);
    
    function isCountdownEnabled() {
        return $gameSwitches.value(enableSwitch);
    }
    
    function processCountdownUpdate() {
        if (!isCountdownEnabled() || !countdown.sprite || countdown.currentIndex >= countdown.images.length) return;
        
        if ($gameSwitches.value(290)) {
            if (!countdown.lastSwitchState) {
                countdown.lastSwitchState = true;
            }
            return;
        } else {
            if (countdown.lastSwitchState) {
                countdown.lastSwitchState = false;
            }
            proceedCountdown();
        }
    }

    function proceedCountdown() {
        countdown.isFading = false;
        countdown.isChanging = false;
        changeToNextImage();
    }

    function updateCountdownImage() {
        processCountdownUpdate();
    }
    
    const _Window_Message_prototype_updateVisibility = Window_Message.prototype.updateVisibility;
    Window_Message.prototype.updateVisibility = function() {
        const wasVisible = this.visible;
        _Window_Message_prototype_updateVisibility.call(this);
        
        if (wasVisible !== this.visible && countdown.sprite) {
            countdown.sprite.visible = this.visible;
        }
    };

    const _Window_Message_prototype_show = Window_Message.prototype.show;
    Window_Message.prototype.show = function() {
        const wasVisible = this.visible;
        _Window_Message_prototype_show.call(this);
        if (!wasVisible && countdown.sprite) {
            countdown.sprite.visible = true;
        }
    };

    const _Window_Message_prototype_hide = Window_Message.prototype.hide;
    Window_Message.prototype.hide = function() {
        const wasVisible = this.visible;
        _Window_Message_prototype_hide.call(this);
        if (wasVisible && countdown.sprite) {
            countdown.sprite.visible = false;
        }
    };
    
    PluginManager.registerCommand(pluginName, "startCountdown", args => {
        if (!isCountdownEnabled()) return;
        
        countdown.active = true;
        countdown.paused = false;
        countdown.hidden = false;
        countdown.currentIndex = 0;
        countdown.x = Number(args.x) || Number(parameters.DefaultX);
        countdown.y = Number(args.y) || Number(parameters.DefaultY);
        countdown.opacity = Number(args.opacity) || Number(parameters.DefaultOpacity);
        countdown.fadeIn = args.fadeIn === "true";
        countdown.fadeOut = args.fadeOut === "true";
        countdown.savedOpacity = countdown.opacity;
        countdown.needsRestore = false;
        countdown.messageCount = 0;
        countdown.lastSwitchState = false;

        if (args.images) {
            const customImages = JSON.parse(args.images);
            if (customImages.length > 0) {
                countdown.images = customImages.map((filename, index) => ({
                    ImageFile: filename,
                    Order: index + 1
                }));
            } else {
                countdown.images = countdownImages;
            }
        } else {
            countdown.images = countdownImages;
        }

        setupCountdownSprite();
        
        if (countdown.fadeIn) {
            countdown.sprite.opacity = 0;
            countdown.isFading = true;
            countdown.fadeTimer = fadeInDuration;
        }
    });

    PluginManager.registerCommand(pluginName, "stopCountdown", args => {
        countdown.active = false;
        removeCountdownSprite();
    });

    PluginManager.registerCommand(pluginName, "pauseCountdown", args => {
        countdown.paused = true;
    });

    PluginManager.registerCommand(pluginName, "resumeCountdown", args => {
        if (!isCountdownEnabled()) return;
        countdown.paused = false;
    });

    PluginManager.registerCommand(pluginName, "hideCountdown", args => {
        if (!countdown.sprite || countdown.hidden) return;
        
        countdown.hidden = true;
        countdown.savedOpacity = countdown.sprite.opacity;
        countdown.sprite.opacity = 0;
    });

    PluginManager.registerCommand(pluginName, "showCountdown", args => {
        if (!countdown.sprite || !countdown.hidden) return;
        
        countdown.hidden = false;
        countdown.sprite.opacity = countdown.savedOpacity;
    });

    const _Game_Switches_setValue = Game_Switches.prototype.setValue;
    Game_Switches.prototype.setValue = function(switchId, value) {
        if (switchId === enableSwitch) {
            if (!value && countdown.active) {
                countdown.wasActiveBeforeSwitch = true;
                countdown.messageCount = 0;
                countdown.active = false;
                removeCountdownSprite();
            } else if (value && countdown.wasActiveBeforeSwitch) {
                countdown.wasActiveBeforeSwitch = false;
                countdown.active = true;
                countdown.currentIndex += countdown.messageCount;
                countdown.messageCount = 0;
                setupCountdownSprite();
            }
        }
        _Game_Switches_setValue.call(this, switchId, value);
    };

    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function() {
        const contents = _DataManager_makeSaveContents.call(this);
        contents.countdown = countdown.toSaveObject();
        return contents;
    };

    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.call(this, contents);
        if (contents.countdown) {
            countdown.fromSaveObject(contents.countdown);
            if (countdown.active && isCountdownEnabled()) {
                countdown.needsRestore = true;
                countdown.savedOpacity = countdown.opacity;
            }
        }
    };

    const _SceneManager_goto = SceneManager.goto;
    SceneManager.goto = function(sceneClass) {
        if (sceneClass === Scene_Title && countdown.active) {
            countdown.active = false;
            removeCountdownSprite();
        }
        _SceneManager_goto.call(this, sceneClass);
    };

    function setupCountdownSprite() {
        if (!isCountdownEnabled()) return;

        removeCountdownSprite();
        if (countdown.currentIndex < countdown.images.length) {
            countdown.sprite = new Sprite();
            countdown.currentImageFile = countdown.images[countdown.currentIndex].ImageFile;
            countdown.sprite.bitmap = ImageManager.loadPicture(countdown.currentImageFile);
            countdown.sprite.x = countdown.x;
            countdown.sprite.y = countdown.y;
            countdown.sprite.opacity = countdown.hidden ? 0 : (countdown.fadeIn ? 0 : countdown.opacity);

            if (SceneManager._scene) {
                const messageWindow = SceneManager._scene._messageWindow;
                if (messageWindow) {
                    countdown.sprite.visible = messageWindow.visible;
                }
                SceneManager._scene.addChild(countdown.sprite);
            }
        }
    }

    function removeCountdownSprite() {
        if (countdown.sprite && countdown.sprite.parent) {
            countdown.sprite.parent.removeChild(countdown.sprite);
            countdown.sprite = null;
        }
    }

    function changeToNextImage() {
        if (!isCountdownEnabled()) return;
        
        countdown.currentIndex++;
        if (countdown.currentIndex < countdown.images.length) {
            countdown.currentImageFile = countdown.images[countdown.currentIndex].ImageFile;
            countdown.sprite.bitmap = ImageManager.loadPicture(countdown.currentImageFile);
            if (countdown.fadeIn) {
                countdown.sprite.opacity = 0;
                countdown.isFading = true;
                countdown.fadeTimer = fadeInDuration;
            }
        } else {
            countdown.active = false;
            removeCountdownSprite();
        }
    }

    function updateFade() {
        if (!isCountdownEnabled() || !countdown.isFading || !countdown.sprite || countdown.hidden) return;

        countdown.fadeTimer--;
        
        if (countdown.isChanging) {
            countdown.sprite.opacity = Math.floor((countdown.fadeTimer / fadeOutDuration) * countdown.opacity);
            if (countdown.fadeTimer <= 0) {
                countdown.isChanging = false;
                changeToNextImage();
            }
        } else {
            countdown.sprite.opacity = Math.floor(((fadeInDuration - countdown.fadeTimer) / fadeInDuration) * countdown.opacity);
            if (countdown.fadeTimer <= 0) {
                countdown.isFading = false;
                countdown.sprite.opacity = countdown.opacity;
            }
        }
        countdown.savedOpacity = countdown.sprite.opacity;
    }

    const _Scene_Base_terminate = Scene_Base.prototype.terminate;
    Scene_Base.prototype.terminate = function() {
        if (countdown.active && countdown.sprite) {
            countdown.needsRestore = true;
        }
        _Scene_Base_terminate.call(this);
    };

    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.call(this);
        restoreCountdownIfNeeded.call(this);
    };

    const _Scene_Battle_start = Scene_Battle.prototype.start;
    Scene_Battle.prototype.start = function() {
        _Scene_Battle_start.call(this);
        restoreCountdownIfNeeded.call(this);
    };

    function restoreCountdownIfNeeded() {
        if (countdown.needsRestore && countdown.active && isCountdownEnabled()) {
            countdown.sprite = null;
            setupCountdownSprite();
            if (countdown.sprite) {
                countdown.sprite.opacity = countdown.savedOpacity;
                countdown.needsRestore = false;
            }
        }
    }

    const _Scene_Message_update = Scene_Message.prototype.update;
    Scene_Message.prototype.update = function() {
        _Scene_Message_update.call(this);
        if (countdown.active && !countdown.paused && isCountdownEnabled()) {
            updateFade();
        }
    };

    const _Window_Message_prototype_update = Window_Message.prototype.update;
    Window_Message.prototype.update = function() {
        const wasVisible = this.visible;
        _Window_Message_prototype_update.call(this);
        
        if (wasVisible !== this.visible && countdown.sprite) {
            countdown.sprite.visible = this.visible;
        }
    };

    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function() {
        if (countdown.active && !countdown.paused && isCountdownEnabled()) {
            if (!$gameSwitches.value(290)) {
                _Window_Message_terminateMessage.call(this);
                updateCountdownImage();
            } else {
                _Window_Message_terminateMessage.call(this);
            }
        } else {
            if (!isCountdownEnabled() && countdown.wasActiveBeforeSwitch) {
                countdown.messageCount++;
            }
            _Window_Message_terminateMessage.call(this);
        }
    };

    const _Scene_Load_onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
        if (countdown.sprite) {
            removeCountdownSprite();
        }
        _Scene_Load_onLoadSuccess.call(this);
        
        if (countdown.active && isCountdownEnabled()) {
            setupCountdownSprite();
            if (countdown.sprite) {
                countdown.sprite.opacity = countdown.savedOpacity;
            }
        }
    };
    
})();