/*:
 * @target MZ
 * @plugindesc メッセージ入力ダイアログプラグイン
 * @author スタジオVR
 *
 * @param Variable ID
 * @desc 入力内容を格納する変数ID
 * @type variable
 * @default 1
 *
 * @param Window Width
 * @desc ダイアログウィンドウのデフォルト幅（ピクセル）
 * @type number
 * @min 100
 * @default 400
 *
 * @param Window Height
 * @desc ダイアログウィンドウのデフォルト高さ（ピクセル）
 * @type number
 * @min 100
 * @default 300
 *
 * @param Open Sound
 * @desc ダイアログを開いたときの効果音
 * @type file
 * @dir audio/se/
 * @default 
 *
 * @param OK Sound
 * @desc OKボタンを押したときの効果音
 * @type file
 * @dir audio/se/
 * @default 
 *
 * @param Cancel Sound
 * @desc キャンセルボタンを押したときの効果音
 * @type file
 * @dir audio/se/
 * @default 
 *
 * @command showInputDialog
 * @text 入力ダイアログを表示
 * @desc メッセージ入力ダイアログを表示します。
 *
 * @arg dialogMessage
 * @text ダイアログメッセージ
 * @desc ダイアログに表示するメッセージ
 * @type string
 * @default メッセージを入力してください
 *
 * @arg characterLimit
 * @text 最大文字数
 * @desc 入力可能な最大文字数（半角）。各行の最大文字数が指定されている場合は無視されます。
 * @type number
 * @min 1
 * @default 20
 *
 * @arg numberOfLines
 * @text 入力行数
 * @desc 入力欄の行数
 * @type number
 * @min 1
 * @default 1
 *
 * @arg lineCharacterLimit
 * @text 各行の最大文字数
 * @desc 各行の最大文字数（半角）。0の場合は全体の最大文字数が適用されます。
 * @type number
 * @min 0
 * @default 0
 *
 * @arg defaultValueVariableId
 * @text デフォルト値の変数ID
 * @desc デフォルトで入力欄に表示される値を格納している変数ID。0の場合はデフォルト値なし。
 * @type variable
 * @default 0
 *
 * @arg windowWidth
 * @text ウィンドウ幅
 * @desc ダイアログウィンドウの幅（ピクセル）。0の場合はプラグインパラメータの値が使用されます。
 * @type number
 * @min 0
 * @default 0
 *
 * @arg windowHeight
 * @text ウィンドウ高さ
 * @desc ダイアログウィンドウの高さ（ピクセル）。0の場合はプラグインパラメータの値が使用されます。
 * @type number
 * @min 0
 * @default 0
 *
 * @help
 * このプラグインは、メッセージ入力が可能なダイアログを表示します。
 * プラグインコマンドを使用してダイアログを表示できます。
 * 入力内容は指定された変数に格納されます。
 * キャンセルされた場合は-1、空欄の場合は0が格納されます。
 * 
 * 各行の最大文字数を0以上で指定すると、その文字数で各行が制限されます。
 * 0の場合は、全体の最大文字数のみが適用されます。
 * 各行の最大文字数が指定されている場合、全体の最大文字数は無視されます。
 *
 * デフォルト値の変数IDを指定すると、その変数の値が初期値として入力欄に表示されます。
 * デフォルト値が-1の場合、テキストボックスは空欄になります。
 * デフォルト値は自動的に文字数制限に合わせてカットされます。
 *
 * 効果音はプラグインパラメータで指定できます。
 * 効果音を指定しない場合、音は再生されません。
 *
 * ウィンドウの幅と高さはプラグインコマンドで指定できます。
 * 0を指定した場合、プラグインパラメータで設定したデフォルト値が使用されます。
 */

(() => {
    const pluginName = "Svr_MessageInputDialog";

    const parameters = PluginManager.parameters(pluginName);
    const variableId = Number(parameters['Variable ID'] || 1);
    const defaultWindowWidth = Number(parameters['Window Width'] || 400);
    const defaultWindowHeight = Number(parameters['Height'] || 300);
    const openSound = parameters['Open Sound'] || '';
    const okSound = parameters['OK Sound'] || '';
    const cancelSound = parameters['Cancel Sound'] || '';

    let isDialogOpen = false;
    let overlay;
    let activeDialog = null;
    let lastGamepadState = null;
    let currentFocus = null;
    let removeEventListeners = null;
    let isProcessingEvent = false;
    const focusableElements = ['inputArea', 'okBtn', 'cancelBtn'];

    // 既存の共通関数は同じまま
    function playSound(filename) {
        if (filename) {
            AudioManager.playSe({
                name: filename,
                pan: 0,
                pitch: 100,
                volume: 100
            });
        }
    }

    function getActualLength(str) {
        return str.replace(/[^\x01-\x7E\xA1-\xDF]/g, "aa").length;
    }

    function truncateString(str, maxLength) {
        let actualLength = 0;
        let truncated = '';
        for (let i = 0; i < str.length; i++) {
            const char = str[i];
            const charLength = getActualLength(char);
            if (actualLength + charLength > maxLength) break;
            actualLength += charLength;
            truncated += char;
        }
        return truncated;
    }

    function moveFocus(direction) {
        if (!activeDialog) return;

        if (!currentFocus) {
            const textarea = document.getElementById('inputArea');
            if (textarea) {
                textarea.focus();
                currentFocus = 'inputArea';
            }
            return;
        }

        const currentIndex = focusableElements.indexOf(currentFocus);
        let nextIndex;
        
        if (direction === 'next') {
            nextIndex = (currentIndex + 1) % focusableElements.length;
        } else {
            nextIndex = (currentIndex - 1 + focusableElements.length) % focusableElements.length;
        }
        
        currentFocus = focusableElements[nextIndex];
        const element = document.getElementById(currentFocus);
        if (element) {
            element.focus();
        }
    }
    
    // ゲームパッド入力の処理も更新
    const handleGamepadInput = function() {
        if (!activeDialog || isProcessingEvent) return;

        try {
            if (Input.isTriggered('ok')) {
                const focusedElement = document.activeElement;
                if (focusedElement && (focusedElement.id === 'okBtn' || focusedElement.id === 'cancelBtn')) {
                    focusedElement.click();
                }
            }

            if (Input.isTriggered('cancel')) {
                const cancelBtn = document.getElementById('cancelBtn');
                if (cancelBtn) cancelBtn.click();
            }

            if (Input.isTriggered('up') || Input.isTriggered('left')) {
                moveFocus('prev');
            }
            if (Input.isTriggered('down') || Input.isTriggered('right')) {
                moveFocus('next');
            }

        } catch (error) {
            console.error('Gamepad handling error:', error);
        }
    };

    // 既存のScene_Map.updateの拡張を修正
    const _originalSceneMapUpdate = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        if (isDialogOpen && !isProcessingEvent) {
            Input.update();  // ゲームパッドの状態を更新
            handleGamepadInput();
            return;
        }
        _originalSceneMapUpdate.call(this);
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        if (isDialogOpen && !isProcessingEvent) {
            handleGamepadInput();
            return;
        }
        _Scene_Map_update.call(this);
    };

    const _Game_Interpreter_updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
    Game_Interpreter.prototype.updateWaitMode = function() {
        if (this._waitMode === 'inputDialog') {
            return isDialogOpen;
        }
        return _Game_Interpreter_updateWaitMode.call(this);
    };

    const setupEventListeners = function(dialog, textarea, okBtn, cancelBtn) {
        const handleGlobalKeydown = (e) => {
            if (!isDialogOpen || isProcessingEvent) return;
            
            if (Input.isTriggered('ok') && !currentFocus) {
                e.preventDefault();
                e.stopPropagation();
                textarea.focus();
                currentFocus = 'inputArea';
                return;
            }
        };

        const handleDialogKeydown = (e) => {
            if (isProcessingEvent) return;
            e.stopPropagation();

            // Tabキーの処理は既存のまま維持
            if (e.key === 'Tab') {
                e.preventDefault();
                moveFocus(e.shiftKey ? 'prev' : 'next');
                return;
            }

            // 決定とキャンセルの処理をInput.isTriggeredに変更
            if (Input.isTriggered('ok')) {
                e.preventDefault();
                if (e.target.tagName === 'TEXTAREA') {
                    const lines = textarea.value.split('\n');
                    if (lines.length >= Number(textarea.getAttribute('rows'))) {
                        okBtn.click();
                    }
                } else {
                    e.target.click();
                }
            } else if (Input.isTriggered('cancel')) {
                e.preventDefault();
                cancelBtn.click();
            }

            // 矢印キーの処理
            if (e.target.tagName !== 'TEXTAREA') {
                if (Input.isTriggered('down') || Input.isTriggered('right')) {
                    e.preventDefault();
                    moveFocus('next');
                } else if (Input.isTriggered('up') || Input.isTriggered('left')) {
                    e.preventDefault();
                    moveFocus('prev');
                }
            }
        };

        document.addEventListener('keydown', handleGlobalKeydown, true);
        dialog.addEventListener('keydown', handleDialogKeydown);

        dialog.addEventListener('focusout', (e) => {
            if (!e.relatedTarget || !dialog.contains(e.relatedTarget)) {
                currentFocus = null;
            }
        });

        dialog.addEventListener('focusin', (e) => {
            const id = e.target.id;
            if (focusableElements.includes(id)) {
                currentFocus = id;
            }
        });

        return () => {
            document.removeEventListener('keydown', handleGlobalKeydown, true);
            dialog.removeEventListener('keydown', handleDialogKeydown);
        };
    };

    PluginManager.registerCommand(pluginName, "showInputDialog", function(args) {
        if (isDialogOpen) return;
        isProcessingEvent = true;

        try {
            // ダイアログの初期化
            this.setWaitMode('inputDialog');
            isDialogOpen = true;
            currentFocus = null;
            lastGamepadState = null;

            playSound(openSound);

            // オーバーレイの作成
            overlay = document.createElement('div');
            overlay.style.position = 'fixed';
            overlay.style.top = '0';
            overlay.style.left = '0';
            overlay.style.width = '100%';
            overlay.style.height = '100%';
            overlay.style.backgroundColor = 'rgba(0, 0, 0, 0.5)';
            overlay.style.zIndex = '9998';
            document.body.appendChild(overlay);

            // パラメータの準備
            const dialogMessage = args.dialogMessage;
            const characterLimit = Number(args.characterLimit) || 20;
            const numberOfLines = Number(args.numberOfLines) || 1;
            const lineCharacterLimit = Number(args.lineCharacterLimit) || 0;
            const defaultValueVariableId = Number(args.defaultValueVariableId) || 0;
            const windowWidth = Number(args.windowWidth) || defaultWindowWidth;
            const windowHeight = Number(args.windowHeight) || defaultWindowHeight;

            let limitMessage = '';
            if (lineCharacterLimit > 0) {
                limitMessage = `（各行の最大文字数：半角${lineCharacterLimit}文字まで）`;
            } else if (characterLimit > 0) {
                limitMessage = `（最大文字数：半角${characterLimit}文字まで）`;
            }

            let defaultValue = '';
            if (defaultValueVariableId > 0) {
                const rawDefaultValue = $gameVariables.value(defaultValueVariableId);
                if (rawDefaultValue !== -1) {
                    defaultValue = String(rawDefaultValue || '');
                    if (lineCharacterLimit > 0) {
                        defaultValue = defaultValue.split('\n')
                            .map(line => truncateString(line, lineCharacterLimit))
                            .join('\n');
                    } else if (characterLimit > 0) {
                        defaultValue = truncateString(defaultValue, characterLimit);
                    }
                }
            }

            // ダイアログの作成
            const dialog = document.createElement('div');
            activeDialog = dialog;
            dialog.style.position = 'fixed';
            dialog.style.top = '50%';
            dialog.style.left = '50%';
            dialog.style.transform = 'translate(-50%, -50%)';
            dialog.style.zIndex = '9999';
            dialog.style.background = '#f0f0f0';
            dialog.style.padding = '20px';
            dialog.style.borderRadius = '10px';
            dialog.style.boxShadow = '0 4px 6px rgba(0,0,0,0.1)';
            dialog.style.width = `${windowWidth}px`;
            dialog.style.height = `${windowHeight}px`;

            dialog.innerHTML = `
                <p style="font-size: 18px; margin-bottom: 5px;">${dialogMessage}</p>
                <p style="font-size: 14px; margin-bottom: 15px; color: #666;">${limitMessage}</p>
                <textarea id="inputArea" rows="${numberOfLines}" style="width: calc(100% - 20px); padding: 10px; font-size: 16px; border-radius: 5px; border: 1px solid #ccc; resize: none; line-height: 1.5;">${defaultValue}</textarea>
                <div style="display: flex; justify-content: space-between; margin-top: 15px;">
                    <button id="okBtn" style="padding: 10px 20px; font-size: 16px; background: #2196F3; color: white; border: none; border-radius: 5px; cursor: pointer; width: 45%;">OK</button>
                    <button id="cancelBtn" style="padding: 10px 20px; font-size: 16px; background: #E0E0E0; color: black; border: none; border-radius: 5px; cursor: pointer; width: 45%;">キャンセル</button>
                </div>
            `;

            document.body.appendChild(dialog);

            // 要素の参照を取得
            const textarea = document.getElementById('inputArea');
            const okBtn = document.getElementById('okBtn');
            const cancelBtn = document.getElementById('cancelBtn');

            // イベントリスナーの設定
            removeEventListeners = setupEventListeners(dialog, textarea, okBtn, cancelBtn);

            // 入力制限の設定
            textarea.addEventListener('input', function() {
                let lines = this.value.split('\n');
                if (lines.length > numberOfLines) {
                    lines = lines.slice(0, numberOfLines);
                }
                if (lineCharacterLimit > 0) {
                    lines = lines.map(line => truncateString(line, lineCharacterLimit));
                } else if (characterLimit > 0) {
                    let totalLength = 0;
                    lines = lines.map(line => {
                        if (totalLength >= characterLimit) return '';
                        const truncatedLine = truncateString(line, characterLimit - totalLength);
                        totalLength += getActualLength(truncatedLine);
                        return truncatedLine;
                    }).filter(line => line !== '');
                }
                this.value = lines.join('\n');
            });

            // ボタンのイベントハンドラー
            cancelBtn.onclick = () => {
                if (isProcessingEvent) return;
                isProcessingEvent = true;
                try {
                    playSound(cancelSound);
                    $gameVariables.setValue(variableId, -1);
                    closeDialog();
                } finally {
                    isProcessingEvent = false;
                }
            };

            okBtn.onclick = () => {
                if (isProcessingEvent) return;
                isProcessingEvent = true;
                try {
                    playSound(okSound);
                    const inputValue = textarea.value.trim();
                    $gameVariables.setValue(variableId, inputValue === '' ? 0 : inputValue);
                    closeDialog();
                } finally {
                    isProcessingEvent = false;
                }
            };

            // 初期フォーカスの設定
            textarea.focus();
            currentFocus = 'inputArea';

        } finally {
            isProcessingEvent = false;
        }
    });

    function closeDialog() {
        if (removeEventListeners) {
            removeEventListeners();
        }
        document.body.removeChild(activeDialog);
        document.body.removeChild(overlay);
        isDialogOpen = false;
        activeDialog = null;
        currentFocus = null;
        lastGamepadState = null;
        removeEventListeners = null;
        isProcessingEvent = false;
    }
})();