//=============================================================================
// Svr_MessagePictureEraser.js
//=============================================================================

/*:ja
 * @target MZ
 * @plugindesc メッセージ送り時ピクチャ消去プラグイン v1.0.0
 * @author YourName
 * @version 1.0.0
 * @description メッセージ送りをするごとに指定されたピクチャを消去します。
 * 
 * @param pictureIds
 * @text 消去するピクチャID
 * @desc メッセージ送り時に消去するピクチャのIDを指定します（カンマ区切りで複数指定可能）
 * @type string
 * @default 1
 * 
 * @param enableSwitch
 * @text 有効化スイッチ
 * @desc このスイッチがONの時のみ機能します（0で常に有効）
 * @type switch
 * @default 0
 * 
 * @help Svr_MessagePictureEraser.js
 * 
 * このプラグインは、メッセージウィンドウでテキストを送るたびに
 * 指定されたピクチャを自動的に消去します。
 * 
 * 【使用方法】
 * 1. プラグインパラメータで消去したいピクチャのIDを設定
 * 2. 複数のピクチャを指定する場合は「1,3,5」のようにカンマ区切りで記述
 * 3. 有効化スイッチを設定すると、そのスイッチがONの時のみ動作
 * 
 * 【注意事項】
 * - メッセージの表示が完了した時（次のメッセージに進む時）にピクチャが消去されます
 * - 選択肢や数値入力などでも同様に動作します
 */

(() => {
    'use strict';
    
    const pluginName = 'Svr_MessagePictureEraser';
    const parameters = PluginManager.parameters(pluginName);
    
    // パラメータの取得と処理
    const pictureIdsParam = parameters['pictureIds'] || '1';
    const enableSwitch = Number(parameters['enableSwitch']) || 0;
    
    // ピクチャIDを配列に変換
    const pictureIds = pictureIdsParam.split(',').map(id => Number(id.trim())).filter(id => id > 0);
    
    // プラグインが有効かどうかをチェック
    function isPluginEnabled() {
        return enableSwitch === 0 || $gameSwitches.value(enableSwitch);
    }
    
    // 指定されたピクチャを消去する関数
    function erasePictures() {
        if (!isPluginEnabled()) return;
        
        pictureIds.forEach(pictureId => {
            if ($gameScreen.picture(pictureId)) {
                $gameScreen.erasePicture(pictureId);
            }
        });
    }
    
    // Window_Messageの処理をオーバーライド
    const _Window_Message_processInput = Window_Message.prototype.processInput;
    Window_Message.prototype.processInput = function() {
        const wasOpen = this.isOpen();
        const hadText = this._textState && this._textState.index < this._textState.text.length;
        
        _Window_Message_processInput.call(this);
        
        // メッセージが進んだ場合（テキストが残っていたのに進んだ、または次のページに移った）
        if (wasOpen && this.isOpen()) {
            const hasTextNow = this._textState && this._textState.index < this._textState.text.length;
            
            // テキストが進んだ、または新しいページが開始された場合
            if (hadText && !hasTextNow) {
                erasePictures();
            }
        }
    };
    
    // メッセージウィンドウが閉じる時の処理もフック
    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function() {
        _Window_Message_terminateMessage.call(this);
        // メッセージが完全に終了した時もピクチャを消去
        erasePictures();
    };
    
    // 次のメッセージに移る時の処理
    const _Window_Message_newPage = Window_Message.prototype.newPage;
    Window_Message.prototype.newPage = function(textState) {
        _Window_Message_newPage.call(this, textState);
        // 新しいページが開始された時にピクチャを消去
        if (textState.index > 0) {  // 最初のページでない場合
            erasePictures();
        }
    };
    
    // デバッグ用：コンソールにログを出力（リリース時は削除推奨）
    if (pictureIds.length > 0) {
//        console.log(`${pluginName}: 監視対象ピクチャID - ${pictureIds.join(', ')}`);
        if (enableSwitch > 0) {
//            console.log(`${pluginName}: 有効化スイッチ - ${enableSwitch}`);
        }
    }
})();