/*:
 * @target MZ
 * @plugindesc メッセージの進捗状況をゲージと数値で表示するプラグイン
 * @author スタジオVR & Assistant
 *
 * @param fontSize
 * @text フォントサイズ
 * @type number
 * @desc 表示するフォントのサイズ
 * @default 16
 * 
 * @param positionX
 * @text X座標
 * @type number
 * @desc 表示位置のX座標
 * @default 0
 * 
 * @param positionY
 * @text Y座標
 * @type number
 * @desc 表示位置のY座標
 * @default 0
 * 
 * @param gaugeWidth
 * @text ゲージ幅
 * @type number
 * @desc プログレスバーの幅
 * @default 200
 * 
 * @param gaugeHeight
 * @text ゲージ高さ
 * @type number
 * @desc プログレスバーの高さ
 * @default 12
 * 
 * @param useShinyEffect
 * @text 光沢エフェクト
 * @type boolean
 * @desc ゲージに光沢効果を適用するかどうか
 * @default true
 * 
 * @command showProgress
 * @text 進捗表示開始
 * @desc メッセージの進捗状況の表示を開始します
 * 
 * @command hideProgress
 * @text 進捗表示終了
 * @desc メッセージの進捗状況の表示を終了します
 * 
 * @command toggleVisibility
 * @text 表示切替
 * @desc 進捗表示の表示/非表示を切り替えます
 * 
 * @command resumeProgress
 * @text 表示再開
 * @desc 非表示にした進捗表示を再開します
 * 
 * @command recalculateProgress
 * @text 進捗再計算
 * @desc 現在のイベントのメッセージ進捗状況を再計算します
 */

(() => {
    'use strict';
    
    const pluginName = "Svr_MessageProgress";
    
    const params = PluginManager.parameters(pluginName);
    const fontSize = Number(params['fontSize'] || 16);
    const posX = Number(params['positionX'] || 0);
    const posY = Number(params['positionY'] || 0);
    const gaugeWidth = Number(params['gaugeWidth'] || 200);
    const gaugeHeight = Number(params['gaugeHeight'] || 12);
    const useShinyEffect = params['useShinyEffect'] !== 'false';
    
    class ProgressTracker {
        constructor() {
            this.reset();
        }
        
        reset() {
            this.isTracking = false;
            this.totalCount = 0;
            this.currentPosition = 1;
            this.messageIndices = [];
            this.isVisible = true;
        }

        toSaveObject() {
            return {
                isTracking: this.isTracking,
                totalCount: this.totalCount,
                currentPosition: this.currentPosition,
                messageIndices: this.messageIndices.slice(),
                isVisible: this.isVisible
            };
        }

        fromSaveObject(saveObject) {
            if (!saveObject) {
                this.reset();
                return;
            }

            this.isTracking = saveObject.isTracking;
            this.totalCount = saveObject.totalCount;
            this.currentPosition = saveObject.currentPosition;
            this.messageIndices = saveObject.messageIndices.slice();
            this.isVisible = saveObject.isVisible;
        }
        
        getCurrentInterpreter() {
            let interpreter = null;
            
            if ($gameMap && $gameMap._interpreter) {
                interpreter = $gameMap._interpreter;
                
                while (interpreter._childInterpreter) {
                    interpreter = interpreter._childInterpreter;
                }
            }
            
            return interpreter;
        }
        
        startTracking() {
            this.reset();
            
            const interpreter = this.getCurrentInterpreter();
            if (!interpreter) return;
            
            const list = interpreter._list;
            if (!list) return;
            
            this.isTracking = true;
            
            for (let i = 0; i < list.length; i++) {
                if (list[i].code === 101) {
                    this.messageIndices.push(i);
                }
            }
            
            this.totalCount = this.messageIndices.length;
            this.updatePosition(interpreter._index);
        }
        
        updatePosition(currentIndex) {
            if (!this.isTracking) return;
            
            let newPosition = 1;
            for (let i = 0; i < this.messageIndices.length; i++) {
                if (currentIndex > this.messageIndices[i]) {
                    newPosition = i + 2;
                }
            }
            
            this.currentPosition = Math.min(newPosition, this.totalCount);
        }
        
        toggleVisibility() {
            this.isVisible = !this.isVisible;
        }
        
        show() {
            this.isVisible = true;
        }
        
        hide() {
            this.isVisible = false;
        }

        shouldDisplay() {
            return this.isTracking && this.isVisible && $gameSwitches.value(1013);
        }

        recalculate() {
            if (!this.isTracking) return;
            
            const interpreter = this.getCurrentInterpreter();
            if (!interpreter || !interpreter._list) return;
            
            this.messageIndices = [];
            
            for (let i = 0; i < interpreter._list.length; i++) {
                if (interpreter._list[i].code === 101) {
                    this.messageIndices.push(i);
                }
            }
            
            this.totalCount = this.messageIndices.length;
            this.updatePosition(interpreter._index);
        }
    }
    
    const progressTracker = new ProgressTracker();
    
    function Window_MessageProgress() {
        this.initialize(...arguments);
    }
    
    Window_MessageProgress.prototype = Object.create(Window_Base.prototype);
    Window_MessageProgress.prototype.constructor = Window_MessageProgress;
    
    Window_MessageProgress.prototype.initialize = function() {
        const width = 240;
        const height = 100;
        this._customX = posX;
        this._customY = posY;
        Window_Base.prototype.initialize.call(this, new Rectangle(this._customX, this._customY, width, height));
        this.createContents();
        this.setBackgroundType(2);
        this._lastSwitchState = false;
        this.hide();
    };
    
    Window_MessageProgress.prototype.update = function() {
        Window_Base.prototype.update.call(this);
        if (this.x !== this._customX || this.y !== this._customY) {
            this.x = this._customX;
            this.y = this._customY;
        }
        const currentSwitchState = $gameSwitches.value(1013);
        if (this._lastSwitchState !== currentSwitchState) {
            this._lastSwitchState = currentSwitchState;
            this.refresh();
        }
    };

    Window_MessageProgress.prototype.drawGauge = function(x, y, width, rate) {
        const fillW = Math.floor(width * rate);
        const gaugeY = y + this.lineHeight() - gaugeHeight - 2;
        const cornerRadius = 6;
        
        this.drawRoundedRect(x, gaugeY, width, gaugeHeight, cornerRadius, ColorManager.gaugeBackColor());
        
        if (fillW > 0) {
            const color1 = ColorManager.tpGaugeColor1();
            const color2 = ColorManager.tpGaugeColor2();
            
            this.drawRoundedGradientRect(x, gaugeY, fillW, gaugeHeight, cornerRadius, color1, color2);
            
            if (useShinyEffect) {
                const highlightHeight = gaugeHeight / 3;
                this.contents.gradientFillRect(
                    x, gaugeY, fillW, highlightHeight,
                    'rgba(255, 255, 255, 0.3)',
                    'rgba(255, 255, 255, 0.1)'
                );
            }
        }
        
        this.drawRoundedRectBorder(x, gaugeY, width, gaugeHeight, cornerRadius, 'rgba(0, 0, 0, 0.3)', 1);
    };

    Window_MessageProgress.prototype.drawRoundedRect = function(x, y, width, height, radius, color) {
        const ctx = this.contents.context;
        ctx.save();
        ctx.beginPath();
        ctx.moveTo(x + radius, y);
        ctx.lineTo(x + width - radius, y);
        ctx.arcTo(x + width, y, x + width, y + radius, radius);
        ctx.lineTo(x + width, y + height - radius);
        ctx.arcTo(x + width, y + height, x + width - radius, y + height, radius);
        ctx.lineTo(x + radius, y + height);
        ctx.arcTo(x, y + height, x, y + height - radius, radius);
        ctx.lineTo(x, y + radius);
        ctx.arcTo(x, y, x + radius, y, radius);
        ctx.closePath();
        ctx.fillStyle = color;
        ctx.fill();
        ctx.restore();
    };

    Window_MessageProgress.prototype.drawRoundedGradientRect = function(x, y, width, height, radius, color1, color2) {
        const ctx = this.contents.context;
        ctx.save();
        
        ctx.beginPath();
        ctx.moveTo(x + radius, y);
        ctx.lineTo(x + width - radius, y);
        ctx.arcTo(x + width, y, x + width, y + radius, radius);
        ctx.lineTo(x + width, y + height - radius);
        ctx.arcTo(x + width, y + height, x + width - radius, y + height, radius);
        ctx.lineTo(x + radius, y + height);
        ctx.arcTo(x, y + height, x, y + height - radius, radius);
        ctx.lineTo(x, y + radius);
        ctx.arcTo(x, y, x + radius, y, radius);
        ctx.closePath();
        ctx.clip();
        
        const gradient = ctx.createLinearGradient(x, y, x, y + height);
        gradient.addColorStop(0, color1);
        gradient.addColorStop(1, color2);
        
        ctx.fillStyle = gradient;
        ctx.fillRect(x, y, width, height);
        ctx.restore();
    };

    Window_MessageProgress.prototype.drawRoundedRectBorder = function(x, y, width, height, radius, color, lineWidth) {
        const ctx = this.contents.context;
        ctx.save();
        ctx.beginPath();
        ctx.moveTo(x + radius, y);
        ctx.lineTo(x + width - radius, y);
        ctx.arcTo(x + width, y, x + width, y + radius, radius);
        ctx.lineTo(x + width, y + height - radius);
        ctx.arcTo(x + width, y + height, x + width - radius, y + height, radius);
        ctx.lineTo(x + radius, y + height);
        ctx.arcTo(x, y + height, x, y + height - radius, radius);
        ctx.lineTo(x, y + radius);
        ctx.arcTo(x, y, x + radius, y, radius);
        ctx.closePath();
        ctx.strokeStyle = color;
        ctx.lineWidth = lineWidth;
        ctx.stroke();
        ctx.restore();
    };
    
    Window_MessageProgress.prototype.refresh = function() {
        if (!progressTracker.shouldDisplay()) {
            this.hide();
            return;
        }
        
        this.show();
        this.contents.clear();
        
        const current = progressTracker.currentPosition;
        const total = progressTracker.totalCount;
        const rate = (current - 1) / Math.max(1, total - 1);
        
        const padding = 10;
        const barWidth = this.width - (padding * 2) - 24;
        
        this.drawGauge(padding + 12, 8, barWidth, rate);
        
        const progressText = `${current}/${total}`;
        const textWidth = this.textWidth(progressText);
        const textX = padding + 12 + (barWidth - textWidth) / 2;
        this.contents.fontSize = fontSize;
        this.drawText(progressText, textX, 24, textWidth + 20, 'left');
    };

    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function() {
        _Game_System_initialize.call(this);
        this._messageProgressData = null;
    };

    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function() {
        const contents = _DataManager_makeSaveContents.call(this);
        if ($gameSystem) {
            $gameSystem._messageProgressData = progressTracker.toSaveObject();
        }
        return contents;
    };

    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.call(this, contents);
        if ($gameSystem && $gameSystem._messageProgressData) {
            progressTracker.fromSaveObject($gameSystem._messageProgressData);
            
            const scene = SceneManager._scene;
            if (scene && scene._progressWindow) {
                scene._progressWindow.refresh();
            }
        }
    };

    const _SceneManager_goto = SceneManager.goto;
    SceneManager.goto = function(sceneClass) {
        if (sceneClass === Scene_Title) {
            progressTracker.reset();
            const scene = SceneManager._scene;
            if (scene && scene._progressWindow) {
                scene._progressWindow.hide();
            }
        }
        _SceneManager_goto.call(this, sceneClass);
    };
    
    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_createAllWindows.call(this);
        this._progressWindow = new Window_MessageProgress();
        this.addWindow(this._progressWindow);
    };

    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.call(this);
        if (this._progressWindow) {
            this._progressWindow.x = posX;
            this._progressWindow.y = posY;
            this._progressWindow.refresh();
        }
    };
    
    PluginManager.registerCommand(pluginName, "showProgress", args => {
        progressTracker.startTracking();
        progressTracker.show();
        
        const scene = SceneManager._scene;
        if (scene && scene._progressWindow) {
            scene._progressWindow.refresh();
        }
    });
    
    PluginManager.registerCommand(pluginName, "hideProgress", args => {
        progressTracker.reset();
        
        const scene = SceneManager._scene;
        if (scene && scene._progressWindow) {
            scene._progressWindow.hide();
        }
    });
    
    PluginManager.registerCommand(pluginName, "toggleVisibility", args => {
        progressTracker.toggleVisibility();
        
        const scene = SceneManager._scene;
        if (scene && scene._progressWindow) {
            scene._progressWindow.refresh();
        }
    });
    
    PluginManager.registerCommand(pluginName, "resumeProgress", args => {
        if (progressTracker.isTracking) {
            progressTracker.show();
            
            const scene = SceneManager._scene;
            if (scene && scene._progressWindow) {
                scene._progressWindow.refresh();
            }
        }
    });

    PluginManager.registerCommand(pluginName, "recalculateProgress", args => {
        if (progressTracker.isTracking) {
            progressTracker.recalculate();
            
            const scene = SceneManager._scene;
            if (scene && scene._progressWindow) {
                scene._progressWindow.refresh();
            }
        }
    });
    
    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function() {
        _Window_Message_terminateMessage.call(this);
        
        if (progressTracker.isTracking) {
            const interpreter = progressTracker.getCurrentInterpreter();
            if (interpreter) {
                progressTracker.updatePosition(interpreter._index);
                
                const scene = SceneManager._scene;
                if (scene && scene._progressWindow) {
                    scene._progressWindow.refresh();
                }
            }
        }
    };

    const _Scene_Load_onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
        _Scene_Load_onLoadSuccess.call(this);
        
        setTimeout(() => {
            const scene = SceneManager._scene;
            if (scene && scene._progressWindow) {
                scene._progressWindow.x = posX;
                scene._progressWindow.y = posY;
                scene._progressWindow.refresh();
            }
        }, 1);
    };
    
})();