/*:
 * @target MZ
 * @plugindesc メッセージの名前を条件にして、指定したコモンイベントを実行し、変数に値を格納する
 * @author スタジオVR
 *
 * @param messageTriggers
 * @text Message Triggers
 * @desc Specify the message names to trigger before showing the message.
 * @type struct<messageTrigger>[]
 * @default []
 *
 * @param eventId
 * @text Common Event ID
 * @desc The ID of the common event to run before showing the message (default if not specified in message triggers).
 * @type common_event
 *
 * @param variableId
 * @text Variable ID
 * @desc The ID of the variable to store the value in (default if not specified in message triggers).
 * @type variable
 *
 * @help Svr_NameCommonEvent.js
 *
 * メッセージの名前を条件にして、指定したコモンイベントを実行し、変数に値を格納する
 *
 * @command enableCommonEvent
 * @text コモンイベント実行ON
 * @desc メッセージの名前を条件にしたコモンイベントの実行を有効にする
 *
 * @command disableCommonEvent
 * @text コモンイベント実行OFF
 * @desc メッセージの名前を条件にしたコモンイベントの実行を無効にする
 *
 */

/*~struct~messageTrigger:
 *
 * @param name
 * @desc The name of the message to trigger the common event for.
 * @type text
 *
 * @param eventId
 * @desc The ID of the common event to run before showing the message.
 * @type common_event
 *
 * @param variableId
 * @desc The ID of the variable to store the value in.
 * @type variable
 */

(() => {
  const pluginName = 'Svr_NameCommonEvent';
  const pluginParams = PluginManager.parameters(pluginName);
  const messageTriggers = JSON.parse(pluginParams['messageTriggers'] || '[]').map(param => JSON.parse(param));
  const globalVariableId = Number(pluginParams['variableId'] || 0);
  const globalEventId = Number(pluginParams['eventId'] || 0);
  let commonEventEnabled = false;

  // ゲームデータをセーブする際にcommonEventEnabledの状態を保存
  const _DataManager_makeSaveContents = DataManager.makeSaveContents;
  DataManager.makeSaveContents = function() {
    const contents = _DataManager_makeSaveContents.call(this);
    contents.commonEventEnabled = commonEventEnabled;
    return contents;
  };

  // ゲームデータをロードする際にcommonEventEnabledの状態を復元
  const _DataManager_extractSaveContents = DataManager.extractSaveContents;
  DataManager.extractSaveContents = function(contents) {
    _DataManager_extractSaveContents.call(this, contents);
    commonEventEnabled = contents.commonEventEnabled || false;
  };

  const _Window_Message_startMessage = Window_Message.prototype.startMessage;
  Window_Message.prototype.startMessage = function() {
    if (commonEventEnabled) {
      const name = $gameMessage.speakerName().replace(/\\n/g, '\\N');
      let matched = false;

      for (const mt of messageTriggers) {
        const trigger_name = mt["name"].replace(/\\n/g, '\\N');
        if (trigger_name === name) {
          matched = true;
          const regexN = /\\N\[(.*?)\]/;
          const regexEnemy = /\\enemy\[(.*?)\]/;
          const matchN = name.match(regexN);
          const matchEnemy = name.match(regexEnemy);
          const valueN = matchN ? Number(matchN[1]) : null;
          const valueEnemy = matchEnemy ? Number(matchEnemy[1]) : null;
          
          // 変数に値を格納
          const triggerVariableId = Number(mt["variableId"] || 0);
          if (triggerVariableId > 0) {
            if (valueN !== null) {
              $gameVariables.setValue(triggerVariableId, valueN);
            } else if (valueEnemy !== null) {
              $gameVariables.setValue(triggerVariableId, valueEnemy);
            }
          } else if (globalVariableId > 0) {
            if (valueN !== null) {
              $gameVariables.setValue(globalVariableId, valueN);
            } else if (valueEnemy !== null) {
              $gameVariables.setValue(globalVariableId, valueEnemy);
            }
          }
          
          // コモンイベントを実行
          const triggerEventId = Number(mt["eventId"] || 0);
          if (triggerEventId > 0) {
            const commonEvent = $dataCommonEvents[triggerEventId];
            const interpreter = new Game_Interpreter();
            interpreter.setup(commonEvent.list);
            while (interpreter.isRunning()) {
              interpreter.update();
            }
          } else if (globalEventId > 0) {
            const commonEvent = $dataCommonEvents[globalEventId];
            const interpreter = new Game_Interpreter();
            interpreter.setup(commonEvent.list);
            while (interpreter.isRunning()) {
              interpreter.update();
            }
          }

          break;
        }
      }

      // 名前が該当しない場合は変数に0を設定してコモンイベントを実行
      if (!matched) {
        if (globalVariableId > 0) {
          $gameVariables.setValue(globalVariableId, 0);
        }
        for (const mt of messageTriggers) {
          const triggerVariableId = Number(mt["variableId"] || 0);
          if (triggerVariableId > 0) {
            $gameVariables.setValue(triggerVariableId, 0);
          }
        }

        // グローバルコモンイベントを実行
        if (globalEventId > 0) {
          const commonEvent = $dataCommonEvents[globalEventId];
          const interpreter = new Game_Interpreter();
          interpreter.setup(commonEvent.list);
          while (interpreter.isRunning()) {
            interpreter.update();
          }
        }
      }
    }
    _Window_Message_startMessage.call(this);
  };

  PluginManager.registerCommand(pluginName, "enableCommonEvent", () => {
    commonEventEnabled = true;
  });

  PluginManager.registerCommand(pluginName, "disableCommonEvent", () => {
    commonEventEnabled = false;
  });
})();
