/*:
 * @target MZ
 * @plugindesc v1.1.0 Options tooltip extension for VisuStella MZ Options Core
 * @author Studio VR
 * @orderAfter VisuMZ_1_OptionsCore
 *
 * @help
 * ============================================================================
 * Introduction
 * ============================================================================
 * This plugin adds tooltip functionality to the Options Menu.
 * When hovering over an option, a tooltip window will appear with
 * additional information about that option.
 * 
 * You can now enable/disable tooltips for individual options through
 * the plugin parameters.
 * 
 * ============================================================================
 * Requirements
 * ============================================================================
 * This plugin requires the following plugins to be installed:
 * - VisuMZ_1_OptionsCore
 *
 * @param TooltipList
 * @text Tooltip List
 * @type struct<Tooltip>[]
 * @desc List of tooltips for option symbols
 */

/*~struct~Tooltip:
 * @param Symbol
 * @text Option Symbol
 * @type text
 * @desc The symbol of the option (must match with Options Core)
 * 
 * @param Description
 * @text Description
 * @type note
 * @desc The description to show in the tooltip
 * 
 * @param Enabled
 * @text Enable Tooltip
 * @type boolean
 * @on Enable
 * @off Disable
 * @default true
 * @desc Enable or disable this tooltip
 */

(() => {
    'use strict';
    
    const pluginName = "Svr_OptionsTooltip";
    
    // Parse Plugin Parameters
    const parameters = PluginManager.parameters(pluginName);
    const param = {
        tooltipList: JSON.parse(parameters['TooltipList'] || '[]').map(tooltip => {
            tooltip = JSON.parse(tooltip);
            return {
                symbol: tooltip.Symbol,
                description: JSON.parse(tooltip.Description),
                enabled: tooltip.Enabled === 'true' // Convert string to boolean
            };
        })
    };

    //=============================================================================
    // Window_OptionsTooltip
    //=============================================================================
    
    function Window_OptionsTooltip() {
        this.initialize(...arguments);
    }

    Window_OptionsTooltip.prototype = Object.create(Window_Base.prototype);
    Window_OptionsTooltip.prototype.constructor = Window_OptionsTooltip;

    Window_OptionsTooltip.prototype.initialize = function(rect) {
        Window_Base.prototype.initialize.call(this, rect);
        this.opacity = 255;
        this.hideTooltip();
    };

    Window_OptionsTooltip.prototype.setTooltip = function(text) {
        if (this._text !== text) {
            this._text = text;
            this.updateDimensions();
            this.refresh();
        }
    };

    Window_OptionsTooltip.prototype.updateDimensions = function() {
        if (!this._text) return;
        
        const textSize = this.calculateTextSize(this._text);
        const padding = this.padding * 2;
        
        const newWidth = textSize.width + padding;
        const newHeight = textSize.height + padding;
        
        if (this.width !== newWidth || this.height !== newHeight) {
            this.width = newWidth;
            this.height = newHeight;
            this.createContents();
        }
    };

    Window_OptionsTooltip.prototype.calculateTextSize = function(text) {
        const maxWidth = Graphics.boxWidth * 0.8;
        
        // テキスト内の \n を実際の改行に変換
        const processedText = text.replace(/\\n/g, '\n');
        
        // 改行で分割してそれぞれの行の幅を計算
        const lines = processedText.split('\n');
        let textWidth = 0;
        let textHeight = 0;
        
        lines.forEach(line => {
            // 各行のテキストステートを作成
            const textState = this.createTextState(line, 0, 0, maxWidth);
            const lineHeight = this.calcTextHeight(textState);
            const lineWidth = this.textSizeEx(line).width;
            
            textWidth = Math.max(textWidth, lineWidth);
            textHeight += lineHeight;
        });
        
        // 最小幅と最大幅の制限を適用
        textWidth = Math.max(textWidth, 100);
        textWidth = Math.min(textWidth, maxWidth);
        
        // 余白を追加
        textHeight += this.itemPadding() * (lines.length - 1);
        
        return {
            width: textWidth,
            height: textHeight
        };
    };

    Window_OptionsTooltip.prototype.textSizeEx = function(text) {
        const textState = this.createTextState(text, 0, 0, 0);
        textState.drawing = false;
        this.processAllText(textState);
        return { width: textState.outputWidth, height: textState.outputHeight };
    };

    Window_OptionsTooltip.prototype.refresh = function() {
        this.contents.clear();
        if (this._text) {
            // \n を実際の改行に変換
            const processedText = this._text.replace(/\\n/g, '\n');
            let y = 0;
            
            // 行ごとに描画
            const lines = processedText.split('\n');
            lines.forEach(line => {
                this.drawTextEx(line, 0, y);
                y += this.calcTextHeight(this.createTextState(line, 0, 0, this.contents.width));
                y += this.itemPadding(); // 行間のパディングを追加
            });
        }
    };

    Window_OptionsTooltip.prototype.showTooltip = function() {
        this.show();
        this.opacity = 255;
    };

    Window_OptionsTooltip.prototype.hideTooltip = function() {
        this.hide();
        this._text = '';
    };

    Window_OptionsTooltip.prototype.updatePosition = function(x, y) {
        const width = this.width;
        const height = this.height;
        
        if (x + width + 10 > Graphics.boxWidth) {
            x = x - width - 10;
        } else {
            x = x + 10;
        }
        
        if (y + height + 10 > Graphics.boxHeight) {
            y = y - height - 10;
        } else {
            y = y + 10;
        }
        
        x = Math.max(0, Math.min(x, Graphics.boxWidth - width));
        y = Math.max(0, Math.min(y, Graphics.boxHeight - height));
        
        this.x = x;
        this.y = y;
    };

    //=============================================================================
    // Window_Options
    //=============================================================================
    
    const _Window_Options_initialize = Window_Options.prototype.initialize;
    Window_Options.prototype.initialize = function(rect) {
        _Window_Options_initialize.call(this, rect);
        this.createTooltipWindow();
    };

    Window_Options.prototype.createTooltipWindow = function() {
        const rect = new Rectangle(0, 0, 1, 1);
        this._tooltipWindow = new Window_OptionsTooltip(rect);
        SceneManager._scene.addChild(this._tooltipWindow);
    };

    const _Window_Options_update = Window_Options.prototype.update;
    Window_Options.prototype.update = function() {
        _Window_Options_update.call(this);
        this.updateTooltip();
    };

    Window_Options.prototype.updateTooltip = function() {
        if (this.isTouchedInsideFrame()) {
            const hitIndex = this.hitIndex();
            if (hitIndex >= 0) {
                const symbol = this.commandSymbol(hitIndex);
                const tooltipData = this.getTooltipData(symbol);
                if (tooltipData && tooltipData.enabled) {
                    this._tooltipWindow.setTooltip(tooltipData.description);
                    this._tooltipWindow.updatePosition(TouchInput.x, TouchInput.y);
                    this._tooltipWindow.showTooltip();
                    return;
                }
            }
        }
        
        if (this._tooltipWindow.visible) {
            this._tooltipWindow.hideTooltip();
        }
    };

    Window_Options.prototype.getTooltipData = function(symbol) {
        return param.tooltipList.find(tooltip => tooltip.symbol === symbol);
    };

})();