/*:
 * @plugindesc マウスオーバーで指定された範囲のピクチャに対するポップアップメッセージを表示するプラグイン
 * @author スタジオVR
 *
 * @help
 * このプラグインは、指定された範囲のピクチャにマウスを重ねた際にポップアップメッセージを表示します。
 */

(function() {
    // 言語ごとのメッセージ定義
    const messages = {
        jp: {
            1: " \\i[4]イベント\\n ランダムなイベントが発生する。 ",
            2: " \\i[5]バトル\\n 敵との戦闘が発生する。 ",
            3: " \\i[5]エリート\\n 強い敵との戦闘が発生する。 ",
            4: " \\i[72]休息\\n [HP回復]か[ステータスUP]が行える。 ",
            5: " \\i[209]ショップ\\n [レアアイテム]を購入する。 ",
            6: " \\i[210]宝箱\\n [アイテム][未鑑定装備][ゴールド]を獲得する。 ",
            7: " \\i[10]ボス\\n ボスとの戦闘が発生する。 ",
            8: " \\i[70]神秘の店\\n ダンジョン内にある不思議なショップ。 ",
            9: " \\i[82]戦果の扉\\n 街へ戻り、報酬を獲得する。 "
        },
        en: {
            1: " \\i[4]Event\\n A random event occurs. ",
            2: " \\i[5]Battle\\n Begin combat with an enemy. ",
            3: " \\i[5]Elite\\n Begin combat with an strong enemy. ",
            4: " \\i[72]Rest\\n Perform [Healing] or [Status Up]. ",
            5: " \\i[209]Shop\\n Purchase [Equipment] or [Items]. ",
            6: " \\i[210]Treasure Chest\\n Obtain [Gold] or [Items]. ",
            7: " \\i[10]Boss\\n Begin combat with a boss. ",
            8: " \\i[70]Mysterious Shop\\n A mysterious shop located within the dungeon. ",
            9: " \\i[82]Door of Triumph\\n Return to the town and claim your rewards. "
        },
        cn: {
            1: " \\i[4]事件\\n 发生随机事件。 ",
            2: " \\i[5]戦斗\\n 开始与敌人戦斗。 ",
            3: " \\i[5]精英\\n 开始与强敌战斗。 ",
            4: " \\i[72]休息\\n 进行[HP恢复]或[状态UP]。 ",
            5: " \\i[209]商店\\n 购买[装备]或[物品]。 ",
            6: " \\i[210]宝箱\\n 获得[金币]或[物品]。 ",
            7: " \\i[10]Boss\\n 开始与Boss戦斗。 ",
            8: " \\i[70]神秘商店\\n 位于地下城内的神秘商店。 ",
            9: " \\i[82]戦果之扉\\n 返回城镇，领取你的奖励。 "
        }
    };

    // 現在の言語設定に基づいてメッセージを取得する関数
    const getPopupMessage = (pictureId) => {
        const languageSetting = $gameVariables.value(226); // 変数226の値を取得
        let currentLanguage;
        switch (languageSetting) {
            case 1:
                currentLanguage = 'jp';
                break;
            case 2:
                currentLanguage = 'en';
                break;
            case 3:
                currentLanguage = 'cn';
                break;
            default:
                currentLanguage = 'jp'; // デフォルトは日本語
        }
        const variableId = 2400 + pictureId;
        const variableValue = $gameVariables.value(variableId);
        return messages[currentLanguage][variableValue]; // 対応するメッセージを返す
    };

    // ピクチャにマウスが重なっているか検出する関数（不透明度を考慮）
    const isMouseOverPicture = (pictureId) => {
        const picture = $gameScreen.picture(pictureId);
        if (!picture) return false;

        const sprite = SceneManager._scene._spriteset._pictureContainer.children.find(sprite => sprite._pictureId === pictureId);
        if (!sprite || sprite.opacity === 0) return false; // 不透明度が0ならfalseを返す

        // スプライトのスケールを考慮
        const scaledWidth = sprite.width * sprite.scale.x;
        const scaledHeight = sprite.height * sprite.scale.y;
        const x = sprite.x - scaledWidth * sprite.anchor.x;
        const y = sprite.y - scaledHeight * sprite.anchor.y;

        const mouseX = TouchInput.x;
        const mouseY = TouchInput.y;

        return mouseX >= x && mouseX <= x + scaledWidth && mouseY >= y && mouseY <= y + scaledHeight;
    };

    // ピクチャの色を変更する関数
    const changePictureColor = (pictureId, color) => {
        const sprite = SceneManager._scene._spriteset._pictureContainer.children.find(sprite => sprite._pictureId === pictureId);
        if (sprite) {
            sprite.setColorTone(color);
        }
    };

    // ポップアップを表示する関数（スイッチを考慮）
    const showPopup = (pictureId) => {
        // スイッチ2431の状態を確認
        if (!$gameSwitches.value(2431)) {
            return; // スイッチ2431がOFFならばポップアップを表示しない
        }
        const message = getPopupMessage(pictureId);
        if (!message) return; // メッセージがない場合は何もしない

        const lines = message.split("\\n");
        const tempWindow = new Window_Base(new Rectangle(0, 0, 0, 0));
        const lineHeights = lines.map(line => tempWindow.calcTextHeight({ text: line }, false));
        const totalHeight = lineHeights.reduce((a, b) => a + b, 0);
        const maxWidth = Math.max(...lines.map(line => tempWindow.drawTextEx(line, 0, tempWindow.contents.height)));

        // ポップアップの幅と高さを動的に設定
        const popupWidth = maxWidth + 20; // テキストの最大幅 + マージン
        const popupHeight = totalHeight + 20; // 全テキストの高さの合計 + マージン

        const picture = $gameScreen.picture(pictureId);
        const sprite = SceneManager._scene._spriteset._pictureContainer.children.find(sprite => sprite._pictureId === pictureId);

        // スプライトのスケールを考慮したサイズを計算
        const scaledWidth = sprite.width * sprite.scale.x;
        const scaledHeight = sprite.height * sprite.scale.y;

        // ピクチャの右側にポップアップを配置するためのX座標
        let x = sprite.x - sprite.anchor.x * scaledWidth + scaledWidth;
        // ピクチャの中央にポップアップを縦に配置するためのY座標
        let y = sprite.y - sprite.anchor.y * scaledHeight + (scaledHeight / 2) - (popupHeight / 2);

        // 画面端でのはみ出しを防ぐ
        const screenWidth = Graphics.width;
        const screenHeight = Graphics.height;
        
        // X座標の調整
        if (x + popupWidth > screenWidth) {
            x = sprite.x - sprite.anchor.x * scaledWidth - popupWidth; // ピクチャの左側に配置
        }
        
        // Y座標の調整
        if (y < 0) {
            y = 0;
        } else if (y + popupHeight > screenHeight) {
            y = screenHeight - popupHeight;
        }

        // 既存のポップアップをクリア
        clearAllPopups();

        // 新しいポップアップを作成
        const popup = new Window_Base(new Rectangle(x, y, popupWidth, popupHeight));
        let offsetY = 0;
        lines.forEach((line, index) => {
            popup.drawTextEx(line, 0, offsetY);
            offsetY += lineHeights[index];
        });
        SceneManager._scene.addChild(popup);
        popups[pictureId] = popup;

        tempWindow.destroy(); // 一時ウィンドウを破棄
    };

    // 既存の全てのポップアップをクリアする関数
    const clearAllPopups = () => {
        for (let key in popups) {
            if (popups[key]) {
                SceneManager._scene.removeChild(popups[key]);
                const sprite = SceneManager._scene._spriteset._pictureContainer.children.find(sprite => sprite._pictureId === parseInt(key));
                if (sprite) {
                    sprite.setColorTone([0, 0, 0, 0]); // 色を元に戻す
                }
                delete popups[key];
            }
        }
    };

    let popups = {};

    const updatePopupDisplay = () => {
        // ピクチャーID71から87までの処理
        for (let pictureId = 71; pictureId <= 87; pictureId++) {
            checkAndDisplayPopup(pictureId);
        }

        // ピクチャーID98の処理を追加
        checkAndDisplayPopup(98);
    };

    // ポップアップ表示のチェックと表示を行う関数を追加
    const checkAndDisplayPopup = (pictureId) => {
        if (isMouseOverPicture(pictureId)) {
            if (!popups[pictureId]) {
                showPopup(pictureId);
            }
        } else {
            if (popups[pictureId]) {
                SceneManager._scene.removeChild(popups[pictureId]);
                const sprite = SceneManager._scene._spriteset._pictureContainer.children.find(sprite => sprite._pictureId === pictureId);
                if (sprite) {
                    sprite.setColorTone([0, 0, 0, 0]); // 色を元に戻す
                }
                delete popups[pictureId];
            }
        }
    };

    // シーンマップのアップデート関数を上書きします。
    const originalSceneMapUpdate = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        originalSceneMapUpdate.call(this);
        updatePopupDisplay();
    };
})();
