//=============================================================================
// Svr_PictureZoomer.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc ピクチャのズームイン/アウトを制御するプラグイン（改良版）
 * @author スタジオVR
 *
 * @param controlSwitch
 * @text 制御スイッチ
 * @desc このスイッチがONの時のみズーム効果が有効になります（停止コマンドは常に有効）
 * @type switch
 * @default 1010
 *
 * @command startZoom
 * @text ズーム開始
 * @desc 指定したピクチャのズーム動作を開始します。
 *
 * @arg pictureId
 * @type string
 * @text ピクチャID
 * @desc ズームするピクチャのID（数値か変数で指定）
 * 変数の場合は v[1] のように指定
 *
 * @arg zoomInDuration
 * @type number
 * @text ズームイン時間
 * @desc ズームインにかかる時間（フレーム）
 * @default 60
 *
 * @arg zoomOutDuration
 * @type number
 * @text ズームアウト時間
 * @desc ズームアウトにかかる時間（フレーム）
 * @default 60
 *
 * @arg maxZoom
 * @type number
 * @text 最大ズーム
 * @desc ズーム時の最大倍率（1 = 100%）
 * @default 1.5
 *
 * @arg waitDuration
 * @type number
 * @text ウェイト時間
 * @desc ズームイン/アウト後のウェイト時間（フレーム）
 * @default 0
 *
 * @arg loopMode
 * @type boolean
 * @text ループモード
 * @desc ズーム動作をループさせるかどうか
 * @default true
 * @on ループする
 * @off 一回のみ
 *
 * @arg shakeMode
 * @type boolean
 * @text シェイクモード
 * @desc ズーム中にシェイク効果を付けるかどうか
 * @default false
 * @on シェイクする
 * @off シェイクしない
 *
 * @arg shakeStrength
 * @type number
 * @text シェイク強さ
 * @desc シェイクの強さ（ピクセル単位）
 * @default 5
 * @min 1
 * @max 100
 *
 * @arg shakeSpeed
 * @type number
 * @text シェイク速度
 * @desc シェイクの速度（1-10、大きいほど速い）
 * @default 5
 * @min 1
 * @max 10
 *
 * @command stopZoom
 * @text ズーム停止
 * @desc 指定したピクチャのズーム動作を停止し、元のサイズに戻します。
 *
 * @arg pictureId
 * @type string
 * @text ピクチャID
 * @desc 停止するピクチャのID（数値か変数で指定）
 * 変数の場合は v[1] のように指定
 *
 * @command stopAllZoom
 * @text 全ズーム停止
 * @desc 全てのピクチャのズーム動作を停止し、元のサイズに戻します。
 *
 * @help
 * このプラグインは、ピクチャのズームイン/アウトを制御します。
 * 
 * プラグインコマンド:
 *   ズーム開始 - 指定したピクチャのズーム動作を開始します。
 *   ズーム停止 - 指定したピクチャのズーム動作を停止し、元のサイズに戻します。
 *   全ズーム停止 - 全てのピクチャのズーム動作を停止し、元のサイズに戻します。
 * 
 * ピクチャIDの指定方法：
 * - 直接数値を指定：例）1, 2, 3
 * - 変数で指定：例）v[1]  # 変数1番の値を使用
 * 
 * プラグインパラメーター：
 * - 制御スイッチ：指定したスイッチがONのときのみズーム効果が有効になります。
 *   ※ズーム停止コマンドは、スイッチの状態に関係なく常に実行できます。
 * 
 * 新機能:
 * - ズームイン/アウト後にウェイト時間を設定できます。
 * - ピクチャIDを変数で指定できます。
 * - ループモードの選択が可能になりました。
 * - シェイク効果を追加できます。
 *   - シェイクの強さと速度を調整可能
 * - 全ピクチャのズーム動作を一括停止できます。
 */

(() => {
    const pluginName = "Svr_PictureZoomer";
    const parameters = PluginManager.parameters(pluginName);
    const zoomingPictures = {};
    const CONTROL_SWITCH_ID = Number(parameters['controlSwitch'] || 1010);

    // Sprite_Picture の拡張
    const _Sprite_Picture_initialize = Sprite_Picture.prototype.initialize;
    Sprite_Picture.prototype.initialize = function(pictureId) {
        _Sprite_Picture_initialize.call(this, pictureId);
        this._pictureId = pictureId;
    };

    const _Sprite_Picture_updatePosition = Sprite_Picture.prototype.updatePosition;
    Sprite_Picture.prototype.updatePosition = function() {
        _Sprite_Picture_updatePosition.call(this);
        if ($gameSwitches.value(CONTROL_SWITCH_ID)) {
            const zoomData = zoomingPictures[this._pictureId];
            if (zoomData && zoomData.shakeMode) {
                this.x += Math.round(Math.cos(zoomData.shakePhase) * zoomData.shakeStrength);
                this.y += Math.round(Math.sin(zoomData.shakePhase * 1.3) * zoomData.shakeStrength);
            }
        }
    };

    // 変数の解析処理
    const getVariableId = function(str) {
        if (str.match(/v\[(\d+)\]/i)) {
            return Number(RegExp.$1);
        }
        return 0;
    };

    // pictureIdの解析処理
    const getPictureId = function(idString) {
        const varId = getVariableId(idString);
        if (varId > 0) {
            return $gameVariables.value(varId);
        }
        return Number(idString);
    };

    PluginManager.registerCommand(pluginName, "startZoom", args => {
        if (!$gameSwitches.value(CONTROL_SWITCH_ID)) return;

        const pictureId = getPictureId(args.pictureId);
        const zoomInDuration = Number(args.zoomInDuration);
        const zoomOutDuration = Number(args.zoomOutDuration);
        const maxZoom = Number(args.maxZoom);
        const waitDuration = Number(args.waitDuration);
        const loopMode = args.loopMode === "true";
        const shakeMode = args.shakeMode === "true";
        const shakeStrength = Number(args.shakeStrength || 5);
        const shakeSpeed = Number(args.shakeSpeed || 5);

        if (pictureId > 0) {
            zoomingPictures[pictureId] = {
                zoomInDuration: zoomInDuration,
                zoomOutDuration: zoomOutDuration,
                maxZoom: maxZoom,
                waitDuration: waitDuration,
                loopMode: loopMode,
                currentZoom: 1,
                zoomingIn: true,
                waiting: false,
                shouldZoomOut: false,
                frame: 0,
                shakeMode: shakeMode,
                shakeStrength: shakeStrength,
                shakeSpeed: shakeSpeed,
                shakePhase: 0
            };
        }
    });

    PluginManager.registerCommand(pluginName, "stopZoom", args => {
        // stopZoomはスイッチの状態に関係なく実行可能
        const pictureId = getPictureId(args.pictureId);
        if (pictureId > 0 && zoomingPictures[pictureId]) {
            const picture = $gameScreen.picture(pictureId);
            if (picture) {
                picture.setZoom(100);
            }
            delete zoomingPictures[pictureId];
        }
    });

    // 新機能：全ピクチャのズーム停止
    PluginManager.registerCommand(pluginName, "stopAllZoom", () => {
        // すべてのズーム中のピクチャを取得
        const zoomingIds = Object.keys(zoomingPictures);
        
        // 各ピクチャのズームを停止
        for (const pictureId of zoomingIds) {
            const picture = $gameScreen.picture(Number(pictureId));
            if (picture) {
                picture.setZoom(100);
            }
            delete zoomingPictures[pictureId];
        }
    });

    const _Game_Screen_update = Game_Screen.prototype.update;
    Game_Screen.prototype.update = function() {
        _Game_Screen_update.call(this);
        if ($gameSwitches.value(CONTROL_SWITCH_ID)) {
            this.updateZoomingPictures();
        }
    };

    Game_Screen.prototype.updateZoomingPictures = function() {
        for (const [pictureId, zoomData] of Object.entries(zoomingPictures)) {
            const picture = this.picture(Number(pictureId));
            if (picture) {
                zoomData.frame++;

                // シェイクの更新
                if (zoomData.shakeMode) {
                    zoomData.shakePhase += zoomData.shakeSpeed * 0.1;
                }
                
                if (zoomData.waiting) {
                    if (zoomData.frame >= zoomData.waitDuration) {
                        zoomData.waiting = false;
                        zoomData.frame = 0;
                        
                        if (!zoomData.loopMode) {
                            zoomData.zoomingIn = false;
                            zoomData.shouldZoomOut = true;
                        } else {
                            zoomData.zoomingIn = !zoomData.zoomingIn;
                        }
                    }
                } else if (zoomData.zoomingIn) {
                    zoomData.currentZoom = this.calculateZoom(zoomData.frame, 1, zoomData.maxZoom, zoomData.zoomInDuration);
                    if (zoomData.frame >= zoomData.zoomInDuration) {
                        zoomData.waiting = true;
                        zoomData.frame = 0;
                    }
                } else {
                    zoomData.currentZoom = this.calculateZoom(zoomData.frame, zoomData.maxZoom, 1, zoomData.zoomOutDuration);
                    if (zoomData.frame >= zoomData.zoomOutDuration) {
                        if (!zoomData.loopMode && zoomData.shouldZoomOut) {
                            picture.setZoom(100);
                            delete zoomingPictures[pictureId];
                            return;
                        }
                        zoomData.waiting = true;
                        zoomData.frame = 0;
                    }
                }
                
                picture.setZoom(zoomData.currentZoom * 100);
            }
        }
    };

    Game_Screen.prototype.calculateZoom = function(frame, startZoom, endZoom, duration) {
        const t = frame / duration;
        return startZoom + (endZoom - startZoom) * t;
    };

    // Game_Picture クラスの拡張
    const _Game_Picture_initBasic = Game_Picture.prototype.initBasic;
    Game_Picture.prototype.initBasic = function() {
        _Game_Picture_initBasic.call(this);
        this._zoomX = 100;
        this._zoomY = 100;
    };

    Game_Picture.prototype.setZoom = function(zoom) {
        this._zoomX = zoom;
        this._zoomY = zoom;
    };

    const _Game_Picture_scaleX = Game_Picture.prototype.scaleX;
    Game_Picture.prototype.scaleX = function() {
        return _Game_Picture_scaleX.call(this) * (this._zoomX / 100);
    };

    const _Game_Picture_scaleY = Game_Picture.prototype.scaleY;
    Game_Picture.prototype.scaleY = function() {
        return _Game_Picture_scaleY.call(this) * (this._zoomY / 100);
    };
})();