/*:
 * @target MZ
 * @plugindesc クエストのクリア状況を管理し、関連変数をインクリメントし、宿屋のレベルに応じて依頼ランクを解放するプラグイン
 * @author スタジオVR
 *
 * @param MainVariableID
 * @text メイン変数ID
 * @desc クエストIDを格納する変数のID（1~100）
 * @type number
 * @default 1141
 *
 * @param SwitchStartID
 * @text スイッチ開始ID
 * @desc クエストスイッチの開始ID
 * @type number
 * @default 1801
 *
 * @param SwitchEndID
 * @text スイッチ終了ID
 * @desc クエストスイッチの終了ID
 * @type number
 * @default 1900
 *
 * @param ItemStartID
 * @text アイテム開始ID
 * @desc クエストアイテムの開始ID
 * @type number
 * @default 801
 *
 * @param ItemEndID
 * @text アイテム終了ID
 * @desc クエストアイテムの終了ID
 * @type number
 * @default 900
 *
 * @param IncrementVariableStartID
 * @text インクリメント変数開始ID
 * @desc クエストに対応する変数の開始ID
 * @type number
 * @default 3401
 *
 * @param IncrementVariableEndID
 * @text インクリメント変数終了ID
 * @desc クエストに対応する変数の終了ID
 * @type number
 * @default 3500
 *
 * @param DebugSwitchID
 * @text デバッグスイッチID
 * @desc デバッグログを表示するスイッチのID
 * @type number
 * @default 1003
 *
 * @param AdditionalVariableID
 * @text 追加機能変数ID（宿屋のレベル）
 * @desc スイッチを制御するための追加変数のID（宿屋のレベルを表す）
 * @type number
 * @default 2901
 *
 * @param Conditions
 * @text 条件設定
 * @type struct<Condition>[]
 * @desc 条件としきい値、スイッチIDの設定
 *
 * @command EnableQuestSwitches
 * @text クエストスイッチ有効化
 * @desc メイン変数の値（1~100）に基づいてスイッチ開始ID~スイッチ終了IDをONにします。
 *
 * @command IncrementQuestVariable
 * @text クエスト変数インクリメント
 * @desc メイン変数に対応する変数開始ID~変数終了IDの数値に1を加算します。
 *
 * @command EnableAdditionalSwitches
 * @text 依頼ランク解放
 * @desc 宿屋のレベルに基づいて依頼ランクのスイッチをONにします。
 *
 * @help
 * ============================================================================
 * Svr_QuestIconManager.js
 * ============================================================================
 * このプラグインは、クエストのクリア状況を管理し、関連する変数をインクリメントします。
 * また、宿屋のレベルに応じて依頼ランクのスイッチを解放します。
 *
 * 【プラグインパラメーター】
 *   - メイン変数ID: クエストIDを格納する変数のIDを設定します。
 *   - スイッチ開始ID / スイッチ終了ID: クエストスイッチのID範囲を設定します。
 *   - アイテム開始ID / アイテム終了ID: クエストアイテムのID範囲を設定します。
 *   - インクリメント変数開始ID / インクリメント変数終了ID: クエストに対応する変数のID範囲を設定します。
 *   - デバッグスイッチID: デバッグログを表示するためのスイッチIDを設定します。
 *   - 追加機能変数ID（宿屋のレベル）: スイッチを制御するための追加変数のIDを設定します。
 *   - 条件設定: 条件としきい値、スイッチIDの設定を自由に追加できます。
 *
 * 【プラグインコマンド】
 *   1. Svr_QuestIconManager EnableQuestSwitches
 *      - メイン変数の値（1~100）に基づいて対応するスイッチ開始ID～スイッチ終了IDをONにします。
 *
 *   2. Svr_QuestIconManager IncrementQuestVariable
 *      - メイン変数に対応する変数開始ID～変数終了IDの数値に1を加算します。
 *
 *   3. Svr_QuestIconManager EnableAdditionalSwitches
 *      - 宿屋のレベルに基づいて依頼ランクのスイッチをONにします。
 *        条件は設定画面で自由に追加できます。
 *
 * 【使用方法】
 *   1. プラグインを導入し、プラグイン管理から有効化します。
 *   2. プラグイン管理画面で各パラメーターを設定します。
 *      - 特に「追加機能変数ID（宿屋のレベル）」には、宿屋のレベルを管理する変数IDを設定します。
 *      - 「条件設定」で必要な数だけ条件を追加し、しきい値とスイッチIDを設定します。
 *   3. クエストが完了した際に、メイン変数をクエストIDに対応する値（1~100）に設定します。
 *   4. その後、プラグインコマンド「EnableQuestSwitches」を実行して対応するスイッチをONにします。
 *   5. クエストの進行に応じて、プラグインコマンド「IncrementQuestVariable」を実行し、
 *      対応する変数開始ID～変数終了IDをインクリメントします。
 *   6. 宿屋のレベルが上昇した際に、プラグインコマンド「EnableAdditionalSwitches」を実行して
 *      依頼ランクのスイッチをONにします。
 *
 * 【注意事項】
 *   - セーブデータに影響を与えるため、使用時には注意してください。
 *   - デバッグログはスイッチID1003がONの場合のみ表示されます。
 *
 * ============================================================================
 */

/*~struct~Condition:
 * @param threshold
 * @text しきい値
 * @desc 追加変数がこの値以上の場合にスイッチをONにします。変数の場合は v[ID] と指定します。
 * @type string
 * 
 * @param switches
 * @text スイッチID
 * @desc 条件を満たす場合にONにするスイッチIDをカンマ区切りで指定します
 * @type string
 */

(() => {
    const PLUGIN_NAME = "Svr_QuestIconManager";
    
    // プラグインパラメータの取得
    const parameters = PluginManager.parameters(PLUGIN_NAME);
    const MAIN_VARIABLE_ID = Number(parameters['MainVariableID'] || 1141);
    const SWITCH_START_ID = Number(parameters['SwitchStartID'] || 1801);
    const SWITCH_END_ID = Number(parameters['SwitchEndID'] || 1900);
    const ITEM_START_ID = Number(parameters['ItemStartID'] || 801);
    const ITEM_END_ID = Number(parameters['ItemEndID'] || 900);
    const INCREMENT_VARIABLE_START_ID = Number(parameters['IncrementVariableStartID'] || 3401);
    const INCREMENT_VARIABLE_END_ID = Number(parameters['IncrementVariableEndID'] || 3500);
    const DEBUG_SWITCH_ID = Number(parameters['DebugSwitchID'] || 1003);
    const ADDITIONAL_VARIABLE_ID = Number(parameters['AdditionalVariableID'] || 2901);
    
    // 条件の設定を解析
    const conditions = JSON.parse(parameters['Conditions'] || '[]').map(conditionJson => {
        const condition = JSON.parse(conditionJson);
        return {
            threshold: condition.threshold,
            switches: condition.switches.split(',').map(s => Number(s.trim()))
        };
    });

    // しきい値の評価関数
    function evaluateThreshold(threshold) {
        if (threshold.toString().startsWith('v[') && threshold.toString().endsWith(']')) {
            const variableId = parseInt(threshold.toString().slice(2, -1));
            return $gameVariables.value(variableId);
        } else {
            return Number(threshold);
        }
    }

    // プラグインコマンドの登録
    PluginManager.registerCommand(PLUGIN_NAME, "EnableQuestSwitches", args => {
        // 変数の値を取得
        let value = $gameVariables.value(MAIN_VARIABLE_ID);

        // 値が1未満または100を超える場合は範囲内に収める
        if (value < 1) {
            value = 1;
        } else if (value > 100) {
            value = 100;
        }

        // 対応するスイッチIDを計算
        const switchIdToEnable = SWITCH_START_ID + value - 1;

        // 対応するスイッチをONにする
        $gameSwitches.setValue(switchIdToEnable, true);

        // デバッグスイッチがONかどうかをチェック
        const isDebug = $gameSwitches.value(DEBUG_SWITCH_ID);

        // デバッグ用ログ
        if (isDebug) {
            console.log(`Svr_QuestIconManager: 変数${MAIN_VARIABLE_ID}の値は${value}です。スイッチ${switchIdToEnable}をONにしました。`);
        }
    });

    PluginManager.registerCommand(PLUGIN_NAME, "IncrementQuestVariable", args => {
        // メイン変数の値を取得
        let questIndex = $gameVariables.value(MAIN_VARIABLE_ID);

        // 値が1未満または100を超える場合は範囲内に収める
        if (questIndex < 1) {
            questIndex = 1;
        } else if (questIndex > 100) {
            questIndex = 100;
        }

        // 対応する変数IDを計算
        const variableIdToIncrement = INCREMENT_VARIABLE_START_ID + questIndex - 1;

        // 対応する変数に1を加算
        let currentValue = $gameVariables.value(variableIdToIncrement);
        $gameVariables.setValue(variableIdToIncrement, currentValue + 1);

        // デバッグスイッチがONかどうかをチェック
        const isDebug = $gameSwitches.value(DEBUG_SWITCH_ID);

        // デバッグ用ログ
        if (isDebug) {
            console.log(`Svr_QuestIconManager: 変数${MAIN_VARIABLE_ID}の値は${questIndex}です。変数${variableIdToIncrement}の値を${currentValue}から${currentValue + 1}に加算しました。`);
        }
    });

    PluginManager.registerCommand(PLUGIN_NAME, "EnableAdditionalSwitches", args => {
        // 追加変数の値を取得（宿屋のレベル）
        let additionalValue = $gameVariables.value(ADDITIONAL_VARIABLE_ID);

        // デバッグスイッチがONかどうかをチェック
        const isDebug = $gameSwitches.value(DEBUG_SWITCH_ID);

        // デバッグ用ログ
        if (isDebug) {
            console.log(`Svr_QuestIconManager: 追加変数${ADDITIONAL_VARIABLE_ID}の値は${additionalValue}です。`);
        }

        // 各条件をチェック
        conditions.forEach((condition, index) => {
            const thresholdValue = evaluateThreshold(condition.threshold);
            const isConditionMet = additionalValue >= thresholdValue;
            
            if (isDebug) {
                console.log(`Svr_QuestIconManager: 条件${index + 1} - 変数${ADDITIONAL_VARIABLE_ID}(${additionalValue}) ${isConditionMet ? '>=' : '<'} しきい値${thresholdValue}`);
            }

            condition.switches.forEach(switchId => {
                if (isConditionMet && !$gameSwitches.value(switchId)) {
                    $gameSwitches.setValue(switchId, true);
                    if (isDebug) {
                        console.log(`Svr_QuestIconManager: スイッチ${switchId}をONにしました。`);
                    }
                } else if (!isConditionMet && $gameSwitches.value(switchId)) {
                    $gameSwitches.setValue(switchId, false);
                    if (isDebug) {
                        console.log(`Svr_QuestIconManager: スイッチ${switchId}をOFFにしました。`);
                    }
                }
            });
        });
    });

})();