/*:
 * @target MZ
 * @plugindesc ランダムテキストを表示するプラグイン
 * @author スタジオVR
 *
 * @param RandomTextList
 * @type struct<RandomTextStruct>[]
 * @desc 各IDごとのテキストリスト
 * @default []
 *
 * @help
 * このプラグインは、\randtext[x] の形式で指定された場所に、
 * ID x に対応するテキストの中からランダムに1つを表示します。
 * 一度表示したテキストは再度表示されず、すべてのテキストが表示された後は履歴がリセットされます。
 *
 * 条件式では以下の記法が使用できます：
 * \S[n] - スイッチn番の状態を参照
 * \V[n] - 変数n番の値を参照
 * 例: \S[1] && \V[2]>=10
 *
 * 各テキストリストで設定したConditionStartID以降の番号のテキストは、
 * 条件を満たす場合のみ表示候補として選択されます。
 * 条件を満たすテキストがない場合は履歴をリセットして再試行します。
 *
 * 使用方法:
 * メッセージ内で \randtext[x] のように指定します。
 * 複数行のメッセージを登録する場合は、プラグインパラメータのTextsに
 * ["１行目\n２行目", "他の文章"] のように \n で区切って入力してください。
 */

/*~struct~RandomTextStruct:
 *
 * @param ID
 * @type number
 * @desc テキストリストのID
 * @default 1
 *
 * @param Texts
 * @type note[]
 * @desc 表示するテキストのリスト（複数行可）
 * @default ["テキスト1","テキスト2","テキスト3"]
 *
 * @param Condition
 * @type text
 * @desc 条件式（空欄の場合は常に表示）
 * 例: \S[1] && \V[2]>=10
 * @default 
 *
 * @param ConditionStartID
 * @type number
 * @desc この番号以降のテキストに条件チェックが適用されます
 * @default 1
 */

(() => {
    'use strict';
    
    const pluginName = "Svr_RandomText";
    
    const parameters = PluginManager.parameters(pluginName);
    const textListData = {};

    // 条件式を評価する関数
    function evaluateCondition(condition) {
        if (!condition) return true;
        
        // スイッチと変数の参照を実際の値に置換
        const evalText = condition
            .replace(/\\S\[(\d+)\]/gi, (_, id) => `$gameSwitches.value(${id})`)
            .replace(/\\V\[(\d+)\]/gi, (_, id) => `$gameVariables.value(${id})`);
        
        try {
            // Function コンストラクタを使用して安全に式を評価
            return (new Function('return ' + evalText))();
        } catch (e) {
            console.error('条件式の評価エラー:', condition, e);
            return false;
        }
    }

    // JSON文字列を安全にパースする関数
    function parseJsonSafe(jsonString) {
        try {
            return JSON.parse(jsonString);
        } catch (e) {
            console.error('JSONパースエラー:', e);
            return null;
        }
    }

    // プラグインパラメータの解析
    try {
        const rawList = parseJsonSafe(parameters["RandomTextList"]) || [];
        rawList.forEach((entryJson) => {
            const entry = parseJsonSafe(entryJson);
            if (entry && entry.ID) {
                const id = Number(entry.ID);
                textListData[id] = {
                    condition: entry.Condition || '',
                    conditionStartID: Number(entry.ConditionStartID || 1),
                    texts: [],
                    normalizedTexts: []
                };

                let texts = parseJsonSafe(entry.Texts);
                if (Array.isArray(texts)) {
                    // オリジナルのテキストを保存
                    textListData[id].texts = texts;
                    // 正規化されたテキストを保存
                    textListData[id].normalizedTexts = texts.map(text => {
                        text = String(text).replace(/^"(.*)"$/, '$1');
                        return text.replace(/\\n/g, '\n')
                                 .split('\n')
                                 .map(line => line.trimRight())
                                 .join('\n');
                    });
                }
            }
        });
    } catch (error) {
        console.error("プラグインパラメータの解析エラー:", error);
    }

    // Game_Systemに履歴を保存するためのプロパティを追加
    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function() {
        _Game_System_initialize.call(this);
        this._randomTextHistory = {};
    };

    // 指定したIDの条件を満たすテキストを取得
    Game_System.prototype.getAvailableTexts = function(textList) {
        let availableTexts = [];
        
        for (let i = 0; i < textList.texts.length; i++) {
            const textIndex = i + 1;  // テキストの番号は1から始まる
            const shouldCheckCondition = textIndex >= textList.conditionStartID;
            
            // 条件チェックが必要ない、または条件を満たす場合
            if (!shouldCheckCondition || evaluateCondition(textList.condition)) {
                availableTexts.push(i); // インデックスを保存
            }
        }
        
        return availableTexts;
    };

    // ランダムテキストを取得する関数
    Game_System.prototype.getRandomText = function(id, retryCount = 0) {
        // 無限ループ防止
        if (retryCount > 3) {
            return `\\C[2]ID ${id} の表示可能なテキストがありません\\C[0]`;
        }

        const textList = textListData[id];
        if (!textList || textList.normalizedTexts.length === 0) {
            return `\\C[2]ID ${id} のテキストが未登録です\\C[0]`;
        }

        // 条件を満たすテキストのインデックスを取得
        const availableIndices = this.getAvailableTexts(textList);
        if (availableIndices.length === 0) {
            return `\\C[2]ID ${id} の条件を満たすテキストがありません\\C[0]`;
        }

        if (!this._randomTextHistory[id]) {
            this._randomTextHistory[id] = [];
        }

        const history = this._randomTextHistory[id];

        // 履歴に含まれていないテキストのインデックスをフィルタリング
        const availableUnusedIndices = availableIndices.filter(
            index => !history.includes(textList.texts[index])
        );

        // 未使用のテキストがない場合は履歴をリセットして再試行
        if (availableUnusedIndices.length === 0) {
            history.length = 0;
            return this.getRandomText(id, retryCount + 1);
        }

        // ランダムに選択
        const randomIndex = Math.floor(Math.random() * availableUnusedIndices.length);
        const selectedIndex = availableUnusedIndices[randomIndex];
        
        // 履歴に追加
        history.push(textList.texts[selectedIndex]);
        
        return textList.normalizedTexts[selectedIndex];
    };

    // Window_Base の drawTextEx を拡張して余分なスペースを削除
    const _Window_Base_drawTextEx = Window_Base.prototype.drawTextEx;
    Window_Base.prototype.drawTextEx = function(text, x, y) {
        if (text) {
            text = text.split('\n')
                      .map(line => line.trimRight())
                      .join('\n');
        }
        return _Window_Base_drawTextEx.call(this, text, x, y);
    };

    // メッセージ処理の拡張
    const _Window_Message_processEscapeCharacter = Window_Message.prototype.processEscapeCharacter;
    Window_Message.prototype.processEscapeCharacter = function(code, textState) {
        switch (code.toUpperCase()) {
            case 'RANDTEXT':
                const text = textState.text.slice(textState.index);
                const match = /^\[(\d+)\]/.exec(text);
                if (match) {
                    textState.index += match[0].length;
                    const id = parseInt(match[1]);
                    const randomText = $gameSystem.getRandomText(id);
                    if (randomText) {
                        const processedText = randomText.split('\n')
                                                      .map(line => line.trimRight())
                                                      .join('\n');
                        this.drawTextEx(processedText, textState.x, textState.y);
                        textState.x += this.textWidth(this.convertEscapeCharacters(processedText));
                    }
                }
                break;
            default:
                _Window_Message_processEscapeCharacter.call(this, code, textState);
                break;
        }
    };

})();