/*:
 * @target MZ
 * @plugindesc ランダムな数値を管理するプラグイン
 * @author Your Name
 * @help
 * 各IDに対して設定された数値の範囲からランダムな順序で10パターン生成し、
 * プラグインコマンドで取得できるプラグインです。
 * 
 * @command GetRandomValue
 * @text ランダム値の取得
 * @desc 指定したIDに対応するランダムな数値を取得します
 * 
 * @arg id
 * @text ID
 * @type string
 * @desc 取得したい数値のID
 * 
 * @command PeekNextValue
 * @text 次の値を確認
 * @desc 次に取得される値を確認します（値は変更されません）
 * 
 * @arg id
 * @text ID
 * @type string
 * @desc 確認したい数値のID
 * 
 * @param ResultVariableId
 * @text 結果格納変数
 * @type variable
 * @desc 取得した値を格納する変数ID
 * @default 1
 * 
 * @param ValueSettings
 * @text 数値設定
 * @type struct<ValueData>[]
 * @desc IDと数値範囲のペアを設定します
 */

/*~struct~ValueData:
 * @param id
 * @text ID
 * @type string
 * @desc 数値を識別するためのID
 * 
 * @param minValue
 * @text 最小値
 * @type number
 * @min -9999
 * @max 9999
 * @desc 生成される数値の最小値
 * 
 * @param maxValue
 * @text 最大値
 * @type number
 * @min -9999
 * @max 9999
 * @desc 生成される数値の最大値
 */

(() => {
    'use strict';
    
    const pluginName = "Svr_RandomValuePlugin";

    //-----------------------------------------------------------------------------
    // Game_RandomValues
    //
    class Game_RandomValues {
        constructor() {
            this.initialize();
        }

        initialize() {
            this._resultVariableId = 1;
            this._valueSettings = [];
            this._randomPatterns = {};
            this._currentIndices = {};
            this._isInitialized = false;
        }

        setParameters(parameters) {
            if (this._isInitialized) return;
            
            const parsed = this._parsePluginParameters(parameters);
            this._resultVariableId = parsed.resultVariableId;
            this._valueSettings = parsed.valueSettings;
            this._initializePatterns();
            this._isInitialized = true;
        }

        _parsePluginParameters(parameters) {
            const resultVarId = Number(parameters['ResultVariableId'] || 1);
            const parsed = JSON.parse(parameters['ValueSettings'] || '[]');
            return {
                resultVariableId: resultVarId,
                valueSettings: parsed.map(valueDataJSON => {
                    const valueData = JSON.parse(valueDataJSON);
                    return {
                        id: valueData.id,
                        minValue: Number(valueData.minValue),
                        maxValue: Number(valueData.maxValue)
                    };
                })
            };
        }

        _initializePatterns() {
            // IDごとの設定を集める
            const settingsByID = this._valueSettings.reduce((acc, setting) => {
                if (!acc[setting.id]) {
                    acc[setting.id] = [];
                }
                acc[setting.id].push({
                    minValue: setting.minValue,
                    maxValue: setting.maxValue
                });
                return acc;
            }, {});

            // 各IDに対して初期化
            Object.entries(settingsByID).forEach(([id, settings]) => {
                this._randomPatterns[id] = Array(10).fill().map(() => this._generateRandomValue(settings[0]));
                this._currentIndices[id] = 0;
            });
        }

        _generateRandomValue(range) {
            const min = Math.min(range.minValue, range.maxValue);
            const max = Math.max(range.minValue, range.maxValue);
            return Math.floor(Math.random() * (max - min + 1)) + min;
        }

        getValue(id) {
            if (!this._randomPatterns[id]) return null;

            const setting = this._valueSettings.find(s => s.id === id);
            const value = this._randomPatterns[id][this._currentIndices[id]];
            this._randomPatterns[id][this._currentIndices[id]] = this._generateRandomValue(setting);
            this._currentIndices[id] = (this._currentIndices[id] + 1) % 10;
            return value;
        }

        peekValue(id) {
            if (!this._randomPatterns[id]) return null;
            return this._randomPatterns[id][this._currentIndices[id]];
        }

        getResultVariableId() {
            return this._resultVariableId;
        }
    }

    //-----------------------------------------------------------------------------
    // プラグインの初期化と管理

    let $gameRandomValues = null;

    const _createGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function() {
        _createGameObjects.call(this);
        $gameRandomValues = new Game_RandomValues();
    };

    const _makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function() {
        const contents = _makeSaveContents.call(this);
        contents.randomValues = $gameRandomValues;
        return contents;
    };

    const _extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _extractSaveContents.call(this, contents);
        if (contents.randomValues) {
            $gameRandomValues = contents.randomValues;
            Object.setPrototypeOf($gameRandomValues, Game_RandomValues.prototype);
        }
    };

    //-----------------------------------------------------------------------------
    // プラグインコマンド

    PluginManager.registerCommand(pluginName, "GetRandomValue", args => {
        $gameRandomValues.setParameters(PluginManager.parameters(pluginName));
        const value = $gameRandomValues.getValue(args.id);
        if (value !== null) {
            $gameVariables.setValue($gameRandomValues.getResultVariableId(), value);
        }
    });

    PluginManager.registerCommand(pluginName, "PeekNextValue", args => {
        $gameRandomValues.setParameters(PluginManager.parameters(pluginName));
        const value = $gameRandomValues.peekValue(args.id);
        if (value !== null) {
            $gameVariables.setValue($gameRandomValues.getResultVariableId(), value);
        }
    });
})();