//=============================================================================
// SvrRememberChoiceCursor.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 選択肢のカーソル位置を記憶するプラグイン
 * @author Your Name
 * 
 * @param DefaultBehavior
 * @text デフォルト動作
 * @type boolean
 * @on 記憶する
 * @off 記憶しない
 * @default true
 * @desc IDが設定されていない選択肢の場合の動作を設定します。
 * 
 * @help
 * Svr_RememberChoiceCursor.js
 * 
 * ■概要
 * このプラグインは、選択肢のカーソル位置を記憶し、次回同じ選択肢を
 * 表示したときに、前回選んだ位置にカーソルを移動させます。
 * また、キャンセル時の動作も記憶し、イベントコマンドで設定した
 * キャンセル時の処理に従って適切な位置を記憶します。
 * 
 * ■主な機能
 * 1. 選択肢のカーソル位置の記憶
 * 2. キャンセル時の選択位置の記憶
 * 3. 選択肢ごとに異なる設定の保存
 * 4. セーブデータへの設定の保存
 * 
 * ■使用方法
 * 
 * ▼基本的な使い方
 * プラグインを導入するだけで、選択肢のカーソル位置が記憶されます。
 * ただし、この場合は全ての選択肢が「デフォルト」として扱われます。
 * 
 * ▼選択肢の識別
 * 特定の選択肢だけを記憶したい場合や、異なる場所にある同じ選択肢を
 * 紐付けたい場合は、選択肢の文章に以下の制御文字を追加します：
 * 
 * \choiceId[識別子]
 * 
 * 例1：町の施設選択の場合
 * \choiceId[town_menu]宿屋
 * \choiceId[town_menu]道具屋
 * \choiceId[town_menu]武器屋
 * 
 * 例2：はい/いいえの選択肢の場合
 * \choiceId[yes_no]はい
 * \choiceId[yes_no]いいえ
 * 
 * ▼キャンセル時の動作
 * イベントコマンドの「選択肢の表示」で設定したキャンセル時の
 * 処理に応じて、適切にカーソル位置を記憶します：
 * 
 * ・分岐の最後の選択肢：最後の選択肢を記憶
 * ・分岐を飛ばす：キャンセル設定を記憶
 * ・特定の選択肢：指定された選択肢を記憶
 * 
 * ▼プラグインパラメータ
 * デフォルト動作：
 * IDが設定されていない選択肢の動作を指定します。
 * ON: カーソル位置を記憶します
 * OFF: カーソル位置を記憶しません
 * 
 * ■注意事項
 * 1. 制御文字(\choiceId[])は必ず選択肢の文章の先頭に配置してください。
 * 2. 同じ選択肢セット内では、同じIDを使用してください。
 * 3. 選択肢の数が変わる場合は、新しい識別子を使用することを推奨します。
 * 4. プラグインコマンドはありません。
 * 
 * ■更新履歴
 * ver 1.0.0 (2024.10.30) - 公開
 */

(() => {
    'use strict';

    const pluginName = 'Svr_RememberChoiceCursor';

    // プラグインパラメータの取得
    const parameters = PluginManager.parameters(pluginName);
    const defaultBehavior = parameters['DefaultBehavior'] !== 'false';

    // Game_Systemの初期化時にカーソル位置を保存するプロパティを追加
    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function() {
        _Game_System_initialize.call(this);
        this._choiceCursorPositions = {};
        this._choiceCancelPositions = {};
    };

    // セーブデータにカーソル位置を含める
    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function() {
        const contents = _DataManager_makeSaveContents.call(this);
        contents.choiceCursorPositions = $gameSystem._choiceCursorPositions;
        contents.choiceCancelPositions = $gameSystem._choiceCancelPositions;
        return contents;
    };

    // セーブデータからカーソル位置を復元
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.call(this, contents);
        if (contents.choiceCursorPositions) {
            $gameSystem._choiceCursorPositions = contents.choiceCursorPositions;
        }
        if (contents.choiceCancelPositions) {
            $gameSystem._choiceCancelPositions = contents.choiceCancelPositions;
        }
    };

    // 制御文字の処理を追加
    const _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
    Window_Base.prototype.convertEscapeCharacters = function(text) {
        text = _Window_Base_convertEscapeCharacters.call(this, text);
        text = text.replace(/\x1bChoiceId\[([^\]]*)\]/gi, '');
        return text;
    };

    // Window_ChoiceListの初期化
    const _Window_ChoiceList_initialize = Window_ChoiceList.prototype.initialize;
    Window_ChoiceList.prototype.initialize = function(messageWindow) {
        _Window_ChoiceList_initialize.call(this, messageWindow);
        this._messageWindow = messageWindow;
        this._lastChoiceId = null;
    };

    // 選択肢からIDを抽出
    Window_ChoiceList.prototype.extractChoiceId = function(text) {
        const match = text.match(/\\choiceId\[([^\]]*)\]/i);
        return match ? match[1] : null;
    };

    // 現在の選択肢セットのIDを取得
    Window_ChoiceList.prototype.getCurrentChoiceSetId = function() {
        if (!Array.isArray(this._list) || this._list.length === 0) return null;
        
        const firstChoice = this._list[0].name;
        const choiceId = this.extractChoiceId(firstChoice);
        
        if (choiceId) {
            const allSameId = this._list.every(item => 
                this.extractChoiceId(item.name) === choiceId
            );
            return allSameId ? choiceId : null;
        }
        
        return defaultBehavior ? 'default' : null;
    };

    // 現在のキャンセル位置を取得
    Window_ChoiceList.prototype.getDefaultCancelPosition = function() {
        // $gameMessageから現在の選択肢のキャンセル時の位置を取得
        return $gameMessage.choiceCancelType();
    };

    // 選択肢の表示テキストからIDを除去
    const _Window_ChoiceList_commandName = Window_ChoiceList.prototype.commandName;
    Window_ChoiceList.prototype.commandName = function(index) {
        const original = _Window_ChoiceList_commandName.call(this, index);
        return original.replace(/\\choiceId\[[^\]]*\]/gi, '');
    };

    // デフォルトの選択位置の設定
    const _Window_ChoiceList_selectDefault = Window_ChoiceList.prototype.selectDefault;
    Window_ChoiceList.prototype.selectDefault = function() {
        const choiceSetId = this.getCurrentChoiceSetId();
        
        if (choiceSetId) {
            // まずプラグインの保存された位置を確認
            const storedIndex = $gameSystem._choiceCursorPositions[choiceSetId];
            if (storedIndex !== undefined && this._list && this._list.length > storedIndex) {
                this.select(storedIndex);
                return;
            }
        }
        
        // 保存された位置がない場合はデフォルト処理を実行
        _Window_ChoiceList_selectDefault.call(this);
    };

    // カーソル位置の保存
    Window_ChoiceList.prototype.saveCursorPosition = function() {
        const choiceSetId = this.getCurrentChoiceSetId();
        if (choiceSetId) {
            $gameSystem._choiceCursorPositions[choiceSetId] = this.index();
            this._lastChoiceId = choiceSetId;
        }
    };

    // 選択時のカーソル位置保存
    const _Window_ChoiceList_callOkHandler = Window_ChoiceList.prototype.callOkHandler;
    Window_ChoiceList.prototype.callOkHandler = function() {
        this.saveCursorPosition();
        _Window_ChoiceList_callOkHandler.call(this);
    };

    // キャンセル時のカーソル位置保存とデフォルトの選択肢の設定
    const _Window_ChoiceList_processCancel = Window_ChoiceList.prototype.processCancel;
    Window_ChoiceList.prototype.processCancel = function() {
        const choiceSetId = this.getCurrentChoiceSetId();
        
        if (this.isCancelEnabled()) {
            if (choiceSetId) {
                // 現在のキャンセル設定を保存
                const defaultCancelPosition = this.getDefaultCancelPosition();
                $gameSystem._choiceCancelPositions[choiceSetId] = defaultCancelPosition;
                
                // 保存されたキャンセル位置があれば、それを使用
                const storedCancelPosition = $gameSystem._choiceCancelPositions[choiceSetId];
                if (storedCancelPosition !== undefined) {
                    // キャンセル位置に応じた処理
                    switch (storedCancelPosition) {
                        case -2: // 分岐の最後の選択肢
                            this.select(this._list.length - 1);
                            break;
                        case -1: // 分岐を飛ばす
                            // デフォルトの処理を使用
                            break;
                        default: // 特定の選択肢
                            if (storedCancelPosition >= 0 && storedCancelPosition < this._list.length) {
                                this.select(storedCancelPosition);
                            }
                            break;
                    }
                }
            }
            
            _Window_ChoiceList_processCancel.call(this);
        }
    };

    // 選択肢ウィンドウを開く時の処理
    const _Window_ChoiceList_start = Window_ChoiceList.prototype.start;
    Window_ChoiceList.prototype.start = function() {
        _Window_ChoiceList_start.call(this);
        
        const choiceSetId = this.getCurrentChoiceSetId();
        if (choiceSetId) {
            const storedIndex = $gameSystem._choiceCursorPositions[choiceSetId];
            if (storedIndex !== undefined && this._list && this._list.length > storedIndex) {
                this.select(storedIndex);
            }
        }
    };

    // ウィンドウを閉じる時の処理
    const _Window_ChoiceList_close = Window_ChoiceList.prototype.close;
    Window_ChoiceList.prototype.close = function() {
        this.saveCursorPosition();
        _Window_ChoiceList_close.call(this);
    };

})();