/*:
 * @target MZ
 * @plugindesc スキル選択確率設定プラグイン（ウィンドウカスタマイズ対応版）
 * @author スタジオVR
 *
 * @param switchOffset
 * @text スイッチのオフセット値
 * @type number
 * @default 0
 * @desc スキルの使用可否を格納するスイッチのオフセット値を設定します。
 *
 * @param skillWindowX
 * @text スキルウィンドウX座標
 * @type number
 * @default 0
 * @desc スキルウィンドウのX座標を指定します。
 *
 * @param skillWindowY
 * @text スキルウィンドウY座標
 * @type number
 * @default 444
 * @desc スキルウィンドウのY座標を指定します。
 *
 * @param skillWindowWidth
 * @text スキルウィンドウの幅
 * @type number
 * @default 816
 * @desc スキルウィンドウの幅を指定します。
 *
 * @param skillWindowVisibleRows
 * @text 表示行数
 * @type number
 * @default 4
 * @desc スキルウィンドウに表示する行数を指定します。
 *
 * @help
 * このプラグインは、戦闘中のスキル選択確率を設定し、
 * 有効なスキルを上部にソートする機能を追加します。
 * スキルのメモ欄に <SkillUnlockRate:xx, yy> と記入することで、
 * そのスキルの選択確率を設定できます（xxは0-100の数値、yyは加算値）。
 * 加算値は現在未対応です。
 * 設定がない場合は、100%選択可能です。
 * 
 * スキルの並び順は <OrderId:x> で制御します。
 * xの値が小さいほど上に表示されます。
 *
 * プラグインパラメータでスキルウィンドウの位置、幅、表示行数を調整できます。
 * 
 * スキル選択後、カーソルは自動的に一番上に戻ります。
 */

(function() {
    const parameters = PluginManager.parameters('Svr_SkillDrawRate');
    const switchOffset = Number(parameters['switchOffset']) || 0;
    const skillWindowX = Number(parameters['skillWindowX'] || 0);
    const skillWindowY = Number(parameters['skillWindowY'] || 444);
    const skillWindowWidth = Number(parameters['skillWindowWidth'] || 816);
    const skillWindowVisibleRows = Number(parameters['skillWindowVisibleRows'] || 4);

    // Window_BattleSkill クラスの拡張
    const _Window_BattleSkill_show = Window_BattleSkill.prototype.show;
    Window_BattleSkill.prototype.show = function() {
        _Window_BattleSkill_show.call(this);
        this.resetScroll();
    };

    Window_BattleSkill.prototype.resetScroll = function() {
        this.scrollTo(0, 0);
        this.select(0);
        this.refresh();
    };

    // Scene_Battle クラスの拡張
    const _Scene_Battle_commandSkill = Scene_Battle.prototype.commandSkill;
    Scene_Battle.prototype.commandSkill = function() {
        _Scene_Battle_commandSkill.call(this);
        this._skillWindow.resetScroll();
    };
    
    const _BattleManager_startTurn = BattleManager.startTurn;
    BattleManager.startTurn = function() {
//        console.log("---------- ターン開始 ----------");
//        $gameParty.members().forEach(actor => actor.determineSkillAvailability());
//        $gameTroop.members().forEach(enemy => enemy.determineSkillAvailability());
        this._isFirstProcessTurn = true;
        _BattleManager_startTurn.call(this);
    };

    const _BattleManager_processTurn = BattleManager.processTurn;
    BattleManager.processTurn = function() {
//        console.log(`BattleManager_processTurn`);
        if (!this._processedActors) {
            this._processedActors = new Set();
            this._isFirstProcessTurn = false;
        } else if (this._isFirstProcessTurn) {
//            console.log("最初の1回目");
            this._processedActors = new Set();
            this._isFirstProcessTurn = false;
        }

        const currentActor = this._subject;
        if (currentActor && currentActor.isActor() && !this._processedActors.has(currentActor.actorId())) {
//            console.log(`----- ${currentActor.name()} のスキル有効性とソートを処理 -----`);
            currentActor.determineSkillAvailability();
            currentActor.sortSkills();
            this._processedActors.add(currentActor.actorId());
        }

        _BattleManager_processTurn.call(this);
    };

    const _BattleManager_endTurn = BattleManager.endTurn;
    BattleManager.endTurn = function() {
        this._processedActors = new Set();
        _BattleManager_endTurn.call(this);
    };

    const _Game_Battler_onAllActionsEnd = Game_Battler.prototype.onAllActionsEnd;
    Game_Battler.prototype.onAllActionsEnd = function() {
        _Game_Battler_onAllActionsEnd.call(this);

        if (this.isActor()) {
            for (let i = switchOffset; i <= switchOffset + 2000; i++) {
                $gameSwitches.setValue(i, false);
            }
        }
    };

    Game_Battler.prototype.determineSkillAvailability = function() {
        let availabilityChanged = false;
        this.skills().forEach(skill => {
            const [initialRate, increment] = this.getSkillDrawRate(skill);
            if (skill._currentRate === undefined) {
                skill._currentRate = initialRate;
                skill._initialRate = initialRate;
            }
            const wasAvailable = skill.isAvailable;
            skill.isAvailable = Math.random() * 100 < skill._currentRate;
            $gameSwitches.setValue(switchOffset + skill.id, !skill.isAvailable);

            if (wasAvailable !== skill.isAvailable) {
                availabilityChanged = true;
            }

//            console.log(`${this.name()} - Skill: ${skill.name}, Available: ${skill.isAvailable}`);
            if (!skill.isAvailable) {
                skill._currentRate = skill._initialRate;
            } else {
                skill._currentRate = skill._initialRate;
            }
        });
        
        if (availabilityChanged && this.isActor()) {
            this.sortSkills();
        }
    };

    Game_Battler.prototype.getSkillDrawRate = function(skill) {
        const match = /<SkillUnlockRate:(\d+),\s*(\d+)>/i.exec(skill.note);
        return match ? [Number(match[1]), Number(match[2])] : [100, 0];
    };

    const _Window_BattleSkill_isEnabled = Window_BattleSkill.prototype.isEnabled;
    Window_BattleSkill.prototype.isEnabled = function(item) {
        return _Window_BattleSkill_isEnabled.call(this, item) && item.isAvailable !== false;
    };

    const _Game_Action_evaluate = Game_Action.prototype.evaluate;
    Game_Action.prototype.evaluate = function() {
        if (this.item() && !this.item().isAvailable) {
            return -1;
        }
        return _Game_Action_evaluate.call(this);
    };

    Game_Battler.prototype.resetSkillAvailability = function() {
        this.skills().forEach(skill => {
            const [initialRate, increment] = this.getSkillDrawRate(skill);
            skill._currentRate = initialRate;
            skill._initialRate = initialRate;
            skill.isAvailable = true;
            $gameSwitches.setValue(switchOffset + skill.id, false);
        });
        if (this.isActor()) {
            this.sortSkills();
        }
    };

    const _BattleManager_endBattle = BattleManager.endBattle;
    BattleManager.endBattle = function(result) {
        this._isFirstProcessTurn = true;
        $gameParty.members().forEach(actor => actor.resetSkillAvailability());
        $gameTroop.members().forEach(enemy => enemy.resetSkillAvailability());
        _BattleManager_endBattle.call(this, result);
    };

    function getSkillOrderId(skill) {
        const match = /<OrderId:(\d+)>/i.exec(skill.note);
        return match ? Number(match[1]) : Infinity;
    }

    Game_Actor.prototype.sortSkills = function() {
//        console.log(`${this.name()} のスキルをソートします。`);
//        console.log("ソート前のスキル:");
        this.skills().forEach(skill => {
//            console.log(`Skill: ${skill.name}, ID: ${skill.id}, Available: ${skill.isAvailable}, OrderId: ${getSkillOrderId(skill)}`);
        });

        this._skills.sort((a, b) => {
            const skillA = $dataSkills[a];
            const skillB = $dataSkills[b];
            if (skillA.isAvailable !== skillB.isAvailable) {
                return skillA.isAvailable ? -1 : 1;
            }
            return getSkillOrderId(skillA) - getSkillOrderId(skillB);
        });

//        console.log("ソート後のスキル:");
        this.skills().forEach(skill => {
//            console.log(`Skill: ${skill.name}, ID: ${skill.id}, Available: ${skill.isAvailable}, OrderId: ${getSkillOrderId(skill)}`);
        });
    };

    Window_SkillList.prototype.makeItemList = function() {
        if (this._actor) {
            this._data = this._actor.skills().filter(item => this.includes(item));
        } else {
            this._data = [];
        }
    };

    const _Window_BattleSkill_refresh = Window_BattleSkill.prototype.refresh;
    Window_BattleSkill.prototype.refresh = function() {
        this.makeItemList();
        _Window_BattleSkill_refresh.call(this);
    };

    const _Scene_Battle_createSkillWindow = Scene_Battle.prototype.createSkillWindow;
    Scene_Battle.prototype.createSkillWindow = function() {
        const rect = this.skillWindowRect();
        this._skillWindow = new Window_BattleSkill(rect);
        this._skillWindow.setHelpWindow(this._helpWindow);
        this._skillWindow.setHandler("ok", this.onSkillOk.bind(this));
        this._skillWindow.setHandler("cancel", this.onSkillCancel.bind(this));
        this.addWindow(this._skillWindow);
    };

    Scene_Battle.prototype.skillWindowRect = function() {
        const wx = skillWindowX;
        const wy = skillWindowY;
        const ww = skillWindowWidth;
        const wh = this.calcSkillWindowHeight();
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_Battle.prototype.calcSkillWindowHeight = function() {
        return this.calcWindowHeight(skillWindowVisibleRows, true);
    };

    const _Window_BattleSkill_initialize = Window_BattleSkill.prototype.initialize;
    Window_BattleSkill.prototype.initialize = function(rect) {
        _Window_BattleSkill_initialize.call(this, rect);
        this.refresh();
    };

    Window_BattleSkill.prototype.maxCols = function() {
        return 1;
    };

    Window_BattleSkill.prototype.numVisibleRows = function() {
        return skillWindowVisibleRows;
    };
})();